<?php


namespace Ysb\IPara\Helpers\Xml;


class XmlNode
{

    #region fields

    /**
     * @var string
     */
    protected $tagName;

    /**
     * @var bool
     */
    protected $selfClose = false;

    /**
     * @var bool
     */
    protected $root = false;

    /**
     * @var string|null
     */
    protected $prefixNs = null;

    /**
     * @var array|string[]
     */
    protected $attributes = array();

    /**
     * @var XmlNode|null
     */
    protected $parentNode = null;

    /**
     * @var array|XmlNode[]
     */
    protected $childNodes = array();

    /**
     * @var string|null
     */
    protected $text = null;

    /**
     * @var string|null
     */
    protected $comments = null;

    /**
     * @var bool
     */
    protected $ignoreNs = false;

    #endregion fields

    #region ctor

    /**
     * XmlNode constructor.
     * @param string|null $tagName
     * @param string|null $text
     * @param bool|null $ignoreNs
     */
    public function __construct($tagName = null, $text = null, $ignoreNs = null)
    {
        if (!is_null($tagName)) {
            if (strpos($tagName, ':') !== false) {
                $tgNms = explode(':', $tagName);
                $this->prefixNs = $tgNms[0];
                $this->tagName = $tgNms[1];
            } else {
                $this->tagName = $tagName;
            }

        }
        if (!is_null($text)) {
            $this->text = $text;
        }
        if (!is_null($ignoreNs)) {
            $this->ignoreNs = $ignoreNs;
        }
    }

    #endregion ctor

    #region properties

    /**
     * @return string
     */
    public function getTagName()
    {
        return $this->tagName;
    }

    /**
     * @param string $tagName
     * @return $this
     */
    public function setTagName($tagName)
    {
        $this->tagName = $tagName;
        return $this;
    }

    /**
     * @return bool
     */
    public function isSelfClose()
    {
        return $this->selfClose;
    }

    /**
     * @param bool $selfClose
     * @return $this
     */
    public function setSelfClose($selfClose = true)
    {
        $this->selfClose = $selfClose;
        return $this;
    }

    /**
     * @return bool
     */
    public function isRoot()
    {
        return $this->root;
    }

    /**
     * @param bool $root
     * @return $this
     */
    public function setRoot($root = true)
    {
        $this->root = $root;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getPrefixNs()
    {
        return $this->prefixNs;
    }

    /**
     * @param string|null $prefixNs
     * @return $this
     */
    public function setPrefixNs($prefixNs)
    {
        $this->prefixNs = $prefixNs;
        return $this;
    }

    /**
     * @return array|string[]
     */
    public function getAttributes()
    {
        return $this->attributes;
    }

    /**
     * @param array|string[] $attributes
     * @return $this
     */
    public function setAttributes($attributes)
    {
        $this->attributes = $attributes;
        return $this;
    }

    /**
     * @return XmlNode|null
     */
    public function getParentNode()
    {
        return $this->parentNode;
    }

    /**
     * @param XmlNode|null $parentNode
     * @return $this
     */
    public function setParentNode($parentNode)
    {
        $this->parentNode = $parentNode;
        return $this;
    }

    /**
     * @return array|XmlNode[]
     */
    public function getChildNodes()
    {
        return $this->childNodes;
    }

    /**
     * @param array|XmlNode[] $childNodes
     * @return $this
     */
    public function setChildNodes($childNodes)
    {
        $this->childNodes = $childNodes;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getText()
    {
        return $this->text;
    }

    /**
     * @param string|null $text
     * @return $this
     */
    public function setText($text)
    {
        $this->text = $text;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getComments()
    {
        return $this->comments;
    }

    /**
     * @param string|null $comments
     * @return $this
     */
    public function setComments($comments)
    {
        $this->comments = $comments;
        return $this;
    }

    /**
     * @return bool
     */
    public function isIgnoreNs()
    {
        return $this->ignoreNs;
    }

    /**
     * @param bool $ignoreNs
     * @return $this
     */
    public function setIgnoreNs($ignoreNs = true)
    {
        $this->ignoreNs = $ignoreNs;
        return $this;
    }


    #endregion properties

    #region methods

    /**
     * @param XmlNode $xmlNode
     * @return $this
     */
    public function addChild($xmlNode)
    {
        $this->childNodes[] = $xmlNode;
        return $this;
    }

    /**
     * @param string $key
     * @param string $value
     * @return $this
     */
    public function addAttribute($key, $value){
        $this->attributes[$key] = $value;
        return $this;
    }

    /**
     * @param string $strAttributes
     * @return XmlNode
     */
    public function parseAttributes($strAttributes)
    {
        $strLinesArr = array();
        if (strpos($strAttributes, "\n") !== false) {
            $strLinesArr = explode("\n", $strAttributes);
        } else {
            $strLinesArr[] = $strAttributes;
        }
        foreach ($strLinesArr as $strLine) {
            if (empty($strLine)) {
                continue;
            }
            $lnArr = explode('=', $strLine);
            $key = $lnArr[0];
            $key = trim($key);
            $val = $lnArr[1];
            $val = trim($val);
            $val = trim($val, '\'');
            $val = trim($val, '"');
            $this->attributes[$key] = $val;
        }
        return $this;
    }

    /**
     * @param bool $useIndent
     * @param int $indentLevel
     * @param string $indentChars
     * @return string
     */
    public function toXmlString($useIndent = false, $indentLevel = 0, $indentChars = '    ')
    {

        if (empty($this->text) && empty($this->childNodes)) {
            $this->selfClose = true;
        }
        $tagName = $this->getTagName();
        $xmlRes =  $this->renderComments($useIndent,$indentLevel,$indentChars);
        $xmlRes .= str_repeat($indentChars, $indentLevel) . "<" . $tagName;
        if (!empty($this->attributes)) {
            foreach ($this->attributes as $key => $value) {
                $xmlRes .= ' ' . $key;
                $xmlRes .= '=' . '"' . str_replace('"', '\"', $value) . '"';
            }
        }
        if ($this->selfClose) {
            $xmlRes .= '/>';
        } else {
            $xmlRes .= '>';
            if ($this->text !== null) {
                $xmlRes .= $this->text;
            } elseif (!empty($this->childNodes)) {
                if($useIndent){
                    $xmlRes .= PHP_EOL;
                }
                foreach ($this->childNodes as $node) {
                    if ($node !== null) {
                        $xmlRes .= $node->toXmlString($useIndent,$indentLevel + 1, $indentChars);
                    }
                }
                if($useIndent){
                    $xmlRes .= str_repeat($indentChars, $indentLevel);
                }
                // $xmlRes .= PHP_EOL;
            }

            $xmlRes .= '</' . $tagName . '>';
        }
        if($useIndent){
            $xmlRes .= PHP_EOL;
        }
        if ($this->root === true) {
            $xmlRes = '<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL . $xmlRes;
        }
        return $xmlRes;
    }

    public function __toString()
    {
        return $this->toXmlString(true);
    }

    #endregion methods

    #region utils

    /**
     * @param bool $useIndent
     * @param int $indentLevel
     * @param string $indentChars
     * @return string
     */
    protected function renderComments($useIndent = false, $indentLevel = 0, $indentChars = '    ')
    {
        $cmnt = '';
        if (isset($this->comments) && !empty($this->comments)) {
            if($useIndent){
                $cmnt .= str_repeat($indentChars,$indentLevel);
            }
            $cmnt .= '<!-- ' . $this->comments . ' -->';
            if($useIndent){
                $cmnt .= PHP_EOL;
            }
        }
        return $cmnt;
    }

    /**
     * @return string
     */
    protected function getTagNameForRender()
    {
        $tagname = $this->tagName;

        if (!empty($this->prefixNs) && ($this->ignoreNs == false)) {
            $tagname =  $this->prefixNs . ':' . $this->tagName;
        }
        return $tagname;
    }

    #endregion utils

}