<?php


namespace Ysb\Inbox;


use Ysb\Inbox\Entities\EmailInfo;
use Ysb\Inbox\Entities\ResponseObject;
use Ysb\Inbox\Entities\SendEmail;
use Ysb\Inbox\Entities\Settings;
use Ysb\Inbox\Entities\TokenResponse;

class InboxClient
{

    #region fields

    protected Settings $settings;

    protected ?string $accessToken = null;

    protected ?string $tokenType = null;

    protected ?int $expiresIn = null;

    protected ?string $refreshToken = null;

    protected int $startTokenSecs = 0;

    #endregion fields

    #region ctor

    public function __construct(Settings $settings)
    {
        $this->settings = $settings;
    }

    #endregion ctor

    #region methods

    /**
     * @return ResponseObject
     */
    public function getToken(){
        $headers = $this->getHeaders(false);
        $url = $this->settings->apiBaseUrl . "token";
        $reqObj = (object)[
            "EmailAddress" => $this->settings->emailAddress,
            "Password" => $this->settings->password,
        ];
        $res = RestHttpCaller::post($url,$headers,json_encode($reqObj));
        $retObj =  ResponseObject::fromJsonString($res);
        return $retObj;

    }

    /**
     * @return ResponseObject
     */
    public function refreshToken(){
        $headers = $this->getHeaders(false);
        $url = $this->settings->apiBaseUrl . "token/refresh";
        $reqObj = (object)[
            "RefreshToken" => $this->refreshToken,
        ];
        $res = RestHttpCaller::post($url,$headers,json_encode($reqObj));
        return ResponseObject::fromJsonString($res);
    }

    public function sendEmail(SendEmail $sendEmail){
        $this->checkToken();

        $headers = $this->getHeaders(true);
        $url = $this->settings->apiBaseUrl . "notify/v1/send";
        $res = RestHttpCaller::post($url,$headers,json_encode($sendEmail));
        $retObj =  ResponseObject::fromJsonString($res);
        return $retObj;

    }

    #endregion methods



    #region utils


    protected function checkToken(){
        if(empty($this->accessToken)){
            $ress = $this->getToken();
            if(false == $ress->resultStatus){
                throw new \Exception($ress->resultMessage);
            }
            $tokenObj = TokenResponse::fromObject($ress->resultObject);
            $this->accessToken = $tokenObj->accessToken;
            $this->expiresIn = $tokenObj->expiresIn;
            $this->tokenType = $tokenObj->tokenType;
            $this->refreshToken = $tokenObj->refreshToken;
            $this->startTokenSecs = time() - 2;
        }else{
            if($this->startTokenSecs + $this->expiresIn <= time() ){
                $res = $this->refreshToken();
                if(false == $res->resultStatus){
                    throw new \Exception($res->resultMessage);
                }
                $tokenObj = TokenResponse::fromObject($res->resultObject);
                $this->accessToken = $tokenObj->accessToken;
                $this->expiresIn = $tokenObj->expiresIn;
                $this->tokenType = $tokenObj->tokenType;
                $this->refreshToken = $tokenObj->refreshToken;
                $this->startTokenSecs = time() - 2;
            }
        }
    }

    /**
     * @param bool $incAuth
     * @return array|string[]
     */
    protected function getHeaders(bool $incAuth = true):array {
        $headers = [
            'Content-Type: application/json'
        ];
        if($incAuth && !empty($this->accessToken)){
            $atHeader = "Authorization: ";
            if(!empty($this->tokenType)){
                $atHeader .= $this->tokenType . " ";
            }
            $atHeader .= $this->accessToken;
            $headers[]= $atHeader;
        }
        return $headers;
    }

    #endregion utils

}