<?php


namespace Ys\ArasKargo\Xml;


class XmlNode
{

    #region fields

    /**
     * @var string
     */
    protected $tagName;

    /**
     * @var bool
     */
    protected $selfClose = false;

    /**
     * @var bool
     */
    protected $root = false;

    /**
     * @var string|null
     */
    protected $prefixNs = null;

    /**
     * @var array|string[]
     */
    protected $attributes = array();

    /**
     * @var XmlNode|null
     */
    protected $parentNode = null;

    /**
     * @var array|XmlNode[]
     */
    protected $childNodes = array();

    /**
     * @var string|null
     */
    protected $text = null;

    /**
     * @var string|null
     */
    protected $comments = null;

    /**
     * @var bool|null
     */
    protected $ignoreNs = null;

    #endregion fields

    #region ctor

    public function __construct( $tagName = null,  $text = null,  $ignoreNs = null)
    {
        if (!is_null($tagName)) {
            if (strpos($tagName, ':') !== false) {
                $tgNms = explode(':', $tagName);
                $this->prefixNs = $tgNms[0];
                $this->tagName = $tgNms[1];
            } else {
                $this->tagName = $tagName;
            }

        }
        if (!is_null($text)) {
            $this->text = $text;
        }
        if (!is_null($ignoreNs)) {
            $this->ignoreNs = $ignoreNs;
        }
    }

    #endregion ctor

    #region properties

    /**
     * @return string
     */
    public function getTagName()
    {
        return $this->tagName;
    }

    /**
     * @param string $tagName
     * @return XmlNode
     */
    public function setTagName($tagName)
    {
        $this->tagName = $tagName;
        return $this;
    }

    /**
     * @return bool
     */
    public function isSelfClose()
    {
        return $this->selfClose;
    }

    /**
     * @param bool $selfClose
     * @return XmlNode
     */
    public function setSelfClose($selfClose)
    {
        $this->selfClose = $selfClose;
        return $this;
    }

    /**
     * @return bool
     */
    public function isRoot()
    {
        return $this->root;
    }

    /**
     * @param bool $root
     * @return XmlNode
     */
    public function setRoot($root)
    {
        $this->root = $root;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getPrefixNs()
    {
        return $this->prefixNs;
    }

    /**
     * @param string|null $prefixNs
     * @return XmlNode
     */
    public function setPrefixNs($prefixNs)
    {
        $this->prefixNs = $prefixNs;
        return $this;
    }

    /**
     * @return array|string[]
     */
    public function getAttributes()
    {
        return $this->attributes;
    }

    /**
     * @param array|string[] $attributes
     * @return XmlNode
     */
    public function setAttributes($attributes)
    {
        $this->attributes = $attributes;
        return $this;
    }

    /**
     * @return XmlNode|null
     */
    public function getParentNode()
    {
        return $this->parentNode;
    }

    /**
     * @param XmlNode|null $parentNode
     * @return XmlNode
     */
    public function setParentNode($parentNode)
    {
        $this->parentNode = $parentNode;
        return $this;
    }

    /**
     * @return array|XmlNode[]
     */
    public function getChildNodes()
    {
        return $this->childNodes;
    }

    /**
     * @param array|XmlNode[] $childNodes
     * @return XmlNode
     */
    public function setChildNodes($childNodes)
    {
        $this->childNodes = $childNodes;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getText()
    {
        return $this->text;
    }

    /**
     * @param string|null $text
     * @return XmlNode
     */
    public function setText($text)
    {
        $this->text = $text;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getComments()
    {
        return $this->comments;
    }

    /**
     * @param string|null $comments
     * @return XmlNode
     */
    public function setComments($comments)
    {
        $this->comments = $comments;
        return $this;
    }

    /**
     * @return bool|null
     */
    public function getIgnoreNs()
    {
        return $this->ignoreNs;
    }

    /**
     * @param bool|null $ignoreNs
     * @return XmlNode
     */
    public function setIgnoreNs($ignoreNs)
    {
        $this->ignoreNs = $ignoreNs;
        return $this;
    }

    #endregion properties

    #region methods

    /**
     * @param XmlNode $xmlNode
     * @return $this
     */
    public function addChild($xmlNode){
        $xmlNode->setParentNode($this);
        $this->childNodes[] = $xmlNode;
        return $this;
    }

    /**
     * @param $key
     * @param $value
     * @return $this
     */
    public function addAttribute($key, $value){
        $this->attributes[$key] = $value;
        return $this;
    }

    /**
     * @param int $indentLevel
     * @return string
     */
    public function toXmlString($indentLevel = 0)
    {
        $indentStr = '    ';

        if (empty($this->text) && empty($this->childNodes)) {
            $this->selfClose = true;
        }
        $tagName = $this->getTagNameForRender();
        $xmlRes =  $this->renderComments(str_repeat($indentStr, $indentLevel));
        $xmlRes .= str_repeat($indentStr, $indentLevel) . "<" . $tagName;
        if (!empty($this->attributes)) {
            foreach ($this->attributes as $key => $value) {
                $xmlRes .= ' ' . $key;
                $xmlRes .= '=' . '"' . str_replace('"', '\"', $value) . '"';
            }
        }
        if ($this->selfClose) {
            $xmlRes .= '/>'.PHP_EOL;
        } else {
            $xmlRes .= '>';
            if ($this->text !== null) {
                $xmlRes .= $this->text;
            } elseif (!empty($this->childNodes)) {
                $xmlRes .= PHP_EOL;
                foreach ($this->childNodes as $node) {
                    if ($node !== null) {
                        $xmlRes .= $node->toXmlString($indentLevel + 1);
                    }
                }
                $xmlRes .= str_repeat($indentStr, $indentLevel);
                // $xmlRes .= PHP_EOL;
            }

            $xmlRes .= '</' . $tagName . '>' . PHP_EOL;
        }
        if ($this->root === true) {
            $xmlRes = '<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL . $xmlRes;
        }
        return $xmlRes;
    }

    /**
     * @return string
     */
    public function __toString()
    {
        return $this->toXmlString(0);
    }

    #endregion methods

    #region utils

    /**
     * @param string $indent
     * @return string
     */
    protected function renderComments($indent = '')
    {
        $cmnt = '';
        if (isset($this->comments) && !empty($this->comments)) {
            $cmnt .= $indent . '<!-- ' . $this->comments . ' -->' . PHP_EOL;
        }
        return $cmnt;
    }

    /**
     * @return string
     */
    protected function getTagNameForRender()
    {
        $tagname = $this->tagName;

        if (!empty($this->prefixNs) && ($this->ignoreNs == false)) {
            $tagname =  $this->prefixNs . ':' . $this->tagName;
        }
        return $tagname;
    }

    #endregion utils

}