<?php


namespace GekTools\Tools\View;


use Exception;
use Gek\Filesystem\Filesystem;
use GekTools\Tools\Cache\DataCache;
use GekTools\Tools\Cache\Ttl;
use SimpleXMLElement;


class MenuManager
{

    #region fields

    /**
     * @var DataCache
     */
    protected DataCache $dataCache;

    #endregion fields

    #region ctor

    public function __construct()
    {
        $this->dataCache = DataCache::instance();
    }

    #endregion ctor

    #region Methods

    /**
     * @param string $xmlPath
     * @return Menu
     */
    public function loadMenu(string $xmlPath):Menu
    {
        $cacheKey = $this->cacheKey($xmlPath);
        return $this->dataCache->getOrSave(
            $cacheKey,
            [$this, 'loadMenuNoCache'],
            [$xmlPath],
            Ttl::fromWeeks(2)
        );
    }

    /**
     * @param string $xmlPath
     * @return Menu
     * @throws Exception
     */
    public function loadMenuNoCache(string $xmlPath):Menu
    {
        $fs = new Filesystem();
        if(! $fs->exists($xmlPath)){
            throw new Exception('Menu dosyası bulunamadı. path: ' . $xmlPath );
        }
        $xmlStr = file_get_contents($xmlPath);
        $root = new SimpleXMLElement($xmlStr);
        $menu = new Menu();
        foreach ($root->children() as $menuItem){
            $menu->addMenuItem($this->readMenuItem($menuItem));
        }
        return $menu;
    }

    /**
     * @param Menu $menu
     * @param string $xmlPath
     * @throws \Throwable
     */
    public function saveMenu(Menu $menu, string $xmlPath){
        $fs = new Filesystem();
        $fs->dumpFile($xmlPath,$menu->toXml());
        $cacheKey = $this->cacheKey($xmlPath);
        $this->dataCache->delete($cacheKey);
    }

    #endregion Methods

    #region utils

    /**
     * @param string $xmlPath
     * @return string
     */
    protected function cacheKey(string $xmlPath): string
    {
        return 'menu_' . md5($xmlPath);
    }

    /**
     * @param SimpleXMLElement $element
     * @return MenuItem
     */
    protected function readMenuItem(SimpleXMLElement $element):MenuItem{
        $mi = new MenuItem();

        foreach ($element->attributes() as $key => $val){
            switch (strtolower($key)){
                case 'text':
                    $mi->setText($val);
                    break;
                case 'url':
                    $mi->setUrl($val);
                    break;
                case 'systemname':
                    $mi->setSystemName($val);
                    break;
                case 'iconclass':
                    $mi->setIconClass($val);
                    break;
                case 'loginrequired':
                    $mi->setLoginRequired((bool)$val);
                    break;
                case 'requiredpermissions':
                    $mi->setRequiredPermissionNames(explode(',',$val));
                    break;
                case 'requiredroles':
                    $mi->setRequiredRoles(explode(',',$val));
                    break;
                default:
                    $mi->addAttribute($key,$val);
                    break;
            }
        }

        foreach ($element->children() as $subMenuItem){
            $mi->addChild($this->readMenuItem($subMenuItem));
        }
        return $mi;
    }

    #endregion utils

}