<?php


namespace GekTools\Tools\Html;


use Gek\Collections\Typed\StringDictionary;

class HtmlHelper
{

    #region fields

    /**
     * @var array|string[]
     */
    protected array $selfClosingTags = array(
        'area',
        'base',
        'basefont',
        'br',
        'col',
        'embed',
        'hr',
        'img',
        'input',
        'keygen',
        'link',
        'meta',
        'param',
        'source',
        'spacer',
        'track',
        'wbr'
    );

    /**
     * @var array|string[]
     */
    protected array $formTypes = array(
        'input',
        'form',
        'textarea',
        'select',
        'button',
    );

    /**
     * @var array
     */
    protected array $allIds = array();

    /**
     * @var array
     */
    protected array $formElementIds = array();

    /**
     * @var string|null
     */
    protected ?string $lastId = null;

    /**
     * @var string|null
     */
    protected ?string $lastFormElementId = null;

    #endregion fields

    #region methods
    /**
     * @param string $id
     * @return bool
     */
    public function isUsedId(string $id): bool
    {
        return in_array($id, $this->allIds);
    }

    /**
     * @param string $id
     * @return string
     */
    public function fixId(string $id): string
    {
        return $id . rand(1, 500);
    }

    /**
     * @param string $id
     * @param string $tag
     * @param bool $autoFix
     * @return string
     */
    public function addId(string $id, string $tag, bool $autoFix = true): string
    {
        if ($autoFix) {
            if ($this->isUsedId($id)) {
                $id = $this->fixId($id);
            }
        }
        $this->allIds[] = $id;
        $this->lastId = $id;
        if (in_array(strtolower(trim($tag)), $this->formTypes)) {
            $this->formElementIds[] = $id;
            $this->lastFormElementId = $id;
        }
        return $id;
    }


    /**
     * @param string $tagName
     * @param array|HtmlNode|string|HtmlNode[]|string[]|null $content
     * @param array $attributes
     * @return HtmlElement
     */
    public function createTag(string $tagName, $content = null, array $attributes = array()): HtmlElement
    {
        $node = new HtmlElement($tagName);
        if (in_array($tagName, $this->selfClosingTags)) {
            $node->setSelfClosing();
        }
        if (!empty($attributes)) {
            if (isset($attributes['id'])) {
                $attributes['id'] = $this->addId($attributes['id'], $tagName, true);
            }
        }
        $node->setAttributes(new StringDictionary($attributes));
        if (!empty($content)) {
            if (is_array($content)) {
                foreach ($content as $c) {
                    $this->addChild($node, $c);
                }
            } else {
                $this->addChild($node, $content);
            }
        }
        return $node;
    }

    /**
     * @param string $text
     * @return TextNode
     */
    public function createTextNode(string $text): TextNode
    {
        return new TextNode($text);
    }

    /**
     * @param string $text
     * @return CommentNode
     */
    public function createCommentNode(string $text): CommentNode
    {
        return new CommentNode($text);
    }


    /**
     * @param string $label
     * @param string|null $for
     * @param array $attributes
     * @return HtmlElement
     */
    public function createLabel(string $label, ?string $for = null, array $attributes = array()): HtmlElement
    {
        if ($for !== null) {
            $attributes['for'] = $for;
        }
        return $this->createTag('label', $label, $attributes);
    }

    /**
     * @param string $name
     * @param string $type
     * @param mixed|null $value
     * @param array $attributes
     * @return HtmlElement
     */
    public function createInput(string $name, string $type, $value = null, array $attributes = array()): HtmlElement
    {
        $attributes['name'] = $name;
        $attributes['type'] = $type;
        if ($value !== null) {
            $attributes['value'] = $value === false ? '0' : strval($value);
        }
        return $this->createTag('input', null, $attributes);
    }

    /**
     * @param string $name
     * @param mixed|null $value
     * @param array $attributes
     * @return HtmlElement
     */
    public function createTextArea(string $name, $value = null, array $attributes = array()): HtmlElement
    {
        $attributes['name'] = $name;
        return $this->createTag('textarea',$value,$attributes);
    }



    #endregion methods

    #region utils

    /**
     * @param $attributes
     * @return array
     */
    protected function fixAttributesArg($attributes)
    {
        if (is_object($attributes)) {
            $attributes = (array)$attributes;
        } elseif ($attributes === null) {
            $attributes = array();
        }
        return $attributes;
    }

    /**
     * @param HtmlNode $base
     * @param HtmlNode|string $child
     */
    protected function addChild(HtmlNode $base, $child)
    {
        if ($child instanceof HtmlNode) {
            if ($child instanceof HtmlElement) {
                if ($child->getId() !== null) {
                    $id = $this->addId($child->getId(), $child->getTagName(), true);
                    $child->setId($id);
                }
            }
            $base->addChild($child);
        } else {
            $base->addChild($this->createTextNode($child));
        }
    }

    #endregion utils

}
