<?php


namespace GekTools\Tools\Cache;

use \CodeIgniter\Cache\CacheInterface;
use Gek\Infrastructure\Str;

class DataCache implements IDataCache
{

    #region fields

    /**
     * @var bool
     */
    protected bool $dataCacheOn = true;


    /**
     * @var \CodeIgniter\Cache\CacheInterface
     */
    protected CacheInterface $cache;

    #endregion fields

    #region ctor

    public function __construct(bool $dataCacheOn = true)
    {

        $this->dataCacheOn = $dataCacheOn;
        $this->cache = \Config\Services::cache();
    }

    #endregion ctor

    #region IDataCache

    /**
     * Takes care of any handler-specific setup that must be done.
     */
    public function initialize()
    {
        $this->cache->initialize();
    }

    /**
     * Attempts to fetch an item from the cache store.
     *
     * @param string $key Cache item name
     *
     * @return mixed
     */
    public function get(string $key)
    {
        return $this->cache->get($key);
    }

    /**
     * Saves an item to the cache store.
     *
     * @param string $key Cache item name
     * @param mixed $value The data to save
     * @param integer $ttl Time To Live, in seconds (default 60)
     *
     * @return mixed
     */
    public function save(string $key, $value, int $ttl = 60)
    {
        return $this->cache->save($key, $value, $ttl);
    }

    /**
     * Deletes a specific item from the cache store.
     *
     * @param string $key Cache item name
     *
     * @return mixed
     */
    public function delete(string $key)
    {
        return $this->cache->delete($key);
    }

    /**
     * Performs atomic incrementation of a raw stored value.
     *
     * @param string $key Cache ID
     * @param integer $offset Step/value to increase by
     *
     * @return mixed
     */
    public function increment(string $key, int $offset = 1)
    {
        return $this->cache->increment($key, $offset);
    }

    /**
     * Performs atomic decrementation of a raw stored value.
     *
     * @param string $key Cache ID
     * @param integer $offset Step/value to increase by
     *
     * @return mixed
     */
    public function decrement(string $key, int $offset = 1)
    {
        return $this->cache->decrement($key, $offset);
    }

    /**
     * Will delete all items in the entire cache.
     *
     * @return mixed
     */
    public function clean()
    {
        return $this->cache->clean();
    }

    /**
     * Returns information on the entire cache.
     *
     * The information returned and the structure of the data
     * varies depending on the handler.
     *
     * @return mixed
     */
    public function getCacheInfo()
    {
        return $this->cache->getCacheInfo();
    }

    /**
     * Returns detailed information about the specific item in the cache.
     *
     * @param string $key Cache item name.
     *
     * @return mixed
     */
    public function getMetaData(string $key)
    {
        return $this->cache->getMetaData($key);
    }

    /**
     * Determines if the driver is supported on this system.
     *
     * @return boolean
     */
    public function isSupported(): bool
    {
        return $this->cache->isSupported();
    }

    /**
     * @param string $key
     * @param callable $func
     * @param array $args
     * @param int $ttl
     * @return mixed
     */
    public function getOrSave(string $key, callable $func, array $args = [], int $ttl = 60)
    {
        $result = $this->dataCacheOn ?
            $this->get($key) :
            null;
        if ($result === null) {
            $result = call_user_func_array($func, $args);
            if ($this->dataCacheOn){
                $this->save($key,$result,$ttl);
            }
        }
        return $result;
    }

    /**
     * @param string $keyRegex
     * @return bool
     */
    public function deleteByRegex(string $keyRegex)
    {
        $cacheInfos = $this->getCacheInfo();
        $foundedKeys = array();
        foreach ($cacheInfos as $key => $val){
            if(preg_match($keyRegex,$key)){
                $foundedKeys[] = $key;
            }
        }

        foreach ($foundedKeys as $cKey){
            $this->delete($cKey);
        }
        return !empty($foundedKeys);
    }

    /**
     * @param string $keyStartWith
     * @return bool
     */
    public function deleteByStartWith(string $keyStartWith)
    {
        $cacheInfos = $this->getCacheInfo();
        $foundedKeys = array();
        foreach ($cacheInfos as $key => $val){
            if(Str::startsWith($key,$keyStartWith,true)){
                $foundedKeys[] = $key;
            }
        }

        foreach ($foundedKeys as $cKey){
            $this->delete($cKey);
        }
        return !empty($foundedKeys);
    }

    #endregion IDataCache

    #region statics

    /**
     * @param string $key
     * @return string
     */
    public static function fixKey(string $key):string {
        $res = str_replace('\\','.',trim($key));
        $res = str_replace([' ', "\t","\n","\r","\0","\x0B"],'_',$res);
        return $res;
    }

    public static function createKey(string $entity, array $fields = array()){
        $key = $entity;
        foreach ($fields as $k => $v){
            $key .= '_' .$k . '-'.$v;
        }
        return static::fixKey($key);
    }


    /**
     * @param bool $cacheOn
     * @return DataCache
     */
    public static function instance(bool $cacheOn = true):self {
        return service('dataCache', $cacheOn,true);
    }

    #endregion statics
}
