<?php
namespace GekTools\Settings\Commands;

use CodeIgniter\CLI\CLI;
use Config\Services;
use CodeIgniter\CLI\BaseCommand;
use Gek\Filesystem\Filesystem;
use Gek\Infrastructure\Str;
use Gek\PhpLang\CodeFactory;
use Gek\PhpLang\LiteralTypes;
use GekTools\Controllers\BaseController;
use GekTools\Settings\ISettings;

class CreateSettings extends BaseCommand
{

    protected $group = 'Ci4Tools';

    protected $name = 'create:settings';

    protected $description = 'Yeni bir settings oluşturur.';

    /**
     * the Command's usage

     *
     * @var string
     */
    protected $usage = 'create:settings [settings_name] [Options]';

    /**
     * the Command's Arguments

     *
     * @var array
     */
    protected $arguments = [
         'settings_name' => 'Settings adı.',
    ];

    /**
     * the Command's Options

     *
     * @var array
     */
    protected $options = [
        '-n' => 'Namespace ayarlar.',
        '-p' => 'path ayarlar'
    ];

    public function run(array $params){
        $settingsName = array_shift($params);

        if (empty($settingsName)) {
            $settingsName = CLI::prompt('Settings adını girin');
        }

        $settingsName = ucfirst($settingsName);

        if(! Str::endsWith($settingsName,'Settings')){
            $settingsName .= 'Settings';
        }

        $ns = $params['-n'] ?? CLI::getOption('n');
        $homepath = APPPATH;

        if (!empty($ns)) {
            // Get all namespaces
            $namespaces = Services::autoloader()->getNamespace();
            foreach ($namespaces as $namespace => $path) {
                if ($namespace === $ns) {
                    $homepath = realpath(reset($path));
                    break;
                }
            }
        } else {
            $ns = 'App';
        }

        $path = $params['-p'] ?? CLI::getOption('p');

        if (!empty($path)) {
            $path = str_replace(["/", "\\"], DIRECTORY_SEPARATOR, $path);
            $path = trim($path);
            $path = trim($path, DIRECTORY_SEPARATOR);
        }else{
            $path = 'Settings' . DIRECTORY_SEPARATOR;
        }
        $settingsPath = $homepath . DIRECTORY_SEPARATOR . $path . DIRECTORY_SEPARATOR . $settingsName . '.php';
        $settingsNs = $ns . '\\' . str_replace(DIRECTORY_SEPARATOR,'\\',$path);
        $settingsNs = trim($settingsNs,'\\');
        $fs = new Filesystem();

        if ($fs->exists($settingsPath)) {
            CLI::error('Dosya zaten mevcut : ' . str_replace($homepath, $ns, $settingsPath));
            return;
        }

        $settings = CodeFactory::classCreate($settingsName)
            ->addImplement(ISettings::class)
            ->setNamespace($settingsNs);

        $sampleField = $settings->addField('sampleSetting', 'string')
            ->setProtected()
            ->setValue('default value', LiteralTypes::SINGLE_QUOTES)
            ->autoDocComment()
            ->setRegion('fields');

        $sampleGetter = $settings->addMethod('getSampleSetting')
            ->setReturnType('string')
            ->addCodeLine('return $this->sampleSetting;')
            ->autoDocComment()
            ->addCommentSummary('sampleSetting getter')
            ->setRegion('properties');

        $sampleSetter = $settings->addMethod('setSampleSetting')
            ->addParam('sampleSetting','string')
            ->setReturnType('self')
            ->addCodeLine('$this->sampleSetting = $sampleSetting;')
            ->addCodeLine('return $this;')
            ->autoDocComment()
            ->addCommentSummary('sampleSetting setter')
            ->setRegion('properties');

        $fs->dumpFile($settingsPath, '<?php' . PHP_EOL . $settings->toIndentedString());
        CLI::write('Dosya oluşturuldu : ' . CLI::color(str_replace($homepath, $ns, $settingsPath), 'green'));

    }

}
