<?php


namespace GekTools\Tools\Database;



use Gek\PhpLang\DocComments\DocComment;
use GekTools\Tools\Validations\ValidationCollection;

class TableFieldComment implements \Serializable
{
    #region fields

    /**
     * @var string|null
     */
    private ?string $fieldName = null;

    /**
     * @var string|null
     */
    private ?string $propName = null;

    /**
     * @var string|null
     */
    private ?string $phpType = null;

    /**
     * @var DbTypes|null
     */
    private ?DbTypes $dbType = null;

    private DocComment $docComment;

    private bool $nullable = false;

    private bool $primaryKey = false;

    private bool $unique = false;

    private bool $index = false;

    /**
     * @var string|null
     */
    private ?string $defaultValue = null;

    private bool $json = false;

    protected ValidationCollection $validations;

    #endregion fields

    #region ctor

    public function __construct()
    {
        $this->docComment = new DocComment();
        $this->validations = new ValidationCollection();
    }

    #endregion ctor

    #region props

    /**
     * field name getter
     *
     * @return string|null
     */
    public function getFieldName():?string{
        return $this->fieldName;
    }

    /**
     * field name setter
     *
     * @param string|null $fieldName
     * @return self
     */
    public function setFieldName(?string $fieldName):self {
        $this->fieldName = $fieldName;
        return $this;
    }

    /**
     * field name tanımlı mı
     *
     * @return bool
     */
    public function isFieldName():bool {
        return !empty($this->fieldName);
    }

    /**
     * prop name getter
     *
     * @return string|null
     */
    public function getPropName():?string {
        return $this->propName;
    }


    /**
     * prop name setter
     *
     * @param string|null $propName
     * @return self
     */
    public function setPropName(?string $propName):self{
        $this->propName = $propName;
        return $this;
    }

    /**
     * prop name tanımlı mı
     *
     * @return bool
     */
    public function isPropName():bool {
        return !empty($this->propName);
    }

    /**
     * php type getter
     *
     * @return string|null
     */
    public function getPhpType():?string{
        return $this->phpType;
    }

    /**
     * php type setter
     *
     * @param string|null $phpType
     * @return self
     */
    public function setPhpType(?string $phpType):self {
        $this->phpType = $phpType;
        return $this;
    }

    /**
     * php type tanımlı mı
     *
     * @return bool
     */
    public function isPhpType():bool{
        return !empty($this->phpType);
    }

    /**
     * db type getter
     *
     * @return DbTypes|null
     */
    public function getDbType():?DbTypes{
        return $this->dbType;
    }

    /**
     * db type setter
     *
     * @param DbTypes|int|string|null $dbType
     * @return self
     * @throws \ReflectionException
     */
    public function setDbType($dbType):self {
        if(!($dbType instanceof DbTypes) && $dbType !== null){
            $dbType = new DbTypes($dbType);
        }
        $this->dbType = $dbType;
        return $this;
    }

    /**
     * db type tanımlı mı
     * @return bool
     */
    public function isDbType():bool {
        return !empty($this->dbType);
    }

    /**
     * docComment getter
     *
     * @return DocComment|null
     */
    public function getDocComment():DocComment{
        return $this->docComment;
    }

    /**
     * docComment setter
     *
     * @param DocComment|null $docComment
     * @return TableFieldComment
     */
    public function setDocComment(DocComment $docComment):self {
        $this->docComment = $docComment;
        return $this;
    }



    /**
     * nullable getter
     *
     * @return bool
     */
    public function getNullable():bool {
        return $this->nullable;
    }

    /**
     * nullable setter
     *
     * @param bool $nullable
     * @return self
     */
    public function setNullable(bool $nullable):self{
        $this->nullable = $nullable;
        return $this;
    }


    /**
     * primaryKey getter
     *
     * @return bool
     */
    public function getPrimaryKey():bool {
        return $this->primaryKey;
    }

    /**
     * primaryKey setter
     *
     * @param bool $primaryKey
     * @return self
     */
    public function setPrimaryKey(bool $primaryKey):self {
        $this->primaryKey = $primaryKey;
        return $this;
    }

    /**
     * unique getter
     *
     * @return bool
     */
    public function getUnique():bool {
        return $this->unique;
    }

    /**
     * unique setter
     *
     * @param bool $unique
     * @return self
     */
    public function setUnique(bool $unique):self {
        $this->unique = $unique;
        return $this;
    }

    /**
     * index getter
     *
     * @return bool
     */
    public function getIndex():bool {
        return $this->index;
    }

    /**
     *
     *
     * @param bool $index
     * @return self
     */
    public function setIndex(bool $index):self {
        $this->index = $index;
        return $this;
    }

    /**
     * defaultValue getter
     *
     * @return string
     */
    public function getDefaultValue():?string {
        return $this->defaultValue;
    }

    /**
     * defaultValue setter
     *
     * @param string $defaultValue
     * @return self
     */
    public function setDefaultValue(?string $defaultValue):self {
        $this->defaultValue = $defaultValue;
        return $this;
    }

    /**
     * json getter
     *
     * @return bool
     */
    public function getJson():bool {
        return $this->json;
    }

    /**
     * json setter
     *
     * @param bool $json
     * @return self
     */
    public function setJson(bool $json):self {
        $this->json = $json;
        return $this;
    }

    /**
     * @return ValidationCollection
     */
    public function getValidations():ValidationCollection{
        return  $this->validations;
    }

    /**
     * @param ValidationCollection $validations
     * @return $this
     */
    public function setValidations(ValidationCollection $validations):self {
        $this->validations = $validations;
        return $this;
    }

    #endregion props

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = [];
        if($this->fieldName !== null){
            $data['fn'] = $this->fieldName;
        }

        if($this->propName !== null){
            $data['pn'] = $this->propName;
        }

        if($this->phpType !== null){
            $data['pt'] = $this->phpType;
        }
        if($this->dbType !== null){
            $data['dt'] = $this->dbType->toInt();
        }
        if(isset($this->docComment) && !$this->docComment->isEmpty() ){
            $data['dc'] = $this->docComment;
        }
        if($this->nullable){
            $data['nlb'] = $this->nullable;
        }
        if($this->primaryKey){
            $data['pk'] = $this->primaryKey;
        }
        if($this->unique){
            $data['unq'] = $this->unique;
        }
        if($this->index ){
            $data['inx'] = $this->index ;
        }
        if($this->defaultValue !== null ){
            $data['dv'] = $this->defaultValue;
        }
        if($this->json){
            $data['js'] = $this->json;
        }

        if($this->validations->any()){
            $data['vld'] = $this->validations->toArrayForSerialization();
        }

        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     * @throws \ReflectionException
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->fieldName = isset($data['fn']) ? $data['fn'] : null;
        $this->propName = isset($data['pn']) ? $data['pn'] : null;
        $this->phpType = isset($data['pt']) ? $data['pt'] : null;
        $this->dbType = isset($data['dt']) ? new DbTypes($data['dt']) : null;
        $this->docComment = isset($data['dc']) ? $data['dc'] : new DocComment();
        $this->nullable = isset($data['nlb']) ? $data['nlb'] : false;
        $this->primaryKey = isset($data['pk']) ? $data['pk'] : false;
        $this->unique = isset($data['unq']) ? $data['unq'] : false;
        $this->index = isset($data['inx']) ? $data['inx'] : false;
        $this->defaultValue = isset($data['dv']) ? $data['dv'] : null;
        $this->json = isset($data['js']) ? $data['js'] : false;
        $this->validations = isset($data['vld']) ?
            ValidationCollection::fromSerializeArray($data['vld']):
            new ValidationCollection();
    }

    #endregion Serializable

}
