<?php


namespace GekTools\Tools;


use CodeIgniter\Entity;
use CodeIgniter\I18n\Time;
use Gek\Infrastructure\ConstClassType;
use Gek\Infrastructure\EnumType;
use Gek\Infrastructure\FlagEnumType;

abstract class BaseEntity extends Entity implements \Serializable
{
    #region fields

    protected string $primaryKeyField = 'id';

    protected $dates = [
        'createdAtUtc',
        'updatedAtUtc',
        'deletedAtUtc',
    ];

    #endregion fields

    #region ctor

    public function __construct(array $data = null)
    {
        parent::__construct($data);
    }

    #endregion ctor

    #region properties

    /**
     * @return string
     */
    public function getPrimaryKeyField(): string
    {
        return $this->primaryKeyField;
    }

    #endregion properties

    #region methods

    /**
     * @param $value
     * @param string $type
     * @return Time|EnumType|FlagEnumType|mixed|null
     * @throws \Exception
     */
    public function castAs($value, string $type)
    {
        if (strpos($type, '?') === 0) {
            if ($value === null) {
                return null;
            }
            $type = substr($type, 1);
        }
        $res = parent::castAs($value, $type);
        if ($res === $value) {
            if (!class_exists($type)) {
                return $res;
            }
            if ($type == Time::class || $type == \DateTime::class) {
                return $this->mutateDate($value);
            }

            if (is_subclass_of($type, EnumType::class)) {
                return $this->mutateEnumType($value, $type);
            }
            if (is_subclass_of($type, FlagEnumType::class)) {
                return $this->mutateFlagEnumType($value, $type);
            }


        }
        return $value;
    }

    /**
     * @return int|null
     */
    public function getPrimaryKeyValue(): ?int
    {
        $pkField = $this->getPrimaryKeyField();
        $method = 'get' . str_replace(' ', '', ucwords(str_replace(['-', '_'], ' ', $pkField)));
        if (method_exists($this, $method)) {
            return $this->$method();
        }
        if (isset($this->attributes[$pkField])) {
            return $this->attributes[$pkField];
        }
        return null;
    }

    /**
     * Returns the raw values of the current attributes.
     *
     * @param boolean $onlyChanged
     *
     * @return array
     */
    public function toRawArray(bool $onlyChanged = false): array
    {
        $return = parent::toRawArray($onlyChanged);
        if (!isset($return[$this->primaryKeyField])) {
            foreach ($this->dates as $field) {
                unset($return[$field]);
            }
        }elseif(isset($return['updatedAtUtc'])){
            unset($return['updatedAtUtc']);
        }

        foreach ($return as  &$itm){
            if($itm instanceof EnumType){
                $itm = $itm->toInt();
            }
            if($itm instanceof FlagEnumType){
                $itm = $itm->toInt();
            }
            if($itm instanceof ConstClassType){
                $itm = strval($itm);
            }
        }

        return $return;
    }

    /**
     * Returns true if a property exists names $key, or a getter method
     * exists named like for __get().
     *
     * @param string $key
     *
     * @return boolean
     */
    public function __isset(string $key): bool
    {
        $key = $this->mapProperty($key);
        if (isset($this->attributes[$key])) {
            return true;
        }
        if (array_key_exists($key, $this->attributes)) {
            return false;
        }


        $method = 'get' . str_replace(' ', '', ucwords(str_replace(['-', '_'], ' ', $key)));

        if (method_exists($this, $method)) {
            return true;
        }

        return false;
    }

    #endregion methods

    #region utils

    /**
     * @param $value
     * @param $type
     * @return mixed|EnumType
     */
    protected function mutateEnumType($value, $type)
    {
        if ($value instanceof $type) {
            return $value;
        }

        try {
            if(is_numeric($value)){
                $value = intval($value);
            }
            return new $type($value);

        } catch (\Throwable $exp) {
            return $value;
        }
    }

    /**
     * @param $value
     * @param $type
     * @return mixed|FlagEnumType
     */
    protected function mutateFlagEnumType($value, $type)
    {
        if ($value instanceof $type) {
            return $value;
        }

        try {
            if(is_numeric($value)){
                $value = intval($value);
            }
            return new $type($value);

        } catch (\Throwable $exp) {
            return $value;
        }

    }

    /**
     * Converts the given string|timestamp|DateTime|Time instance
     * into a \CodeIgniter\I18n\Time object.
     *
     * @param $value
     *
     * @return \CodeIgniter\I18n\Time
     * @throws \Exception
     */
    protected function mutateDate($value)
    {

        if ($value instanceof Time)
        {
            return $value;
        }

        if ($value instanceof \DateTime)
        {
            return Time::instance($value);
        }

        if (is_numeric($value))
        {

            return Time::createFromTimestamp($value,'UTC');
        }

        if (is_string($value))
        {
            return Time::parse($value,'UTC');
        }

        return $value;
    }

    #endregion utils,

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {

        $data = [
            'attr' => $this->processDataForSerialize($this->attributes),
            'orgs' => $this->processDataForSerialize($this->original)
        ];
        return serialize($data);
    }

    protected function processDataForSerialize(array $data): array
    {
        $res = array();
        foreach ($data as $key => $value) {
            if ($value instanceof Time) {
                $val = array(
                    'tz' => $value->getTimezoneName(),
                    'ts' => strval($value)
                );
                $res[$key] = [
                    Time::class => $val
                ];

            } else {
                $res[$key] = $value;
            }
        }
        return $res;
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->attributes = $this->processDataForUnserialize($data['attr']);
        $this->original = $this->processDataForUnserialize($data['orgs']);
    }

    protected function processDataForUnserialize(array $data): array
    {
        $res = array();
        foreach ($data as $key => $value) {
            if (is_array($value) && isset($value[Time::class])) {
                $val = $value[Time::class];
                $time = Time::parse($val['ts'], $val['tz']);
                $res[$key] = $time;
            } else {
                $res[$key] = $value;
            }
        }
        return $res;
    }

    #endregion Serializable

}
