<?php


namespace GekTools\Commands;


use CodeIgniter\CLI\BaseCommand;
use CodeIgniter\CLI\CLI;
use CodeIgniter\Config\Services;
use Gek\Filesystem\Filesystem;
use Gek\PhpLang\CodeFactory;
use Gek\PhpLang\PhpTypeDeclared;
use Gek\PhpLang\UseItem;
use Gek\Infrastructure\Str;
use GekTools\Tools\BaseModelData;
use GekTools\Tools\Database\DbInfo;
use GekTools\Tools\DocComments\ModelDataTags;
use GekTools\Tools\Validations\Validation;
use GekTools\Tools\Validations\ValidationCollection;

class CreateModelData extends BaseCommand
{

    protected $group = 'Ci4Tools';
    protected $name = 'create:modeldata';
    protected $description = 'Yeni bir model data oluşturur.';

    /**
     * the Command's usage
     *
     * @var string
     */
    protected $usage = 'create:modeldata [model_data_name] [Options]';

    /**
     * the Command's Arguments
     *
     * @var array
     */
    protected $arguments = [
        'model_data_name' => 'Model data adı'
    ];

    /**
     * the Command's Options
     *
     * @var array
     */
    protected $options = [
        '-t' => 'veritabanı tablo adı',
        '-n' => 'namespace ayarlar.',
        '-p' => 'path'
    ];

    protected array $notSetterFields = [
        'createdAtUtc',
        'updatedAtUtc',
        'deletedAtUtc'
    ];

    /**
     * Actually execute a command.
     * This has to be over-ridden in any concrete implementation.
     *
     * @param array $params
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     * @throws \Throwable
     */
    public function run(array $params)
    {
        helper('inflector');

        $modelDataName = array_shift($params);
        if (empty($modelDataName)) {
            $modelDataName = CLI::prompt('Model Data adını girin');
        }
        $modelDataName = ucfirst($modelDataName);

        if (!Str::endsWith($modelDataName, 'ModelData')) {
            $modelDataName .= 'ModelData';
        }

        $ns = $params['-n'] ?? CLI::getOption('n');
        $homepath = APPPATH;
        $tableName = $params['-t'] ?? CLI::getOption('t');
        $path = $params['-p'] ?? CLI::getOption('p');

        if(empty($ns) && !empty($tableName)) {
            $dbInfo = new DbInfo();
            $tableComment = $dbInfo->getTableComment($tableName);
            if(!empty($tableComment) && $tableComment->isNamespace()){
                $ns = $tableComment->getNamespace();
            }
        }

        if (!empty($ns)) {
            // Get all namespaces
            $namespaces = Services::autoloader()->getNamespace();

            foreach ($namespaces as $namespace => $pt) {
                if ($namespace === $ns) {
                    $homepath = realpath(reset($pt));
                    break;
                }
            }
        } else {
            $ns = 'App';
        }



        if (!empty($path)) {
            $path = str_replace(["/", "\\"], DIRECTORY_SEPARATOR, $path);
            $path = trim($path);
            $path = trim($path, DIRECTORY_SEPARATOR);
        }

        $modelDataPath = $homepath . DIRECTORY_SEPARATOR . "ModelData" . DIRECTORY_SEPARATOR;
        if (!empty($path)) {
            $modelDataPath .= $path . DIRECTORY_SEPARATOR;
        }
        $modelDataPath .= $modelDataName . '.php';
        $modelDataNs = $ns . '\\ModelData';
        if (!empty($path)) {
            $modelDataNs .= "\\" . str_replace(DIRECTORY_SEPARATOR, "\\", $path);
        }
        $fs = new Filesystem();

        if ($fs->exists($modelDataPath)) {
            CLI::error('Dosya zaten mevcut : ' . str_replace($homepath, $ns, $modelDataPath));
            return;
        }

        $modelData = CodeFactory::classCreate($modelDataName)
            ->setExtends(BaseModelData::class)
            ->setNamespace($modelDataNs);

        $modelData->addConstructor();

        if(!empty($tableName)){
            $dbInfo = new DbInfo();
            $tableComment = $dbInfo->getTableComment($tableName);
            if(!empty($tableComment) &&
                $tableComment->isNamespace() &&
                $tableComment->getNamespace() != $ns
            ){
                CLI::error(Str::format("Girilen namespace ({0}) tabloda belirtilen namespace ({1}) ile uyuşmuyor.",$ns,$tableComment->getNamespace()));
            }
            $fieldInfos = $dbInfo->getFieldInfos($tableName);
            if(!empty($fieldInfos)){
                foreach ($fieldInfos as $fldInf){
                    $propName = $fldInf->getFieldName();
                    if($fldInf->issetComment()){
                        if($fldInf->getComment()->isPropName()){
                            $propName =  $fldInf->getComment()->getPropName();
                        }elseif ($fldInf->getComment()->getFieldName() != $propName){
                            $propName = $fldInf->getComment()->getFieldName();
                        }
                    }
                    $prop = $modelData->addField($propName)
                        ->setPublic()
                        ->setRegion('properties');
                    if($fldInf->issetComment()){
                        $prop->setDocComment($fldInf->getComment()->getDocComment());
                        /** @var ValidationCollection $validations */
                        $validations = $fldInf->getComment()->getValidations()
                            ->where(function (Validation $vldt){
                                return $vldt->getUseModelData();
                            })->toTypedClass(ValidationCollection::class);
                        if($validations->any()){
                            $prop->addCommentTag(ModelDataTags::VALIDATION, $validations->__toString());
                        }

                        $className = $fldInf->getComment()->getPhpType();
                        if(Str::startsWith($className,'?')){
                            $className = substr($className,1);
                        }
                        if(class_exists($className)){
                            if(!$modelData->getUses()->any(function (UseItem $item)use($className){
                                return $item->getFullName() == $className;
                            })){
                              $modelData->addUse($className);
                            }

                        }
                        $type =$fldInf->getComment()->getPhpType();
                        if(!Str::startsWith($type,'?') && $fldInf->getNullable()){
                            $type = '?'.$type;
                        }
                        $type = new PhpTypeDeclared($type);

                        $prop->addCommentTag('var', $type->renderForComment());
                    }else{
                        $type = $fldInf->getType();
                        if($fldInf->getNullable()){
                            $type .= '|null';
                        }
                        $prop->addCommentTag('var', $type);
                    }
                }
            }
        }



        $fs->dumpFile($modelDataPath, '<?php' . PHP_EOL . $modelData->toIndentedString());
        CLI::write('Dosya oluşturuldu : ' . CLI::color(str_replace($homepath, $ns, $modelDataPath), 'green'));
    }

}