<?php


namespace GekTools\Locale;


use GekTools\Locale\Entities\Language;
use GekTools\Locale\Entities\LocalizedProperty;
use GekTools\Locale\Models\LanguagesModel;
use GekTools\Locale\Models\LocalizedPropertiesModel;
use GekTools\Tools\BaseEntity;
use GekTools\Tools\Cache\DataCache;
use GekTools\Tools\Cache\Ttl;
use GekTools\Tools\Collections\PagedResult;

class LocaleService
{

    #region fields

    /**
     * @var LocalizedPropertiesModel
     */
    protected LocalizedPropertiesModel $localizedPropertiesModel;

    /**
     * @var LanguagesModel
     */
    protected LanguagesModel $languagesModel;

    /**
     * @var DataCache
     */
    protected DataCache $dataCache;



    #endregion fields

    #region ctor

    public function __construct()
    {
        $this->languagesModel = model(LanguagesModel::class);
        $this->localizedPropertiesModel = model(LocalizedPropertiesModel::class);
        $this->dataCache = service('dataCache',true);
        $this->languagesModel->onAfterDelete([$this,'cleanLanguagesCache']);
        $this->languagesModel->onAfterInsert([$this,'cleanLanguagesCache']);
        $this->languagesModel->onAfterUpdate([$this,'cleanLanguagesCache']);
    }

    #endregion ctor

    #region properties

    /**
     * @return LocalizedPropertiesModel
     */
    public function getLocalizedPropertiesModel(): LocalizedPropertiesModel
    {
        return $this->localizedPropertiesModel;
    }

    /**
     * @return LanguagesModel
     */
    public function getLanguagesModel(): LanguagesModel
    {
        return $this->languagesModel;
    }

    #endregion properties

    #region methods

    /**
     * @param BaseEntity $entity
     * @param int|null $languageId
     * @return PagedResult|LocalizedProperty[]
     */
    public function getAllLocalizedPropertiesByEntity(BaseEntity $entity,?int $languageId = null){
        $fi = LocalizedProperty::getFieldsInfo();
        $argArr = [
            $fi->entityFullName = get_class($entity),
            $fi->entityId = $entity->getPrimaryKeyValue(),
        ];
        if($languageId !== null){
            $argArr[$fi->languageId] = $languageId;
        }
        $cacheKey = $this->dataCache::createKey(LocalizedProperty::class,$argArr);
        return $this->dataCache->getOrSave(
            $cacheKey,
            [$this,'getAllLocalizedPropertiesByEntityNoCache'],
            [$entity,$languageId],
            Ttl::fromDays(15)
        );

    }

    /**
     * @param BaseEntity $entity
     * @param int|null $languageId
     * @return PagedResult|LocalizedProperty[]
     */
    public function getAllLocalizedPropertiesByEntityNoCache(BaseEntity $entity,?int $languageId = null){
        $fi = LocalizedProperty::getFieldsInfo();
        if($languageId !== null){
            $this->getLocalizedPropertiesModel()
                ->where($fi->languageId, $languageId);
        }
        return $this->getLocalizedPropertiesModel()
            ->where($fi->entityFullName, get_class($entity))
            ->where($fi->entityId,$entity->getPrimaryKeyValue())
            ->getAll();

    }

    /**
     * @param string $systemName
     * @return Language|array|BaseEntity|object|null
     */
    public function getLanguageBySystemName(string $systemName){
        $fi = Language::getFieldsInfo();
        $cacheKey = $this->dataCache::createKey(Language::class,[$fi->systemName => $systemName]);
        return $this->dataCache->getOrSave(
            $cacheKey,
            [$this,'getLanguageBySystemNameNoCache'],
            [$systemName],
            Ttl::fromDays(45)
        );
    }

    /**
     * @param string $systemName
     * @return Language|array|BaseEntity|object|null
     */
    public function getLanguageBySystemNameNoCache(string $systemName){
        $fi = Language::getFieldsInfo();

        return $this->getLanguagesModel()
            ->where($fi->systemName, $systemName)
            ->first();
    }


    /**
     * @param bool $onlyPublished
     * @return PagedResult|Language[]
     */
    public function getAllLanguages(bool $onlyPublished = false){
        $cacheKey = $this->dataCache::createKey(Language::class);
        $cacheKey .= 'getAllLanguages(onlyPublished-'.$onlyPublished.')';
        return $this->dataCache->getOrSave(
            $cacheKey,
            [$this,'getAllLanguagesNoCache'],
            [$onlyPublished],
            Ttl::fromDays(30)
        );
    }

    /**
     * @param int $id
     * @return Language|array|BaseEntity|object|null
     */
    public function getLanguageById(int $id){
        $fi = Language::getFieldsInfo();
        $cacheKey = $this->dataCache::createKey(Language::class,[$fi->id => $id]);
        return $this->dataCache->getOrSave(
            $cacheKey,
            [$this,'getLanguageByIdNoCache'],
            [$id],
            Ttl::fromDays(45)
        );
    }

    /**
     * @param int $id
     * @return Language|array|BaseEntity|object|null
     */
    public function getLanguageByIdNoCache(int $id){
        return $this->getLanguagesModel()->getById($id);
    }

    /**
     * @param bool $onlyPublished
     * @return PagedResult|Language[]
     */
    public function getAllLanguagesNoCache(bool $onlyPublished = false){

        if($onlyPublished){
            $fi = Language::getFieldsInfo();
            $this->getLanguagesModel()
                ->where($fi->published, true);
        }
        return $this->getLanguagesModel()->getAll();
    }


    #endregion methods

    #region utils

    /**
     * @internal
     * @param $eventData
     * @return mixed
     */
    public function cleanLanguagesCache($eventData){
        $key = $this->dataCache::createKey(Language::class);
        $this->dataCache->deleteByStartWith($key);
        return $eventData;
    }

    #endregion utils

    #region statics

    public static function instance():self {
        return service('localeService',true);
    }

    #endregion statics

}