<?php


namespace GekTools\GenericAttributes;


use Gek\Infrastructure\Enum;
use Gek\Infrastructure\EnumType;
use Gek\Infrastructure\FlagEnum;
use Gek\Infrastructure\FlagEnumType;
use Gek\Infrastructure\Str;
use GekTools\GenericAttributes\Entities\GenericAttribute;
use GekTools\GenericAttributes\Models\GenericAttributesModel;
use GekTools\Tools\BaseEntity;
use GekTools\Tools\Cache\DataCache;

class GenericAttributesService
{
    #region fields

    /**
     * @var GenericAttributesModel
     */
    protected GenericAttributesModel $model;

    /** @var DataCache */
    protected DataCache $dataCache;

    #endregion fields

    #region ctor

    public function __construct()
    {
        $this->model = model(GenericAttributesModel::class);
        $this->dataCache = DataCache::instance();
    }

    #endregion ctor

    #region properties

    /**
     * @return GenericAttributesModel
     */
    public function getModel(): GenericAttributesModel
    {
        return $this->model;
    }

    #endregion properties

    #region dbAccess

    /**
     * @param BaseEntity $baseEntity
     * @param string $key
     * @return GenericAttribute|array|object|null
     */
    public function getByEntityAndKey(BaseEntity $baseEntity, string $key)
    {
        $ent = GenericAttribute::class . '-' . get_class($baseEntity);
        $cachekey = $this->dataCache::createKey(
            $ent,
            [
                $baseEntity->getPrimaryKeyField() => $baseEntity->getPrimaryKeyValue(),
                'attr' => $key,
            ]
        );
        return $this->dataCache->getOrSave($cachekey, [$this, 'realGetByEntityAndKey'], [$baseEntity, $key], 6000);
    }

    public function realGetByEntityAndKey(BaseEntity $baseEntity, string $key)
    {
        $entityFullName = get_class($baseEntity);
        $entityId = $baseEntity->getPrimaryKeyValue();
        $fi = GenericAttribute::getFieldsInfo();
        return $this->model
            ->where($fi->entityId, $entityId)
            ->where($fi->entityFullName, $entityFullName)
            ->where($fi->key, $key)
            ->first();
    }

    /**
     * @param BaseEntity $baseEntity
     * @param string $key
     * @param mixed $value
     * @param string|null $type
     * @param bool|null $nullable
     * @return bool
     * @throws \ReflectionException
     */
    public function setByEntityAndKey(BaseEntity $baseEntity, string $key, $value, ?string $type = null, ?bool $nullable = false)
    {
        if ($type !== null && Str::startsWith($type, '?')) {
            $type = substr($type, 1);
            $nullable = true;
        }
        $genAttr = $this->getByEntityAndKey($baseEntity, $key);
        if (!empty($genAttr)) {
            if ($type == null) {
                $type = $genAttr->getType();
            }
            if ($nullable == null) {
                $nullable = $genAttr->getNullable();
            }
        } else {
            $genAttr = new GenericAttribute();
            $genAttr->setEntityFullName(get_class($baseEntity))
                ->setEntityId($baseEntity->getPrimaryKeyValue())
                ->setKey($key);
        }
        if ($type === null) {
            $type = $this->getTypeFromValue($value);
        } elseif ($value !== null) {
            $valType = $this->getTypeFromValue($value);
            if ($type != $valType) {
                $type = $valType;
            }
        }
        if ($nullable === null) {
            $nullable = $value === null;
        }
        $strValue = $this->valueToString($value, $type, $nullable);
        $genAttr->setType($type)
            ->setNullable($nullable)
            ->setValue($strValue);
        $res = $this->model->save($genAttr);

        $ent = GenericAttribute::class . '-' . get_class($baseEntity);
        $cachekey = $this->dataCache::createKey(
            $ent,
            [
                $baseEntity->getPrimaryKeyField() => $baseEntity->getPrimaryKeyValue(),
                'attr' => $key,
            ]
        );
        $this->dataCache->delete($cachekey);
        return $res;
    }

    /**
     * @param BaseEntity $baseEntity
     * @param string $key
     * @return bool|float|int|mixed|null
     * @throws \Exception
     */
    public function getValueByEntityAndKey(BaseEntity $baseEntity, string $key)
    {
        $genAttr = $this->getByEntityAndKey($baseEntity, $key);
        if (empty($genAttr)) {
            return null;
        }
        return $this->stringToValue($genAttr->getValue(), $genAttr->getType(), $genAttr->getNullable());
    }


    #endregion dbAccess

    #region methods


    #endregion methods

    #region utils

    /**
     * @param $value
     * @param bool $nullable
     * @return string
     */
    public function getTypeFromValue($value, bool &$nullable = false)
    {
        $default = 'string';
        if ($value === null) {
            $nullable = true;
            return $default;
        }
        if (is_float($value)) {
            return 'float';
        }
        if (is_int($value)) {
            return 'int';
        }
        if (is_string($value)) {
            return 'string';
        }
        if (is_array($value)) {
            return 'array';
        }
        if (is_object($value) && !($value instanceof \stdClass)) {
            return get_class($value);
        }
        if (is_object($value)) {
            return 'object';
        }
        return $default;
    }

    /**
     * @param mixed $value
     * @param string $type
     * @param bool $nullable
     * @return string|null
     * @throws \Exception
     */
    public function valueToString($value, string $type, bool $nullable): ?string
    {
        if ($value === null && $nullable) {
            return null;
        }
        if ($value === null) {
            throw new \Exception('value null olamaz.');
        }
        switch (true) {
            case $type === 'string':
                return $value;
                break;
            case $type === 'int':
            case $type === 'float':
            case is_subclass_of($type, EnumType::class):
            case is_subclass_of($type, FlagEnumType::class):
                return strval($value);
                break;
            case $type === 'bool':
                return $value ? '1' : '0';
                break;

            case $type === 'array':
            case $type === 'object':
            default:
                return serialize($value);
                break;
        }
    }

    /**
     * @param string|null $strValue
     * @param string $type
     * @param bool $nullable
     * @return bool|float|int|mixed|null
     */
    public function stringToValue(?string $strValue, string $type, bool $nullable)
    {
        if ($strValue === null && $nullable) {
            return null;
        }
        switch (true) {
            case $type === 'string':
                return $strValue;
                break;
            case $type === 'float':
                return floatval($strValue);
                break;
            case $type === 'bool':
                return boolval($strValue);
                break;
            case is_subclass_of($type, EnumType::class):
            case is_subclass_of($type, FlagEnumType::class):
                return new $type($strValue);
            case $type === 'int':
            case is_subclass_of($type, Enum::class):
            case is_subclass_of($type, FlagEnum::class):
                return intval($strValue);
                break;
            case $type === 'array':
            case $type === 'object':
            default:
                return unserialize($strValue);
                break;
        }
    }


    #endregion utils

    #region static

    /**
     * @return \GekTools\GenericAttributesService
     */
    public static function instance(): self
    {
        return service('genericAttributesService', true);
    }

    #endregion static
}
