<?php


namespace GekTools\Tools\Xml;


use Gek\Collections\Typed\StringDictionary;
use Gek\Infrastructure\Str;

class XmlNode
{
    #region ctor

    /**
     * XmlNode constructor.
     * @param string|null $tagName
     * @param string|null $text
     * @param bool|null $ignoreNs
     */
    public function __construct(?string $tagName = null, ?string $text = null, ?bool $ignoreNs = null)
    {
        $this->childNodes = new XmlNodeList();
        $this->attributes = new StringDictionary();
        if (!is_null($tagName)) {
            if (strpos($tagName, ':') !== false) {
                $tgNms = explode(':', $tagName);
                $this->prefixNs = $tgNms[0];
                $this->tagName = $tgNms[1];
            } else {
                $this->tagName = $tagName;
            }

        }
        if (!is_null($text)) {
            $this->text = $text;
        }
        if (!is_null($ignoreNs)) {
            $this->ignoreNs = $ignoreNs;
        }
    }

    #endregion ctor

    #region properties

    /**
     * @var string
     */
    public string $tagName;

    /**
     * @var bool
     */
    public bool $isSelfClose = false;

    /**
     * @var bool
     */
    public bool $isRoot = false;

    /**
     * @var string|null
     */
    public ?string $prefixNs = null;

    /**
     * @var StringDictionary|string[]
     */
    public StringDictionary $attributes;


    /**
     * @var \Gek\Config\xml\XmlNode|null
     */
    public ?XmlNode $parentNode = null;

    /**
     * @var XmlNodeList|XmlNode[]
     */
    public XmlNodeList $childNodes;

    /**
     * @var string|null
     */
    public ?string $text = null;

    /**
     * @var string
     */
    public ?string  $comments;

    #endregion properties

    #region methods

    public function __toString()
    {
        return $this->toXmlString(0);
    }

    /**
     * @param int $indentLevel
     * @return string
     */
    public function toXmlString(int $indentLevel = 0)
    {
        $indentStr = '    ';

        if (empty($this->text) && ! $this->childNodes->any()) {
            $this->isSelfClose = true;
        }
        $tagName = $this->getTagName();
        $xmlRes =  $this->renderComments(str_repeat($indentStr, $indentLevel));
        $xmlRes .= str_repeat($indentStr, $indentLevel) . "<" . $tagName;
        if ($this->attributes->any()) {
            foreach ($this->attributes as $key => $value) {
                $xmlRes .= ' ' . $key;
                $xmlRes .= '=' . '"' . str_replace('"', '\"', $value) . '"';
            }
        }
        if ($this->isSelfClose) {
            $xmlRes .= '/>'.PHP_EOL;
        } else {
            $xmlRes .= '>';
            if ($this->text !== null) {
                $xmlRes .= $this->text;
            } elseif (!empty($this->childNodes)) {
                $xmlRes .= PHP_EOL;
                foreach ($this->childNodes as $node) {
                    if ($node !== null) {
                        $xmlRes .= $node->toXmlString($indentLevel + 1);
                    }
                }
                $xmlRes .= str_repeat($indentStr, $indentLevel);
                // $xmlRes .= PHP_EOL;
            }

            $xmlRes .= '</' . $tagName . '>' . PHP_EOL;
        }
        if ($this->isRoot === true) {
            $xmlRes = '<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL . $xmlRes;
        }
        return $xmlRes;
    }

    /**
     * @param string $strAttributes
     */
    public function parseAttributes($strAttributes)
    {
        $strLinesArr = array();
        if (strpos($strAttributes, "\n") !== false) {
            $strLinesArr = explode("\n", $strAttributes);
        } else {
            $strLinesArr[] = $strAttributes;
        }
        foreach ($strLinesArr as $strLine) {
            if (empty($strLine)) {
                continue;
            }
            $lnArr = explode('=', $strLine);
            $key = $lnArr[0];
            $key = trim($key);
            $val = $lnArr[1];
            $val = trim($val);
            $val = trim($val, '\'');
            $val = trim($val, '"');
            $this->attributes[$key] = $val;
        }
    }


    /**
     * @param \
     */
    public function addChild(XmlNode $xmlNode)
    {
        $this->childNodes->add($xmlNode);
    }

    /**
     * @param string $key
     * @param string $value
     */
    public function addAttribute(string $key, string $value){
        if(! $this->attributes->tryAddKeyValue($key,$value)){
            $this->attributes[$key] = $value;
        }
    }

    #endregion methods

    #region utils

    /**
     * @param string $indent
     * @return string
     */
    private function renderComments(string $indent = ''): string
    {
        $cmnt = '';
        if (isset($this->comments) && !empty($this->comments)) {
            $cmnt .= $indent . '<!-- ' . $this->comments . ' -->' . PHP_EOL;
        }
        return $cmnt;
    }

    /**
     * @return string
     */
    private function getTagName(): string
    {
        $tagname = $this->tagName;

        if (!empty($this->prefixNs) && ($this->ignoreNs == false)) {
            $tagname = Str::Format('{0}:{1}', $this->prefixNs, $this->tagName);
        }
        return $tagname;
    }


    #endregion utils
}