<?php


namespace GekTools\Tools\Traits;


use CodeIgniter\Database\BaseConnection;
use CodeIgniter\Database\Forge;
use CodeIgniter\I18n\Time;
use Gek\Collections\Typed\StringList;
use Gek\Infrastructure\ConstClass;
use Gek\Infrastructure\ConstClassType;
use Gek\Infrastructure\Enum;
use Gek\Infrastructure\EnumType;
use Gek\Infrastructure\FlagEnumType;
use Gek\Infrastructure\Str;
use GekTools\Tools\Contrats\ICreatedAtUtc;
use GekTools\Tools\Contrats\IDisplayOrder;
use GekTools\Tools\Contrats\ISoftDeletable;
use GekTools\Tools\Contrats\ISystemName;
use GekTools\Tools\Contrats\IUpdatedAtUtc;
use GekTools\Tools\ControlTypes;
use GekTools\Tools\Database\DbTypes;
use GekTools\Tools\Database\DbInfo;
use GekTools\Tools\Database\TableComment;
use GekTools\Tools\Database\TableFieldComment;
use GekTools\Tools\Database\TableFieldInfo;
use GekTools\Tools\Database\TableInfo;
use GekTools\Tools\DocComments\ModelDataTags;

/**
 * Trait MigrationTrait
 * @package GekTools\Tools\Traits
 * @property  string $DBGroup
 * @property  BaseConnection $db
 * @property  Forge $forge
 */
trait MigrationTrait
{

    #region fields


    private ?TableInfo $tableInfo;

    public ?StringList $tableInterfaces = null;

    public ?StringList $removedTableInterfaces = null;

    /**
     * @var string | null
     */
    public ?string $entityName = null;

    /**
     * @var string|null
     */
    public ?string $entityBaseClass = null;

    public ?string $namespace = null;

    /**
     * @var StringList|null
     */
    private ?StringList $keys = null;

    /**
     * @var StringList|null
     */
    private ?StringList $removeKeys = null;

    /**
     * @var bool
     */
    public $changeMode = false;

    /**
     * @var StringList|null
     */
    private ?StringList $prKeys = null;

    /**
     * @var array
     */
    private array $fieldsArray = array();

    private array $changedFieldsArray = array();

    private array $removedFields = array();

    #endregion fields

    #region methods

    #region standard fields

    /**
     * @param string $fieldName
     * @param bool $nullable
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @return self
     */
    public function addBoolField(string $fieldName, bool $nullable = FALSE,
                                 $defaultValue = null, ?TableFieldComment $comments = null):self
    {


        if($comments === null){
            $comments = new TableFieldComment();
        }
        if(!$comments->isPhpType()){
            $comments->setPhpType('bool');
        }

        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::CHECKBOX);
        }

        $fieldInfo = new TableFieldInfo($fieldName);

        if ($nullable == false && $defaultValue === null) {
            $defaultValue = false;
        }
        $comments->getValidations()->addLessThanEqualTo(1,true,false);

        $fieldInfo->setDefault($defaultValue)
            ->setDbType(DbTypes::TINYINT())
            ->setLength(1)
            ->setNullable($nullable)
            ->setUnsigned(false)
            ->setUnique(false)
            ->setComment($comments);

        $this->addField($fieldInfo);
        return $this;

    }

    /**
     * @param string $fieldName
     * @param bool $nullable
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @return self
     */
    public function addTextField(string $fieldName,
                                 bool $nullable = false,
                                 $defaultValue = null,
                                 ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }
        if(!$comments->isPhpType()){
            $comments->setPhpType('string');
        }

        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::TEXTAREA);
        }
        $comments->getValidations()->addAlpha();

        $fieldInfo = new TableFieldInfo($fieldName);
        $fieldInfo->setDbType(DbTypes::TEXT())
            ->setNullable($nullable)
            ->setUnsigned(false)
            ->setUnique(false)
            ->setDefault($defaultValue)
            ->setComment($comments);

        $this->addField($fieldInfo);
        return $this;
    }

    /**
     * @param string $fieldName
     * @param int $maxByteSize
     * @param bool $nullable
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @return self
     * @throws \Exception
     */
    public function addBlobField(string $fieldName,
                                 int $maxByteSize,
                                 bool $nullable = false,
                                 $defaultValue = null,
                                 ?TableFieldComment $comments = null):self {

        if($comments === null){
            $comments = new TableFieldComment();
        }
        if(!$comments->isPhpType()){
            $comments->setPhpType("string");
        }



        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::FILE);
        }

        $fieldInfo = new TableFieldInfo($fieldName);

        $tinyBlobMax = pow(2,8);
        $blobMax = pow(2,16);
        $mediumBlobMax = pow(2,24);
        $longBlobMax = pow(2,32);

        if($maxByteSize <= $tinyBlobMax){
            $dbType = DbTypes::TINYBLOB();
        }elseif ($maxByteSize <= $blobMax){
            $dbType = DbTypes::BLOB();
        }elseif ($maxByteSize <= $mediumBlobMax){
            $dbType = DbTypes::MEDIUMBLOB();
        }elseif ($maxByteSize <= $longBlobMax){
            $dbType = DbTypes::LONGBLOB();
        }elseif ($maxByteSize > $longBlobMax){
            throw new \Exception('Blob alanı azami ' . $longBlobMax . ' byte  uzunlukta olabilir.');
        }

        $comments->getValidations()->addMaxSize($fieldName,(int)($maxByteSize / 1024),false);

        $fieldInfo->setDbType($dbType)
            ->setNullable($nullable)
            ->setUnsigned(false)
            ->setUnique(false)
            ->setDefault($defaultValue)
            ->setComment($comments);

        $this->addField($fieldInfo);
        return $this;
    }


    /**
     * @param string $fieldName
     * @param bool $nullable
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @return self
     */
    public function addJsonField(string $fieldName,
                                 bool $nullable = true,
                                 $defaultValue = null,
                                 ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }
        if(!$comments->isPhpType()){
            $comments->setPhpType("string");
        }

        $comments->setJson(true);

        $comments->getValidations()->addValidJson();

        $fieldInfo = new TableFieldInfo($fieldName);

        $fieldInfo->setDbType(DbTypes::JSON())
            ->setDbType(DbTypes::JSON())
            ->setNullable($nullable)
            ->setUnsigned(false)
            ->setUnique(false)
            ->setDefault($defaultValue)
            ->setComment($comments);



        $this->addField($fieldInfo);
        return $this;
    }

    /**
     * @param $fieldName
     * @param int $size
     * @param bool $nullable
     * @param bool $unique
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @return self
     */
    public function addStringField(string $fieldName,
                                   int $size,
                                   bool $nullable = false,
                                   bool $unique = false,
                                   $defaultValue = null,
                                   ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }
        if(!$comments->isPhpType()){
            $comments->setPhpType('string');
        }
        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::TEXTBOX);
        }

        $comments->getValidations()->addMaxLength($size);
        $comments->getValidations()->addString();

        $fieldInfo = new TableFieldInfo($fieldName);

        $fieldInfo->setDbType(DbTypes::VARCHAR())
            ->setLength($size)
            ->setNullable($nullable)
            ->setUnsigned(false)
            ->setUnique($unique)
            ->setDefault($defaultValue)
            ->setComment($comments);

        $this->addField($fieldInfo);
        return $this;
    }

    /**
     * @param string $fieldName
     * @param bool $nullable
     * @param bool $unsigned
     * @param bool $unique
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @param bool $autoIncrement
     * @return self
     */
    public function addIntField(string $fieldName,
                                bool $nullable = false,
                                bool $unsigned = false,
                                bool $unique = false,
                                $defaultValue = null,
                                ?TableFieldComment $comments = null,
                                bool $autoIncrement = false):self
    {
        if($comments == null){
            $comments = new TableFieldComment();
        }
        if(!$comments->isPhpType()){
            $comments->setPhpType('int');
        }

        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::INT);
        }

        if($unsigned){
            $comments->getValidations()->addGreaterThanEqualTo(0);
        }
        $comments->getValidations()->addInteger();

        $fieldInfo = new TableFieldInfo($fieldName);

        $fieldInfo->setDbType(DbTypes::INT())
            ->setNullable($nullable)
            ->setUnsigned($unsigned)
            ->setUnique($unique)
            ->setDefault($defaultValue)
            ->setComment($comments)
            ->setAutoIncrement($autoIncrement);

        $this->addField($fieldInfo);
        return $this;
    }

    /**
     * @param string $fieldName
     * @param bool $nullable
     * @param bool $unsigned
     * @param bool $unique
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @param bool $autoIncrement
     * @return self
     */
    public function addLongField(string $fieldName,
                                 bool $nullable = false,
                                 bool $unsigned = false,
                                 bool $unique = false,
                                 $defaultValue = null,
                                 ?TableFieldComment $comments = null,
                                 bool $autoIncrement = false):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }
        if(!$comments->isPhpType()){
            $comments->setPhpType('int');
        }


        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::INT);
        }

        if($unsigned){
            $comments->getValidations()->addGreaterThanEqualTo(0);
        }

        $comments->getValidations()->addNumeric();

        $fieldInfo = new TableFieldInfo($fieldName);

        $fieldInfo->setDbType(DbTypes::BIGINT())
            ->setNullable($nullable)
            ->setUnsigned($unsigned)
            ->setUnique($unique)
            ->setDefault($defaultValue)
            ->setComment($comments)
            ->setAutoIncrement($autoIncrement);

        $this->addField($fieldInfo);
        return $this;
    }

    /**
     * @param string $fieldName
     * @param null | mixed $size
     * @param bool $nullable
     * @param bool $unique
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @return self
     */
    public function addFloatField(string $fieldName,
                                  $size = '18,4',
                                  bool $nullable = false,
                                  bool $unique = false,
                                  $defaultValue = null,
                                  ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }
        if(!$comments->isPhpType()){
            $comments->setPhpType('float');
        }


        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::FLOAT);
        }

        $sizeParts = explode(',',$size);
        $tam = (int) $sizeParts[0];
        $ondalik = isset($sizeParts[1]) ? (int)$sizeParts[1] : 0;
        $maxVal = str_repeat('9',$tam);
        if($ondalik > 0){
            $maxVal .= '.' . str_repeat('9',$ondalik);
        }

        $comments->getValidations()->addLessThanEqualTo($maxVal);
        $comments->getValidations()->addDecimal();

        $fieldInfo = new TableFieldInfo($fieldName);

        $fieldInfo->setDbType(DbTypes::DECIMAL())
            ->setLength($size)
            ->setNullable($nullable)
            ->setUnsigned(false)
            ->setUnique($unique)
            ->setDefault($defaultValue)
            ->setComment($comments);

        $this->addField($fieldInfo);
        return $this;
    }


    /**
     * @param string $fieldName
     * @param bool $nullable
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @return self
     */
    public function addTimestampField(string $fieldName,
                                      bool $nullable = true,
                                      $defaultValue = null,
                                      ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }
        if(!$comments->isPhpType()){
            $comments->setPhpType('string');
        }

        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::DATETIME);
        }
        $comments->getValidations()->addValidDate('Y-m-d H:i:s');

        $fieldInfo = new TableFieldInfo($fieldName);

        $fieldInfo->setDbType(DbTypes::TIMESTAMP())
            ->setNullable($nullable)
            ->setUnsigned(false)
            ->setUnique(false)
            ->setDefault($defaultValue)
            ->setComment($comments);


        $this->addField($fieldInfo);
        return $this;
    }

    /**
     * @param string $fieldName
     * @param bool $nullable
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @return self
     */
    public function addDateTimeField(string $fieldName,
                                     bool $nullable = false,
                                     $defaultValue = null,
                                     ?TableFieldComment $comments = null):self
    {

        if($comments === null){
            $comments = new TableFieldComment();
        }
        if(!$comments->isPhpType()){
            $comments->setPhpType(Time::class);
        }

        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::DATETIME);
        }

        $comments->getValidations()->addValidDate('Y-m-d H:i:s');

        $fieldInfo = new TableFieldInfo($fieldName);

        $fieldInfo->setDbType(DbTypes::DATETIME())
            ->setNullable($nullable)
            ->setUnsigned(false)
            ->setUnique(false)
            ->setDefault($defaultValue)
            ->setComment($comments);

        $this->addField($fieldInfo);
        return $this;
    }

    /**
     * @param string $fieldName
     * @param bool $nullable
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @return self
     */
    public function addDateField(string $fieldName,
                                 bool $nullable = false,
                                 $defaultValue = null,
                                 ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }
        if(!$comments->isPhpType()){
            $comments->setPhpType(Time::class);
        }

        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::DATE);
        }

        $comments->getValidations()->addValidDate('Y-m-d');

        $fieldInfo = new TableFieldInfo($fieldName);

        $fieldInfo->setDbType(DbTypes::DATE())
            ->setNullable($nullable)
            ->setUnsigned(false)
            ->setUnique(false)
            ->setDefault($defaultValue)
            ->setComment($comments);


        $this->addField($fieldInfo);
        return $this;
    }

    /**
     * @param string $fieldName
     * @param bool $nullable
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @return self
     */
    public function addTimeField(string $fieldName,
                                 bool $nullable = false,
                                 $defaultValue = null,
                                 ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }
        if(!$comments->isPhpType()){
            $comments->setPhpType(Time::class);
        }

        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::TIME);
        }
        $comments->getValidations()->addValidDate('H:i:s');

        $fieldInfo = new TableFieldInfo($fieldName);

        $fieldInfo->setDbType(DbTypes::TIME())
            ->setNullable($nullable)
            ->setUnsigned(false)
            ->setUnique(false)
            ->setDefault($defaultValue)
            ->setComment($comments);


        $this->addField($fieldInfo);
        return $this;
    }


    #endregion standard fields

    #region specific fields

    /**
     * @param string $fieldName
     * @param bool $nullable
     * @param bool $unique
     * @param mixed|null $defaultValue
     * @param TableFieldComment|null $comments
     * @return self
     */
    public function addMoneyField(string $fieldName,
                                  bool $nullable = false,
                                  bool $unique = false,
                                  $defaultValue = null,
                                  ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }



        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::MONEY);
        }

        $this->addFloatField(
            $fieldName,
            '18,4',
            $nullable,
            $unique,
            $defaultValue,
            $comments
        );
        $this->addKey($fieldName);
        return $this;
    }

    /**
     * @param string $fieldName
     * @return self
     */
    public function removeMoneyField($fieldName):self {
        $this->removeField($fieldName);
        $this->removeKey($fieldName);
        return $this;
    }

    /**
     * @param string $fieldName
     * @param bool $nullable
     * @param bool $unique
     * @param null|TableFieldComment $comments
     * @return self
     */
    public function addEmailField(string $fieldName = 'Email',
                                  bool $nullable = false,
                                  bool $unique = false,
                                  ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }

        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::EMAIL);
        }

        $comments->getValidations()->addValidEmail();

        $this->addStringField(
            $fieldName,
            300,
            $nullable,
            $unique,
            null,
            $comments);
        if ($unique == false) {
            $this->addKey($fieldName);
        }
        return $this;
    }

    /**
     * @param string $fieldName
     * @param bool $unique
     * @return self
     */
    public function removeEmailField(string $fieldName = 'Email', bool $unique = false):self
    {
        $this->removeField($fieldName);
        if ($unique == false) {
            $this->removeKey($fieldName);
        }
        return $this;
    }


    /**
     * @param string $fieldName
     * @param null | string $flagEnumType
     * @param null | int $default_value
     * @param TableFieldComment|null $comments
     * @return self
     */
    public function addFlagField(string $fieldName,
                                 ?string $flagEnumType = null,
                                 $default_value = null,
                                 ?TableFieldComment $comments = null):self
    {

        if($comments === null){
            $comments = new TableFieldComment();
        }



        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::FLAG);
        }

        $nullable = false;
        if($flagEnumType !== null){
            if(Str::startsWith($flagEnumType,'?')){
                $flagEnumType = substr($flagEnumType,1);
                $nullable = true;
            }
            if(class_exists($flagEnumType) && is_subclass_of($flagEnumType,FlagEnumType::class)){
                $comments->setPhpType($flagEnumType);
            }
        }

        $this->addLongField($fieldName, $nullable,
            false, false, $default_value, $comments);
        $this->addKey($fieldName);
        return $this;
    }

    /**
     * @param string $field_name
     * @return self
     */
    public function removeFlagField($field_name):self {
        $this->removeField($field_name);
        $this->removeKey($field_name);
        return $this;
    }

    /**
     * @param string $fieldName
     * @param null | string $enumType
     * @param null | int $defaultValue
     * @param TableFieldComment|null $comments
     * @return self
     */
    public function addEnumField(string $fieldName,
                                 ?string $enumType = null,
                                 ?int $defaultValue = null,
                                 ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }

        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::SELECT);
        }


        if($enumType !== null){
            if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::OPTIONS_FIELD)){
                $comments->getDocComment()->getTags()
                    ->addOrUpdateTag(ModelDataTags::OPTIONS_FIELD, $enumType);
            }
            $nullable = false;
            if($enumType !== null){
                if(Str::startsWith($enumType,'?')){
                    $enumType = substr($enumType,1);
                    $nullable = true;
                }
                if(class_exists($enumType)){
                    if(is_subclass_of($enumType,Enum::class)){
                        $values = $enumType::getEnumValues();
                        $comments->getValidations()->addInList($values);
                        if(is_subclass_of($enumType,EnumType::class)){
                            $comments->setPhpType($enumType);
                        }
                    }
                }
            }
        }

        $this->addIntField($fieldName, $nullable, false,
            false, $defaultValue, $comments);
        return $this;
    }

    /**
     * @param string $fieldName
     * @param int $size
     * @param null | string $constClassType
     * @param null | string $default_value
     * @param TableFieldComment|null $comments
     * @return self
     */
    public function addConstantsClassField(string $fieldName,
                                           int $size = 100,
                                           ?string $constClassType = null,
                                           ?string $default_value = null,
                                           ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }

        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()
                ->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::SELECT);
        }

        if($constClassType !== null){
            if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::OPTIONS_FIELD)){
                $comments->getDocComment()->getTags()
                    ->addOrUpdateTag(ModelDataTags::OPTIONS_FIELD, $constClassType);
            }

            if(class_exists($constClassType) && is_subclass_of($constClassType,ConstClass::class)){
                $values = array_values($constClassType::getConstValues());
                $comments->getValidations()->addInList($values);
                if(is_subclass_of($constClassType,ConstClassType::class)){
                    $comments->setPhpType($constClassType);
                }
            }
        }

        $this->addStringField($fieldName, $size, false, false,
            $default_value, $comments);
        return $this;
    }

    /**
     * @param $fieldName
     * @param bool $nullable
     * @param bool $unique
     * @param TableFieldComment|null $comments
     * @param bool $setDefaultFunc
     * @return self
     */
    public function addGuidField(string $fieldName,
                                 bool $nullable = false,
                                 bool $unique = true,
                                 ?TableFieldComment $comments = null
    ):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }



        $comments->getDocComment()->getTags()
            ->addOrUpdateTag(ModelDataTags::GUID);
        if(!$comments->getDocComment()->getTags()->containsTag(ModelDataTags::CONTROL)){
            $comments->getDocComment()->getTags()->addOrUpdateTag(ModelDataTags::CONTROL,ControlTypes::HIDDEN);
        }
        $comments->getValidations()->addMaxLength(36);
        $comments->getValidations()->addMinLength(36);

        $this->addStringField(
            $fieldName,
            36,
            $nullable,
            $unique,
            null,
            $comments
        );
        if ($unique == false) {
            $this->addKey($fieldName);
        }
        return $this;
    }

    /**
     * @param string $field_name
     * @param bool $unique
     * @return self
     */
    public function removeGuidField(string $field_name, bool $unique = true):self
    {
        $this->removeField($field_name);
        if ($unique == false) {
            $this->removeKey($field_name);
        }
        return $this;
    }


    /**
     * @param $fieldName
     * @param bool $nullable
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @return self
     */
    public function addLocalizedTextField(string $fieldName,
                                          bool $nullable = false,
                                          $defaultValue = null,
                                          ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }


        $comments->getDocComment()->getTags()
            ->addOrUpdateTag(ModelDataTags::LOCALIZED_PROPERTY);

        $this->addTextField($fieldName, $nullable, $defaultValue, $comments);
        return $this;
    }

    /**
     * @param $fieldName
     * @param int $size
     * @param bool $nullable
     * @param bool $unique
     * @param null | mixed $defaultValue
     * @param null | TableFieldComment $comments
     * @return self
     */
    public function addLocalizedStringField(string $fieldName,
                                            int $size = null,
                                            bool $nullable = false,
                                            bool $unique = false,
                                            $defaultValue = null,
                                            ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }


        $comments->getDocComment()->getTags()
            ->addOrUpdateTag(ModelDataTags::LOCALIZED_PROPERTY);

        $this->addStringField($fieldName, $size, $nullable, $unique, $defaultValue, $comments);
        return $this;
    }



    #endregion specific fields

    #region entity interfaces

    /**
     * @param TableFieldComment|null $comments
     * @return MigrationTrait
     */
    public function addCreatedAtUtc( ?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }


        $this->addDateTimeField(
            'createdAtUtc',
            false,
            null,
            $comments
        );
        $this->addKey('createdAtUtc');
        $this->addEntityInterface(ICreatedAtUtc::class);
        return $this;
    }

    /**
     * @return MigrationTrait
     */
    public function removeCreatedAtUtc():self
    {
        $this->removeField('createdAtUtc');
        $this->removeKey('createdAtUtc');
        $this->removeEntityInterface(ICreatedAtUtc::class);
        return $this;
    }

    /**
     * @param TableFieldComment|null $comments
     * @return self
     */
    public function addUpdatedAtUtc(?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }

        $this->addDateTimeField(
            'updatedAtUtc',
            true,
            null,
            $comments
        );
        $this->addKey('updatedAtUtc');
        $this->addEntityInterface(IUpdatedAtUtc::class);
        return $this;
    }

    /**
     * @return self
     */
    public function removeUpdatedAtUtc():self
    {
        $this->removeField('updatedAtUtc');
        $this->removeKey('updatedAtUtc');
        $this->removeEntityInterface(IUpdatedAtUtc::class);
        return $this;
    }

    /**
     * @param TableFieldComment|null $comments
     * @return self
     */
    public function addDisplayOrder(?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }

        $this->addIntField(
            'displayOrder',
            false,
            false,
            false,
            null,
            $comments
        );
        $this->addKey('displayOrder');
        $this->addEntityInterface(IDisplayOrder::class);
        return $this;
    }

    /**
     * @return self
     */
    public function removeDisplayOrder():self
    {
        $this->removeField('displayOrder');
        $this->removeKey('displayOrder');
        $this->removeEntityInterface(IDisplayOrder::class);
        return $this;
    }

    /**
     * @param TableFieldComment|null $comments
     * @return self
     */
    public function addSoftDeletableField(?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }



        $this->addDateTimeField(
            'deletedAtUtc',
            true,
            null,
            $comments
        );
        $this->addKey('deletedAtUtc');
        $this->addEntityInterface(ISoftDeletable::class);
        return $this;
    }


    /**
     * @return self
     */
    public function removeSoftDeletableField():self
    {
        $this->removeField('deletedAtUtc');
        $this->removeKey('deletedAtUtc');
        $this->removeEntityInterface(ISoftDeletable::class);
        return $this;
    }

    /**
     * @param TableFieldComment|null $comments
     * @return self
     */
    public function addSystemNameField(?TableFieldComment $comments = null):self
    {
        if($comments === null){
            $comments = new TableFieldComment();
        }


        $this->addStringField(
            'systemName',
            1000,
            true,
            false,
            null,
            $comments
        );
        $this->addKey('systemName');
        $this->addEntityInterface(ISystemName::class);
        return $this;
    }

    /**
     * @return self
     */
    public function removeSystemNameField():self
    {
        $this->removeField('systemName');
        $this->removeKey('systemName');
        $this->removeEntityInterface(ISystemName::class);
        return $this;
    }

    #endregion entity interfaces

    /**
     * @param string $fieldName
     * @param TableFieldComment|null $commenta
     * @return self
     */
    public function addPrimaryKeyField($fieldName = 'id', ?TableFieldComment $commenta = null):self
    {
        $this->addIntField($fieldName,
            false,
            true,
            false,
            null,
            null,
            true);
        $this->addKey($fieldName, true);
        return $this;
    }

    /**
     * @param string $interfaceName
     * @return self
     */
    public function addEntityInterface(string $interfaceName):self
    {
        if ($this->tableInterfaces == null) {
            $this->tableInterfaces = new StringList();
        }
        $safeName = $interfaceName; // str_replace("\\", "\\\\", $interfaceName);

        if (!$this->tableInterfaces->contains($safeName)) {
            $this->tableInterfaces->add($safeName);
        }
        return $this;
    }

    /**
     * @param string $interfaceName
     * @return self
     */
    public function removeEntityInterface(string $interfaceName):self
    {
        if ($this->removedTableInterfaces == null) {
            $this->removedTableInterfaces = new StringList();
        }
        $safeName = str_replace("\\", "\\\\", $interfaceName);

        if (!$this->removedTableInterfaces->contains($safeName)) {
            $this->removedTableInterfaces->add($safeName);
        }
        return $this;
    }


    /**
     * @param string $field
     * @param bool $primary
     * @return self
     */
    public function addKey(string $field, bool $primary = false):self
    {
        if ($primary) {
            if($this->prKeys === null){
                $this->prKeys = new StringList();
            }
            if(!$this->prKeys->contains($field)){
                $this->prKeys->add($field);
            }
        } else {
            if($this->keys === null){
                $this->keys = new StringList();
            }
            if(!$this->keys->contains($field)){
                $this->keys->add($field);
            }
        }
        return $this;
    }


    /**
     * @param string $field
     * @return self
     */
    public function removeKey(string $field):self
    {
        if($this->removeKeys === null){
            $this->removeKeys = new StringList();
        }
        if(!$this->removeKeys->contains($field)){
            $this->removeKeys->add($field);
        }
        return $this;
    }

    /**
     * @param string $field
     * @return self
     */
    public function removeField(string $field):self
    {
        if (!in_array($field, $this->removedFields)) {
            $this->removedFields[] = $field;
        }
        return $this;
    }

    /**
     * @param string $table
     * @param bool $ifNotExists
     * @param array $attributes
     */
    public function createTable(string $table, bool $ifNotExists = false, array $attributes = array())
    {
        if($this->keys == null){
            $this->keys = new StringList();
        }
        if($this->prKeys == null){
            $this->prKeys = new StringList();
        }
        foreach ($this->fieldsArray as $key => $value) {
            if (!isset($this->fieldsArray[$key]['comment'])) {
                $this->fieldsArray[$key]['comment'] = new TableFieldComment();
            }
            /** @var TableFieldComment $comm */
            $comm = & $this->fieldsArray[$key]['comment'];
            if ($this->keys->contains($key)) {
                $comm->setIndex(true);
                //$this->fieldsArray[$key]['comment']->index = true;
            }
            if ($this->prKeys->contains($key)) {
                //$this->fieldsArray[$key]['comment']->get = true;
                $comm->setPrimaryKey(true);
            }
            $this->fieldsArray[$key]['comment'] = serialize($comm);
        }

        $this->forge->addField($this->fieldsArray);

        foreach ($this->prKeys as $pky) {
            $this->forge->addKey($pky, true);
        }

        foreach ($this->keys as $ky) {
            $this->forge->addKey($ky);
        }

        $tblComments = $this->_get_table_comment();
        // Tablo için 2048 karakter uzunluğunda (MySQL 5.5.3'ten önceki 60 karakter) bir yorum.
        $tableAttrs = $tblComments == null ? array() : array('COMMENT' => "'" . $tblComments . "'");
        $tableAttrs = array_merge($attributes, $tableAttrs);
        $this->forge->createTable($table, $ifNotExists, $tableAttrs);

        $this->removedFields = array();
        $this->fieldsArray = array();
        $this->changedFieldsArray = array();
        $this->removeKeys = null;
        $this->keys = null;
        $this->prKeys = null;
        $this->tableInterfaces = null;
        $this->removedTableInterfaces = null;

    }



    public function changeTable($table)
    {
        if($this->keys == null){
            $this->keys = new StringList();
        }
        if($this->prKeys == null){
            $this->prKeys = new StringList();
        }
        foreach ($this->removedFields as $rmvField) {
            $this->forge->dropColumn($table, $rmvField);
        }

        foreach ($this->fieldsArray as $key => $value) {
            if (!isset($this->fieldsArray[$key]['comment'])) {
                $this->fieldsArray[$key]['comment'] = new TableFieldComment();
            }
            /** @var TableFieldComment $comm */
            $comm = &$this->fieldsArray[$key]['comment'];
            if ($this->keys->any($key)) {
                $comm->setIndex(true);
                //$this->fieldsArray[$key]['comment']->index = true;
            }
            if ($this->prKeys->any($key)) {
                $comm->setPrimaryKey(true);
                //$this->fieldsArray[$key]['comment']->primary_key = true;
            }
            $this->fieldsArray[$key]['comment'] = serialize($comm);
        }

        foreach ($this->changedFieldsArray as $key => $value) {
            if (!isset($this->changedFieldsArray[$key]['comment'])) {
                $this->changedFieldsArray[$key]['comment'] = new TableFieldComment();
            }
            /** @var TableFieldComment $cmt */
            $cmt = & $this->changedFieldsArray[$key]['comment'];
            if ($this->keys->any($key)) {
                $cmt->setIndex(true);
                // $this->changedFieldsArray[$key]['comment']->index = true;
            }
            if ($this->prKeys->any($key)) {
                $cmt->setPrimaryKey(true);
                // $this->changedFieldsArray[$key]['comment']->primary_key = true;
            }
            $this->changedFieldsArray[$key]['comment'] = serialize($cmt);
        }

        if (!empty($this->changedFieldsArray)) {
            $this->forge->modifyColumn($table, $this->changedFieldsArray);
        }

        if (!empty($this->fieldsArray)) {
            $this->forge->addColumn($table, $this->fieldsArray);
        }


        $oldtblComments = $this->getExistingTableCommentObject($table);

        $tblComments = $this->_get_table_comment(false);
        //var_dump($tblComments);
        //var_dump($oldtblComments);
        if (!empty($oldtblComments)) {
            if($oldtblComments->getInterfaces()->any()){
                foreach ($oldtblComments->getInterfaces() as &$infc) {
                    if (strpos($infc, "\\\\") === false) {
                        $infc = str_replace("\\", "\\\\", $infc);
                    }
                }
            }
            if (!$tblComments->isEntity() && $oldtblComments->isEntity()) {
                $tblComments->setEntityName($oldtblComments->getEntityName());
            }
            if (!$tblComments->isBaseClass()  && $oldtblComments->isBaseClass()) {
                $tblComments->setBaseClass($oldtblComments->getBaseClass());
            }

            if($oldtblComments->getInterfaces()->any()){
                $tblComments->getInterfaces()->addRange(...$oldtblComments->getInterfaces());
                if($this->removedTableInterfaces !== null && $this->removedTableInterfaces->any()){
                    $removed =  $this->removedTableInterfaces;
                    $tblComments->interfaces = $tblComments->interfaces
                        ->where(function (string  $itm) use($removed){
                            return !$removed->contains($itm);
                        })->toClass(StringList::class);
                }
            }

        }

        $tbl_comment_text = serialize($tblComments);
        // $tbl_comment_text = $this->db->escape($tbl_comment_text);

        $sql = "ALTER TABLE " . $this->db->escapeIdentifiers($table) . " COMMENT = '" . $tbl_comment_text . "';";
        $this->db->query($sql);

        foreach ($this->removeKeys as $ky) {
            $sql = "ALTER TABLE " . $this->db->escapeIdentifiers($table) . " DROP INDEX (" . $this->db->escapeIdentifiers($ky) . ");";
            $this->db->query($sql);
        }

        foreach ($this->keys as $ky) {
            $sql = "ALTER TABLE " . $this->db->escapeIdentifiers($table) . " ADD INDEX (" . $this->db->escapeIdentifiers($ky) . ");";
            $this->db->query($sql);
        }

        $this->removedFields = array();
        $this->fieldsArray = array();
        $this->changedFieldsArray = array();
        $this->removeKeys = null;
        $this->keys = null;
        $this->prKeys = null;
        $this->tableInterfaces = null;
        $this->removedTableInterfaces = null;


    }

    public function getExistingTableCommentObject(string $table):TableComment
    {
        $dbInfo = new DbInfo();
        $tblComments = $dbInfo->getTableComment($table);
        return !empty($tblComments) ? unserialize($tblComments) : new TableComment();

    }

    public function getExistingFieldCommentObject($table, $field)
    {
        $dbInfo = new DbInfo();
        $fieldComObj = $dbInfo->getFieldComment($table, $field);
        if (empty($fieldComObj)) {
            return $fieldComObj;
        }

        $retObj = unserialize($fieldComObj);
        return $retObj;
    }

    #endregion methods

    #region utils


    /**
     * @param TableFieldInfo $fieldInfo
     */
    private function addField(TableFieldInfo $fieldInfo)
    {
        if(!$fieldInfo->getNullable()){
            $fieldInfo->getComment()->getValidations()->addRequired();
        }
        if ($this->changeMode && $fieldInfo->issetNewName()) {
            $fd = array(
                $fieldInfo->getFieldName() => array(
                    'name' => $fieldInfo->getNewName(),
                    'type' => strval($fieldInfo->getDbType()),
                    'null' => $fieldInfo->getNullable(),
                )
            );
            $fieldInfo->getComment()->setFieldName($fieldInfo->getNewName());
        }else{
            $fd = array(
                $fieldInfo->getFieldName() => array(
                    'type' => strval($fieldInfo->getDbType()),
                    'null' => $fieldInfo->getNullable(),
                )
            );
            $fieldInfo->getComment()->setFieldName($fieldInfo->getFieldName());

        }
        if($fieldInfo->getAutoIncrement())
        {
            $fd[$fieldInfo->getFieldName()]['auto_increment'] = true;

        }
        if($fieldInfo->issetLength()){
            $fd[$fieldInfo->getFieldName()]['constraint'] = $fieldInfo->getLength();
        }
        if($fieldInfo->getUnsigned()){
            $fd[$fieldInfo->getFieldName()]['unsigned'] = $fieldInfo->getUnsigned();
        }

        if($fieldInfo->issetDefault() && $fieldInfo->getDefault() !== null ){
            $defVal = $fieldInfo->getDefault();
            if(is_bool($defVal)){
                $defVal =  $defVal ? 1 : 0;
            }elseif ($defVal instanceof ConstClass){
                $defVal = strval($defVal);
            }elseif ($defVal instanceof EnumType){
                $defVal = $defVal->toInt();
            }elseif ($defVal instanceof FlagEnumType){
                $defVal = $defVal->toInt();
            }
            $fd[$fieldInfo->getFieldName()]['default'] = $defVal;
            $fieldInfo->getComment()->setDefaultValue($defVal);
        }elseif ($fieldInfo->getNullable()){
            $fd[$fieldInfo->getFieldName()]['default'] = null;
            $fieldInfo->getComment()->setDefaultValue(null);
        }

        if($fieldInfo->getUnique())
        {
            $fd[$fieldInfo->getFieldName()]['unique'] = $fieldInfo->getUnique();
            $fieldInfo->getComment()->setUnique($fieldInfo->getUnique());
        }
        $fd[$fieldInfo->getFieldName()]['comment'] = $fieldInfo->getComment();

        if($this->changeMode){
            $this->changedFieldsArray = array_merge($this->changedFieldsArray, $fd);
        }else{
            $this->fieldsArray = array_merge($this->fieldsArray, $fd);
        }

    }

    /**
     * @param bool $encode
     * @return string|null|TableComment
     */
    private function _get_table_comment($encode = true)
    {

        $res = new TableComment();

        if (!empty($this->entityName)) {
            $res->setEntityName($this->entityName);
        }

        if ($this->tableInterfaces != null) {
            $res->setInterfaces($this->tableInterfaces);
        }

        if ($this->entityBaseClass != null) {
            $res->setBaseClass($this->entityBaseClass);
        }
        if ($this->namespace != null) {
            $res->setNamespace($this->namespace);
        }

        return $encode ? serialize($res) : $res;
    }

    private function handleTableInfo(){
        if($this->tableInfo === null){
            $this->tableInfo = new TableInfo();
        }
    }

    #endregion utils
}
