<?php


namespace GekTools\Tools\Html;


use Gek\Collections\Enumerable;
use Gek\Collections\Typed\StringDictionary;

class HtmlElement extends HtmlNode
{
    #region fields

    /**
     * @var string
     */
    protected string $tagName;

    /**
     * @var StringDictionary
     */
    protected StringDictionary $attributes;


    /**
     * @var bool
     */
    protected bool $selfClosing = false;

    #endregion fields

    #region ctor

    public function __construct(string $tagName)
    {
        parent::__construct();
        $this->tagName = $tagName;
        $this->attributes = new StringDictionary();
    }

    #endregion ctor

    #region Properties

    /**
     * @return string
     */
    public function getTagName(): string
    {
        return $this->tagName;
    }

    /**
     * @param string $tagName
     * @return HtmlNode
     */
    public function setTagName(string $tagName): HtmlNode
    {
        $this->tagName = $tagName;
        return $this;
    }

    /**
     * @return StringDictionary
     */
    public function getAttributes(): StringDictionary
    {
        return $this->attributes;
    }

    /**
     * @param StringDictionary $attributes
     * @return HtmlNode
     */
    public function setAttributes(StringDictionary $attributes): HtmlNode
    {
        $this->attributes = $attributes;
        return $this;
    }

    /**
     * @return bool
     */
    public function isSelfClosing(): bool
    {
        return $this->selfClosing;
    }

    /**
     * @param bool $selfClosing
     * @return HtmlNode
     */
    public function setSelfClosing(bool $selfClosing = true): HtmlNode
    {
        $this->selfClosing = $selfClosing;
        return $this;
    }

    #endregion Properties

    #region methods


    /**
     * @param string $key
     * @return string|null
     */
    public function getAttr(string $key): ?string
    {
        $key = strtolower($key);
        if ($this->attributes->containsKey($key)) {
            return $this->attributes->getByKey($key);
        }
        return null;
    }

    /**
     * @param string $key
     * @param string $value
     * @return $this
     */
    public function setAttr(string $key, string $value): self
    {
        $key = strtolower($key);
        if (!$this->attributes->tryAddKeyValue($key, $value)) {
            $this->attributes[$key] = $value;
        }
        return $this;
    }

    /**
     * @param string $key
     * @return bool
     */
    public function removeAttr(string $key): bool
    {
        $key = strtolower($key);
        return $this->attributes->removeByKey($key);
    }

    /**
     * @return string|null
     */
    public function getId(): ?string
    {
        return $this->getAttr('id');
    }

    /**
     * @param string $id
     * @return $this
     */
    public function setId(string $id): self
    {
        $this->setAttr('id', $id);
        return $this;
    }

    /**
     * @param string $class
     * @return bool
     */
    public function hasClass(string $class): bool
    {
        $class = trim($class);
        if (empty($class)) {
            return false;
        }

        $classes = $this->getAttr('class');
        if (empty($classes)) {
            return false;
        }
        return Enumerable::fromArray(explode(' ', trim($classes)))
            ->where(function (string $c) {
                return !empty(trim($c));
            })->any(function (string $c) use ($class) {
                return trim($c) == $class;
            });
    }

    /**
     * @param string $class
     * @return $this
     */
    public function addClass(string $class): self
    {
        $classes = $this->getAttr('class');
        if (empty($classes)) {
            $this->setAttr('class', $class);
        } elseif (!$this->hasClass($class)) {
            $classes .= " " . $class;
            $this->setAttr('class', $classes);
        }
        return $this;
    }


    public function toHtmlString(): string
    {
        $htmlRes = "<" . $this->tagName;

        if ($this->attributes->any()) {
            $htmlRes .= " ";
            foreach ($this->attributes as $k => $v) {
                $htmlRes .= $k;
                if ($v !== null) {
                    if (((strpos($v, '[') !== false) || (strpos($v, '{') !== false)) && static::isJson($v)) {
                        $v = htmlspecialchars($v);
                    } else {
                        $v = str_replace('"', '\"', $v);
                    }
                    $htmlRes .= '=' . '"' . $v . '"';
                }
                $htmlRes .= ' ';
            }
        }
        if ($this->selfClosing) {
            $htmlRes .= '/>';
            return $htmlRes;
        }
        $htmlRes .= '>';
        if ($this->childNodes->any()) {
            foreach ($this->childNodes as $chNd) {
                $htmlRes .= $chNd->toHtmlString();
            }
        }
        $htmlRes .= '</' . $this->tagName . '>';
        return $htmlRes;
    }

    #endregion methods


    #region utils

    /**
     * @param string $string
     * @return bool
     */
    protected static function isJson(string $string): bool
    {
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }

    #endregion utils
}
