<?php


namespace GekTools\Controllers;


use CodeIgniter\Controller;
use Config\App;
use Config\Services;
use GekTools\Membership\SystemRoleNames;
use GekTools\Tools\BaseModelData;
use GekTools\Tools\View\NotificationType;
use GekTools\Tools\View\PageData;
use GekTools\WorkContext;

class BaseController extends Controller
{
    #region fields

    /**
     * @var WorkContext
     */
    protected WorkContext $workContext;

    /**
     * @var string
     */
    protected string $loginPath = 'login';

    /**
     * @var PageData
     */
    protected PageData $pageData;

    protected array $sessionNotifications = array();

    /**
     * An array of helpers to be loaded automatically upon
     * class instantiation. These helpers will be available
     * to all other controllers that extend BaseController.
     *
     * @var array
     */
    protected $helpers = [];

    #endregion fields

    #region Methods

    /**
     * Constructor.
     */
    public function initController(\CodeIgniter\HTTP\RequestInterface $request, \CodeIgniter\HTTP\ResponseInterface $response, \Psr\Log\LoggerInterface $logger)
    {
        // Do Not Edit This Line
        parent::initController($request, $response, $logger);

        //--------------------------------------------------------------------
        // Preload any models, libraries, etc, here.
        //--------------------------------------------------------------------
        // E.g.:
        // $this->session = \Config\Services::session();

        $this->workContext = WorkContext::instance();
        $appCnf = new App();
        if (isset($appCnf->loginPath)) {
            $this->loginPath = $appCnf->loginPath;
        }
        $this->pageData = new PageData();
    }


    /**
     * @throws \Exception
     */
    protected function loginRequire()
    {
        $user = $this->workContext->getCurrentUser();
        if (!empty($user) && $user->isRegistered()) {
            return;
        }

        $this->response->redirect($this->loginPath)->send();
        exit(EXIT_SUCCESS);
    }

    /**
     * @param string $roleSystemName
     * @throws \Exception
     */
    protected function roleRequire(string $roleSystemName)
    {
        $user = $this->workContext->getCurrentUser();
        if (!empty($user) && ($user->isInRole($roleSystemName) || $user->isInRole(SystemRoleNames::SYSTEM_ADMIN))) {
            return;
        }
        $this->show403();
    }

    /**
     * @param string $permissionRecordSystemName
     * @throws \Exception
     */
    protected function permissionRequire(string $permissionRecordSystemName)
    {
        $user = $this->workContext->getCurrentUser();
        if (!empty($user) && ($user->checkPermission($permissionRecordSystemName) || $user->isInRole(SystemRoleNames::SYSTEM_ADMIN))) {
            return;
        }
        $this->show403();
    }


    protected function show403():void
    {
        $this->showError('403 forbidden', "<p>Ulaşmaya çalıştığınız sayfayı görüntüleme yetkiniz yok.<br>" .
            "Bu durumun bir hata olduğunu düşünüyorsanız lütfen sistem yöneticinize başvurun. </p>",
            403);
    }

    /**
     * @param string $title
     * @param string $message
     * @param int $statusCode
     */
    protected function showError(string $title, string $message, int $statusCode = 500)
    {
        $data = [
            'title' => $title,
            'message' => $message,
        ];
        $html = view('errors/html/error_general', $data);
        $this->response->setStatusCode($statusCode)
            ->setBody($html)
            ->send();
        exit(0);
    }

    /**
     * @param string $name
     * @param BaseModelData|null $modelData
     * @param array $options
     * @return string
     */
    public function view(string $name, ?BaseModelData $modelData = null, array $options = []): string
    {
        $data = [
            'pageData' => $this->pageData,
        ];
        if($modelData !== null){
            $data['modelData'] = $modelData;
        }
        if(! isset($options['saveData'])){
            $options['saveData'] = true;
        }
        return view($name, $data, $options);
    }

    /**
     * @param string $message
     * @param string $type
     * @param bool $session
     */
    public function setNotification(string $message, string $type = NotificationType::INFO, bool $session = true):void
    {
        if($session){
            $notification = (object)array(
                'message' => $message,
                'type' => $type,
            );
            $this->sessionNotifications[] = $notification;
            $ss = Services::session();
            $ss->setFlashdata('ntfs', $this->sessionNotifications);
        }else{
            $this->pageData->addNotification($message,$type);
        }
    }

    #endregion Methods


}