<?php


namespace GekTools\Commands;


use CodeIgniter\CLI\BaseCommand;
use CodeIgniter\CLI\CLI;
use CodeIgniter\Database\Migration;
use CodeIgniter\Entity;
use CodeIgniter\I18n\Time;
use Config\Services;
use Gek\Collections\Enumerable;
use Gek\Filesystem\Filesystem;
use Gek\PhpLang\ClassField;
use Gek\PhpLang\CodeFactory;
use Gek\PhpLang\CodeLine;
use Gek\PhpLang\LiteralTypes;
use Gek\PhpLang\PhpClass;
use Gek\Infrastructure\Str;
use GekTools\Tools\BaseEntity;
use GekTools\Tools\Database\DbInfo;
use GekTools\Tools\Database\TableFieldInfo;
use GekTools\Tools\Traits\MigrationTrait;
use function PHPSTORM_META\type;

class UpdateEntities extends BaseCommand
{

    protected $group = 'Ci4Tools';
    protected $name = 'update:entities';
    protected $description = 'Veritabanı tablo bilgilerine göre bütün varlıkları günceller/oluşturur.';

    /**
     * the Command's usage
     *
     * @var string
     */
    protected $usage = 'update:entities [Options]';

    /**
     * the Command's Arguments
     *
     * @var array
     */
    protected $arguments = [
    ];

    /**
     * the Command's Options
     *
     * @var array
     */
    protected $options = [
        '-n' => 'Entity namespace ayarlar.',
    ];

    protected array $notSetterFields = [
        'createdAtUtc',
        'updatedAtUtc',
        'deletedAtUtc'
    ];

    /**
     * Actually execute a command.
     * This has to be over-ridden in any concrete implementation.
     *
     * @param array $params
     */
    public function run(array $params)
    {
        helper('inflector');


        $ns = $params['-n'] ?? CLI::getOption('n');

        $homepath = APPPATH;
        $isSetNs = false;
        if (!empty($ns)) {
            // Get all namespaces
            $namespaces = Services::autoloader()->getNamespace();

            foreach ($namespaces as $namespace => $path) {
                if ($namespace === $ns) {
                    $homepath = realpath(reset($path));
                    break;
                }
            }
            $isSetNs = true;
        } else {
            $ns = 'App';
        }

        try {
            $dbInfo = new DbInfo(null, true);
            $fs = new Filesystem();
            $tableNames = $dbInfo->getAllTableNames();
            $traitCount = 0;
            $entityCount = 0;


            foreach ($tableNames as $tn) {

                $tableComment = $dbInfo->getTableComment($tn->TABLE_NAME);
                if (empty($tableComment) || !$tableComment->isEntity()) {
                    continue;
                }
                $fieldInfos = $dbInfo->getFieldInfos($tn->TABLE_NAME);
                $tNs = $tableComment->getNamespace();
                if (!empty($tNs) && $isSetNs && $tNs != $ns) {
                    continue;
                }
                if (empty($tNs)) {
                    $tNs = $ns;
                }
                $curHomePath = $this->getHomePath($tNs);
                $traitName = $tableComment->getEntityName() . 'Trait';
                $traitNs = $tNs . '\\Entities\\Extensions';
                $traitPath = $curHomePath . '/Entities/Extensions/' . $traitName . '.php';
                if (!$fs->exists($traitPath)) {
                    try {
                        $trait = CodeFactory::traitCreate($traitName)
                            ->setNamespace($traitNs)
                            ->addCommentSummary('Trait ' . $traitName);
                        $fs->dumpFile($traitPath, '<?php' . PHP_EOL . $trait->toIndentedString());
                        CLI::write('Dosya oluşturuldu : ' . CLI::color(str_replace($homepath, $ns, $traitPath), 'green'));
                        $traitCount++;
                    } catch (\Throwable $exp) {
                        throw $exp;
                        CLI::error($exp->getMessage());
                        return;
                    }
                }
                $entityName = $tableComment->getEntityName();
                $entityPath = $curHomePath . '/Entities/' . $entityName . '.php';
                $extends = $tableComment->isBaseClass() ?
                    $tableComment->getBaseClass() :
                    BaseEntity::class;
                $entityNs = $tNs . '\\Entities';
                try {
                    $entity = CodeFactory::classCreate($entityName)
                        ->setExtends($extends)
                        ->addTrait($traitNs . '\\' . $traitName)
                        ->setNamespace($entityNs);
                    if ($tableComment->isInterfaces()) {
                        foreach ($tableComment->getInterfaces() as $interface) {
                            $entity->addImplement($interface);
                        }
                    }
                    $attributes = $this->handleAttributes($entity, $fieldInfos);
                    $defaultValues = $this->handleDefaultValues($entity, $fieldInfos);

                    $datamap = $this->handleDataMap($entity, $fieldInfos);

                    $casts = $this->handleCasts($entity, $fieldInfos);
                    $primaryKeyField = $this->handlePrimaryKeyField($entity, $fieldInfos);

                    $entityFieldInfoName = $entityName . 'FieldsInfo';
                    $entityFieldInfo = CodeFactory::classCreate($entityFieldInfoName);
                    $priKey = false;
                    foreach ($fieldInfos as $fldInf) {
                        $fName = null;
                        $fType = null;
                        $nullable = null;
                        if ($fldInf->issetComment()) {
                            $fName = $fldInf->getComment()->getFieldName();
                            if ($fldInf->getComment()->isPropName()) {
                                $fName = $fldInf->getComment()->getPropName();
                            }
                            if ($fldInf->getComment()->isPhpType()) {
                                $fType = $fldInf->getComment()->getPhpType();
                                if ($fldInf->getComment()->getNullable() && !Str::startsWith($fType, '?')) {
                                    $fType = '?' . $fType;
                                    $nullable = true;
                                } else {
                                    $nullable = false;
                                }
                            }
                        }
                        if ($fType === null) {
                            $fType = $fldInf->getType();
                        }
                        if ($fldInf->getNullable() && !Str::startsWith($fType, '?')) {
                            $fType = '?' . $fType;
                        }
                        if ($fName === null) {
                            $fName = $fldInf->getFieldName();
                        }

                        $isPriKey = $fldInf->getKey() === 'PRI' && !$priKey;
                        if($isPriKey && !$priKey){
                            $priKey = true;
                        }
                        $isSetter = $isPriKey === false  &&
                            !in_array($fldInf->getFieldName(), $this->notSetterFields);

                        $getterType = $fType;
                        if(! Str::startsWith($getterType,'?')){
                            $getterType = '?' . $getterType;
                        }
                        $getter = $entity->addMethod('get' . ucfirst($fName))
                            ->setReturnType($getterType)
                            ->setRegion('properties')
                            ->addCodeLine('$_attrKey = $this->mapProperty(\'' . $fName . '\');')
                            ->addCodeLine('$result = $this->attributes[$_attrKey];')
                            ->addCodeLine('$result = $this->castAs($result, \'' . $fType . '\');')
                            ->addCodeLine('return $result;')
                            ->autoDocComment()
                            ->addCommentTag('throws', '\Exception')
                            ->addCommentSummary($fName . ' getter');

                        if ($isSetter) {
                            $setter = $entity->addMethod('set' . ucfirst($fName))
                                ->setReturnType('self')
                                ->addParam($fName, $fType)
                                ->setRegion('properties')
                                ->addCodeLine('$_attrKey = $this->mapProperty(\'' . $fName . '\');')
                                ->addCodeLine('$this->attributes[$_attrKey] = $' . $fName . ';')
                                ->addCodeLine('return $this;')
                                ->autoDocComment()
                                ->addCommentSummary($fName . ' setter');
                        }
                        $entityFieldInfo->addField($fldInf->getFieldName(), 'string')
                            ->setValue($fldInf->getFieldName(), LiteralTypes::SINGLE_QUOTES)
                            ->setPublic()
                            ->autoDocComment();
                    }

                    $entity->addMethod('getFieldsInfo')
                        ->setReturnType($entityFieldInfoName)
                        ->setStatic()
                        ->setPublic()
                        ->addCodeLine('static $fInfo = null;')
                        ->addCodeLine('if($fInfo == null){')
                        ->addCodeLine('$fInfo = new ' . $entityFieldInfoName . '();', 1)
                        ->addCodeLine('}')
                        ->addCodeLine('return $fInfo;')
                        ->autoDocComment()
                        ->addCommentSummary('get db Fields info')
                        ->setRegion('statics');

                    $fs->dumpFile($entityPath, '<?php' . PHP_EOL . $entity->toIndentedString() . PHP_EOL . PHP_EOL . $entityFieldInfo->toIndentedString());
                    CLI::write('Dosya oluşturuldu : ' . CLI::color(str_replace($curHomePath, $tNs, $entityPath), 'green'));
                    $entityCount++;
                } catch (\Throwable $exp) {
                    throw $exp;
                    CLI::error($exp->getMessage());
                    return;
                }
            }

            CLI::write(Str::format('Toplam trait : {0}, toplam entity : {1}.', $traitCount, $entityCount), 'green');
        } catch (\Throwable $exp) {
            throw $exp;
            CLI::error($exp->getMessage());
            return;
        }


    }


    /**
     * @param PhpClass $entity
     * @param array|TableFieldInfo[] $fieldInfos
     * @return ClassField
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    protected function handleDefaultValues(PhpClass &$entity, array $fieldInfos): ClassField
    {
        $attrVal = array();
        foreach ($fieldInfos as $fldInf) {
            $fName = $fldInf->getFieldName();
            if ($fldInf->issetComment() && $fldInf->getComment()->getDefaultValue() !== null) {
                $defVal = $fldInf->getComment()->getDefaultValue();
            } elseif ($fldInf->issetDefault()) {
                if ($fldInf->getDefault() === null && $fldInf->getNullable())
                    $defVal = 'null';

            }
            if (isset($defVal)) {
                if ($defVal !== 'null' && $fldInf->getType() === 'string') {
                    $defVal = "'" . $defVal . "'";
                }
            } else {
                if ($fldInf->issetComment() && $fldInf->getComment()->isPhpType()) {
                    $type = $fldInf->getComment()->getPhpType();
                    $nullable = $fldInf->getComment()->getNullable();
                } else {
                    $type = $fldInf->getType();
                    $nullable = $fldInf->getNullable();
                }
                $defVal = $this->getDefaultValue($type, $nullable);
            }
            $attrVal[$fName] = $defVal;
            unset($defVal);
        }
        $strVal = '[';
        if (!empty($attrVal)) {
            $strVal .= PHP_EOL;
        }
        foreach ($attrVal as $k => $v) {
            $strVal .= (new CodeLine(Str::format("'{0}' => {1},", $k, $v), 2)) . PHP_EOL;
        }
        $strVal .= empty($attrVal) ? ']' : (new CodeLine(']', 1));
        $defaultValues = $entity->addField('defaultValues')
            ->setProtected()
            ->setValue($strVal, LiteralTypes::NONE)
            ->setRegion('fields');
        return $defaultValues;
    }

    /**
     * @param PhpClass $entity
     * @param array|TableFieldInfo[] $fieldInfos
     * @return ClassField
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    protected function handleAttributes(PhpClass &$entity, array $fieldInfos): ClassField
    {
        $attrVal = array();
        foreach ($fieldInfos as $fldInf) {
            $fName = $fldInf->getFieldName();
            if ($fldInf->issetComment() && $fldInf->getComment()->getDefaultValue() !== null) {
                $defVal = $fldInf->getComment()->getDefaultValue();
            } elseif ($fldInf->issetDefault()) {
                $defVal = $fldInf->getDefault();
            }
            if (isset($defVal)) {
                if ($defVal === null && $fldInf->getNullable()) {
                    $defVal = 'null';
                } elseif ($fldInf->getType() === 'string') {
                    $defVal = "'" . $defVal . "'";
                }
            } else {
                $defVal = 'null';
            }

            $attrVal[$fName] = $defVal;
            unset($defVal);
        }
        $strVal = '[';
        if (!empty($attrVal)) {
            $strVal .= PHP_EOL;
        }
        foreach ($attrVal as $k => $v) {
            $strVal .= (new CodeLine(Str::format("'{0}' => {1},", $k, $v), 2)) . PHP_EOL;
        }
        $strVal .= empty($attrVal) ? ']' : (new CodeLine(']', 1));
        $attributes = $entity->addField('attributes')
            ->setProtected()
            ->setValue($strVal, LiteralTypes::NONE)
            ->setRegion('fields');
        return $attributes;
    }


    /**
     * @param PhpClass $entity
     * @param array|TableFieldInfo[] $fieldInfos
     * @return ClassField
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    protected function handleDataMap(PhpClass &$entity, array $fieldInfos): ClassField
    {
        $datamapVal = array();
        foreach ($fieldInfos as $fldInf) {
            if ($fldInf->issetComment()) {
                $fName = $fldInf->getComment()->getFieldName();
                if ($fldInf->getComment()->isPropName()) {
                    $fName = $fldInf->getComment()->getPropName();
                }
                if ($fName !== null && $fName != $fldInf->getFieldName()) {
                    $datamapVal[$fName] = $fldInf->getFieldName();
                }
            }
        }
        $strVal = '[';
        if (!empty($datamapVal)) {
            $strVal .= PHP_EOL;
        }
        foreach ($datamapVal as $k => $v) {
            $strVal .= (new CodeLine(Str::format("'{0}' => '{1}',", $k, $v,), 2)) . PHP_EOL;
        }
        $strVal .= empty($datamapVal) ? ']' : (new CodeLine(']', 1));
        $datamap = $entity->addField('datamap')
            ->setProtected()
            ->setValue($strVal, LiteralTypes::NONE)
            ->setRegion('fields');
        return $datamap;
    }

    /**
     * @param PhpClass $entity
     * @param array|TableFieldInfo[] $fieldInfos
     * @return ClassField
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    protected function handleCasts(PhpClass &$entity, array $fieldInfos): ClassField
    {
        $castsVal = array();
        foreach ($fieldInfos as $fldInf) {
            $fName = $fldInf->getFieldName();
            $fType = null;
            if ($fldInf->issetComment() && $fldInf->getComment()->isPhpType()) {
                $fType = $fldInf->getComment()->getPhpType();
                if ($fldInf->getComment()->getNullable() && !Str::startsWith($fType, '?')) {
                    $fType = '?' . $fType;
                }
            } else {
                $fType = $fldInf->getType();
            }
            if ($fldInf->getNullable() && !Str::startsWith($fType, '?')) {
                $fType = '?' . $fType;
            }
            $castsVal[$fName] = $fType;
        }
        $strVal = '[';
        if (!empty($castsVal)) {
            $strVal .= PHP_EOL;
        }
        foreach ($castsVal as $k => $v) {
            $strVal .= (new CodeLine(Str::format("'{0}' => '{1}',", $k, $v), 2)) . PHP_EOL;
        }
        $strVal .= empty($castsVal) ? ']' : (new CodeLine(']', 1));
        $casts = $entity->addField('casts')
            ->setProtected()
            ->setValue($strVal, LiteralTypes::NONE)
            ->setRegion('fields');
        return $casts;
    }

    /**
     * @param PhpClass $entity
     * @param array|TableFieldInfo[] $fieldInfos
     * @return ClassField
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    protected function handlePrimaryKeyField(PhpClass &$entity, array $fieldInfos): ?ClassField
    {
        $pkField = null;
        foreach ($fieldInfos as $fldInf) {
            if ($fldInf->isKey() && $fldInf->getKey() == 'PRI') {
                $pkField = $fldInf->getFieldName();
                break;
            }
        }
        if ($pkField && $pkField != 'id') {
            $primaryKeyField = $entity->addField('primaryKeyField', 'string')
                ->setProtected()
                ->setValue($pkField, LiteralTypes::SINGLE_QUOTES)
                ->setRegion('fields');
            return $primaryKeyField;
        }
        return null;
    }

    /**
     * @param string|null $ns
     * @return string
     */
    protected function getHomePath(?string &$ns): string
    {
        if (empty($ns)) {
            $ns = 'App';
            return APPPATH;
        }
        // Get all namespaces
        $namespaces = Services::autoloader()->getNamespace();

        foreach ($namespaces as $namespace => $path) {
            if ($namespace === $ns) {
                return realpath(reset($path));
            }
        }
        $ns = 'App';
        return APPPATH;
    }

    protected function getDefaultValue(string $type, bool $nullable): string
    {
        if (!$nullable && Str::startsWith($type, '?')) {
            $nullable = true;
        }
        if ($nullable) {
            return 'null';
        }

        switch ($type) {
            case 'int':
                return '0';
                break;
            case 'float':
                return '0.0';
                break;
            case 'bool':
                return 'false';
                break;
            case 'string':
                return "''";
                break;
            case \DateTime::class:
            case Time::class:
                return "'0000-00-00 00:00:00'";
                break;
            default:
                return '0';
                break;
        }
    }

}
