<?php


namespace GekTools\Tools\View;


use GekTools\Tools\Xml\XmlNode;
use GekTools\Tools\Xml\ItoXml;

class MenuItem implements ItoXml
{

    #region fields

    /**
     * @var string|null
     */
    protected ?string $text = null;

    /**
     * @var string|null
     */
    protected ?string $url = null;

    /**
     * @var string|null
     */
    protected ?string $systemName = null;

    /**
     * @var string|null
     */
    protected ?string $iconClass = null;

    /**
     * @var bool
     */
    protected bool $loginRequired = false;

    /**
     * @var array|string[]
     */
    protected array $requiredPermissionNames = array();

    /**
     * @var array|string[]
     */
    protected array $requiredRoles = array();

    /**
     * @var array
     */
    protected array $attributes = array();


    /**
     * @var array|MenuItem[]
     */
    protected array $childs = array();

    /**
     * @var bool|null
     */
    protected ?bool $active = null;

    #endregion fields

    #region ctor

    public function __construct()
    {
    }

    #endregion ctor

    #region properties

    /**
     * @return string|null
     */
    public function getText(): ?string
    {
        return $this->text;
    }

    /**
     * @param string|null $text
     * @return MenuItem
     */
    public function setText(?string $text): MenuItem
    {
        $this->text = $text;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getUrl(): ?string
    {
        return $this->url;
    }

    /**
     * @param string|null $url
     * @return MenuItem
     */
    public function setUrl(?string $url): MenuItem
    {
        $this->url = $url;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getSystemName(): ?string
    {
        return $this->systemName;
    }

    /**
     * @param string|null $systemName
     * @return MenuItem
     */
    public function setSystemName(?string $systemName): MenuItem
    {
        $this->systemName = $systemName;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getIconClass(): ?string
    {
        return $this->iconClass;
    }

    /**
     * @param string|null $iconClass
     * @return MenuItem
     */
    public function setIconClass(?string $iconClass): MenuItem
    {
        $this->iconClass = $iconClass;
        return $this;
    }

    /**
     * @return bool
     */
    public function isLoginRequired(): bool
    {
        return $this->loginRequired;
    }

    /**
     * @param bool $loginRequired
     * @return MenuItem
     */
    public function setLoginRequired(bool $loginRequired = true): MenuItem
    {
        $this->loginRequired = $loginRequired;
        return $this;
    }

    /**
     * @return array|string[]
     */
    public function getRequiredPermissionNames()
    {
        return $this->requiredPermissionNames;
    }

    /**
     * @param array|string[] $requiredPermissionNames
     * @return MenuItem
     */
    public function setRequiredPermissionNames($requiredPermissionNames)
    {
        $this->requiredPermissionNames = $requiredPermissionNames;
        return $this;
    }

    /**
     * @return array|string[]
     */
    public function getRequiredRoles()
    {
        return $this->requiredRoles;
    }

    /**
     * @param array|string[] $requiredRoles
     * @return MenuItem
     */
    public function setRequiredRoles($requiredRoles)
    {
        $this->requiredRoles = $requiredRoles;
        return $this;
    }

    /**
     * @return array
     */
    public function getAttributes(): array
    {
        return $this->attributes;
    }

    /**
     * @param array $attributes
     * @return MenuItem
     */
    public function setAttributes(array $attributes): MenuItem
    {
        $this->attributes = $attributes;
        return $this;
    }

    /**
     * @return array|MenuItem[]
     */
    public function getChilds()
    {
        return $this->childs;
    }

    /**
     * @param array|MenuItem[] $childs
     * @return MenuItem
     */
    public function setChilds($childs)
    {
        $this->childs = $childs;
        return $this;
    }

    #endregion properties

    #region Methods

    /**
     * @param string $roleSystemName
     * @return MenuItem
     */
    public function addRoleRequired(string $roleSystemName): MenuItem
    {
        $this->requiredRoles[] = $roleSystemName;
        return $this;
    }

    /**
     * @param string $permissionSystemName
     * @return MenuItem
     */
    public function addPermissionRequire(string $permissionSystemName): MenuItem
    {
        $this->requiredPermissionNames[] = $permissionSystemName;
        return $this;
    }

    /**
     * @param MenuItem $menuItem
     * @return MenuItem
     */
    public function addChild(MenuItem $menuItem): MenuItem
    {
        $this->childs[] = $menuItem;
        return $this;
    }

    /**
     * @param string $key
     * @param string $value
     * @return MenuItem
     */
    public function addAttribute(string $key, string $value): MenuItem
    {
        $this->attributes[$key] = $value;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasChilds(): bool
    {
        return !empty($this->childs);
    }

    /**
     * @return bool
     */
    public function hasPermissionRequire(): bool
    {
        return !empty($this->requiredPermissionNames);
    }

    /**
     * @return bool
     */
    public function hasRoleRequired(): bool
    {
        return !empty($this->requiredRoles);
    }

    /**
     * @return bool
     */
    public function hasAttributes(): bool
    {
        return !empty($this->attributes);
    }

    /**
     * @param string $systemName
     * @return bool
     */
    public function isActive(string $systemName):bool {
        if($this->active !== null){
            return $this->active;
        }
        if(empty($systemName)){
            $this->active = false;
            return false;
        }
        $this->active = $this->getSystemName() == $systemName;
        if($this->active == false && $this->hasChilds()){
            foreach ($this->getChilds() as $child){
                $res = $child->isActive($systemName);
                if($res){
                    $this->active = $res;
                    break;
                }
            }
        }
        return $this->active;
    }

    #endregion Methods

    #region ItoXml

    /**
     * @return XmlNode
     */
    public function toXmlNode(): XmlNode
    {
        $node = new XmlNode('item');
        $node->addAttribute('text', $this->getText());
        if($this->getUrl()){
            $node->addAttribute('url',$this->getUrl());
        }

        if($this->getSystemName()){
            $node->addAttribute('systemName',$this->getSystemName());
        }

        if($this->getIconClass()){
            $node->addAttribute('iconClass',$this->getIconClass());
        }
        if($this->isLoginRequired()){
            $node->addAttribute('loginRequired','1');
        }
        if($this->hasPermissionRequire()){
            $node->addAttribute('requiredPermissions',implode(',',$this->requiredPermissionNames));
        }
        if($this->hasRoleRequired()){
            $node->addAttribute('requiredRoles',implode(',',$this->requiredRoles));
        }

        if($this->hasAttributes()){
            foreach ($this->attributes as $key => $val){
                $node->addAttribute($key,$val);
            }
        }
        if($this->hasChilds()){
            foreach ($this->childs as $mi){
                $node->addChild($mi->toXmlNode());
            }
        }

        return $node;

    }

    /**
     * @return string
     */
    public function toXml(): string
    {
        return $this->toXmlNode()->toXmlString(0);
    }

    #endregion ItoXml
}