<?php
namespace GekTools\Membership\Entities\Extensions;

use GekTools\GenericAttributes\Traits\GenericAttributesTrait;
use GekTools\Membership\Entities\User;
use GekTools\Membership\Entities\UserRole;
use GekTools\Membership\MembershipService;
use GekTools\Membership\SystemRoleNames;
use GekTools\Membership\SystemUserNames;
use GekTools\Membership\UserService;
use GekTools\Tools\Collections\PagedResult;

/**
 * Trait UserTrait
 * @mixin User
 */ 
trait UserTrait 
{
    use GenericAttributesTrait;

    #region fields

    /**
     * @var PagedResult|UserRole[]|null
     */
    protected ?PagedResult $userRoles = null;

    #endregion fields

    #region methods

    /**
     * @return bool
     * @throws \Exception
     */
    public function isSearchEngineAccount():bool {
        return $this->getSystemName() == SystemUserNames::SEARCH_ENGINE;
    }

    /**
     * @return bool
     * @throws \Exception
     */
    public function isBackgroundTaskAccount():bool{
        return $this->getSystemName() == SystemUserNames::BACKGROUND_TASK;
    }

    /**
     * @param string $roleSystemName
     * @return bool
     * @throws \Exception
     */
    public function isInRole(string $roleSystemName):bool {
        return $this->getUserRoles()->any(function (UserRole $role) use($roleSystemName){
            return $role->getSystemName() == $roleSystemName;
        });
    }

    /**
     * @return bool
     * @throws \Exception
     */
    public function isRegistered():bool {
        return $this->getUserRoles()->any(function (UserRole $role){
            return in_array(
                $role->getSystemName(),
                [
                    SystemRoleNames::REGISTERED,
                    SystemRoleNames::ADMINISTARATORS,
                    SystemRoleNames::SYSTEM_ADMIN,
                ]
            );
        });
    }

    /**
     * @return UserRole[]|PagedResult
     * @throws \Exception
     */
    public function getUserRoles(){
        if($this->userRoles === null){
            $us =  UserService::instance();
            $this->userRoles = $us->getUserRolesByUserId($this->getId());
        }
        return $this->userRoles;

    }

    /**
     * @param UserRole $role
     * @return bool
     * @throws \Exception
     */
    public function removeRole(UserRole $role):bool {
        if($this->getId()){
           $res = UserService::instance()->removeRoleOnUser($this->getId(),$role->getId());
            $this->userRoles = null;
            if($res){
                return true;
            }
        }
        return false;
    }

    /**
     * @param UserRole $role
     * @return bool
     * @throws \Exception
     */
    public function addRole(UserRole $role):bool {
        if($this->getId()){
            $res = UserService::instance()->addRoleOnUser($this->getId(),$role->getId());
            $this->userRoles = null;
            if($res){
                return true;
            }
        }
        return false;
    }

    /**
     * @param string $permissionSystemName
     * @return bool
     * @throws \Exception
     */
    public function checkPermission(string $permissionSystemName):bool{
        return $this->getUserRoles()->any(function (UserRole $role) use($permissionSystemName){
            return $role->checkPermission($permissionSystemName);
        });
    }

    public function refresh()
    {
        $this->genericAttributes = array();
        $this->userRoles = null;
        if (!empty($this->getId())) {
            $usr = UserService::instance()->getUserByUserGuid($this->getUserGuid());
            if (!empty($usr)) {
                $this->fill($usr->toRawArray());
            }
        }
    }

    public function toRawArray(bool $onlyChanged = false): array
    {
        $res = parent::toRawArray($onlyChanged);
        if($onlyChanged){
            $fi = User::getFieldsInfo();
            if(! isset($res[$fi->userGuid])){
                $res[$fi->userGuid] = $this->attributes[$fi->userGuid];
            }
        }
        return $res;
    }


    #endregion methods

}
