<?php


namespace GekTools\Commands;


use CodeIgniter\CLI\BaseCommand;
use CodeIgniter\CLI\CLI;
use CodeIgniter\Database\ConnectionInterface;
use CodeIgniter\Model;
use CodeIgniter\Validation\ValidationInterface;
use Config\Services;
use Gek\Filesystem\Filesystem;
use Gek\PhpLang\ClassField;
use Gek\PhpLang\CodeFactory;
use Gek\PhpLang\CodeLine;
use Gek\PhpLang\LiteralTypes;
use Gek\PhpLang\MethodParam;
use Gek\PhpLang\PhpClass;
use Gek\Infrastructure\Str;
use GekTools\Tools\BaseEntity;
use GekTools\Tools\BaseModel;
use GekTools\Tools\Contrats\ICreatedAtUtc;
use GekTools\Tools\Contrats\ISoftDeletable;
use GekTools\Tools\Database\DbInfo;
use GekTools\Tools\Database\TableComment;
use GekTools\Tools\Database\TableFieldInfo;
use GekTools\Tools\Validations\Validation;
use GekTools\Tools\Validations\ValidationCollection;

class CreateModel extends BaseCommand
{

    protected $group = 'Ci4Tools';
    protected $name = 'create:model';
    protected $description = 'Yeni bir model oluşturur.';

    /**
     * the Command's usage
     *
     * @var string
     */
    protected $usage = 'create:model [table_name] [Options]';

    /**
     * the Command's Arguments
     *
     * @var array
     */
    protected $arguments = [
        'table_name' => 'Veritabanı tablo adı'
    ];

    /**
     * the Command's Options
     *
     * @var array
     */
    protected $options = [
        '-m' => 'model adı.',
        '-n' => 'namespace ayarlar.',
    ];

    protected array $notSetterFields = [
        'createdAtUtc',
        'updatedAtUtc',
        'deletedAtUtc'
    ];

    /**
     * Actually execute a command.
     * This has to be over-ridden in any concrete implementation.
     *
     * @param array $params
     */
    public function run(array $params)
    {
        helper('inflector');

        $tableName = array_shift($params);

        if (empty($tableName)) {
            $tableName = CLI::prompt('Tablo adını girin');
        }

        $ns = $params['-n'] ?? CLI::getOption('n');
        $modelName = $params['-m'] ?? CLI::getOption('m');

        if (empty($modelName)) {
            $modelName = '';
            if (Str::contains($tableName, '_')) {
                $words = explode('_', $tableName);
                foreach ($words as $word) {
                    if (!empty($word)) {
                        $modelName .= ucfirst($word);
                    }
                }
            } else {
                $modelName = ucfirst($tableName);
            }

        }
        if (!Str::endsWith($modelName, 'Model')) {
            $modelName .= 'Model';
        }

        $dbInfo = new DbInfo(null, true);

        $tableComment = $dbInfo->getTableComment($tableName);
        if (empty($ns) && !empty($tableComment) && $tableComment->isNamespace()) {
            $ns = $tableComment->getNamespace();
        }


        $homepath = APPPATH;

        if (!empty($ns)) {
            // Get all namespaces
            $namespaces = Services::autoloader()->getNamespace();

            foreach ($namespaces as $namespace => $path) {
                if ($namespace === $ns) {
                    $homepath = realpath(reset($path));
                    break;
                }
            }
        } else {
            $ns = 'App';
        }

        $modelsNs = $ns . '\\Models';
        $modelPath = $homepath . "/Models/" . $modelName . '.php';
        $fs = new Filesystem();

        if ($fs->exists($modelPath)) {
            CLI::error('Dosya zaten mevcut : ' . str_replace($homepath, $ns, $modelPath));
            return;
        }

        $model = CodeFactory::classCreate($modelName)
            ->setNamespace($modelsNs)
            ->setExtends(BaseModel::class);
        $model->addField('table', null)
            ->setValue($tableName, LiteralTypes::SINGLE_QUOTES)
            ->setProtected()
            ->setRegion('fields');


        $fieldInfos = $dbInfo->getFieldInfos($tableName);

        $returnType = $this->handleReturnType($model, $tableComment, $ns);
        $primaryKey = $this->handlePrimaryKey($model, $fieldInfos);
        $useSoftDeletes = $this->handleUseSoftDeletes($model, $tableComment);
        $useTimestamps = $this->handleUseTimestamps($model, $tableComment);
        $allowedFields = $this->handleAllowedFields($model, $fieldInfos);
        $validationRules = $this->handleValidationRules($model, $fieldInfos);

        $model->addConstructor(true)
            ->addParam((new MethodParam('db', ConnectionInterface::class, false, true))->setValue('null'))
            ->addParam((new MethodParam('validation', ValidationInterface::class))->setValue('null'))
            ->addCodeLine('parent::__construct($db, $validation);')
            ->autoDocComment();


        $fs->dumpFile($modelPath, '<?php' . PHP_EOL . $model->toIndentedString());
        CLI::write('Dosya oluşturuldu : ' . CLI::color(str_replace($homepath, $ns, $modelPath), 'green'));
    }

    /**
     * @param PhpClass $model
     * @param TableComment|null $tableComment
     * @param string $ns
     * @return ClassField
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    protected function handleReturnType(PhpClass &$model, ?TableComment $tableComment, string $ns): ClassField
    {
        $retType = $model->addField('returnType')
            ->setProtected()
            ->setRegion('fields');
        if (empty($tableComment) || !$tableComment->isEntity()) {
            $retType->setValue('object', LiteralTypes::SINGLE_QUOTES);
        } else {
            $entity = $tableComment->getEntityName();
            $fullName = $ns . '\\Entities\\' . $entity;
            if (!class_exists($fullName)) {
                CLI::error('Entity class mevcut değil : ' . $fullName);
            }
            $model->addUse($fullName);
            $retType->setValue($entity . '::class', LiteralTypes::NONE);
        }
        return $retType;
    }

    /**
     * @param PhpClass $model
     * @param TableComment|null $tableComment
     * @return ClassField
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    protected function handleUseSoftDeletes(PhpClass &$model, ?TableComment $tableComment): ClassField
    {
        $ret = $model->addField('useSoftDeletes')
            ->setProtected()
            ->setRegion('fields');
        if (empty($tableComment) || !$tableComment->isInterfaces()) {
            $ret->setValue('false', LiteralTypes::NONE);
        } else {
            $val = 'false';
            if ($tableComment->getInterfaces()->contains(ISoftDeletable::class)) {
                $val = 'true';
            }
            $ret->setValue($val, LiteralTypes::NONE);
        }
        return $ret;
    }

    /**
     * @param PhpClass $model
     * @param TableComment|null $tableComment
     * @return ClassField
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    protected function handleUseTimestamps(PhpClass &$model, ?TableComment $tableComment): ClassField
    {
        $ret = $model->addField('useTimestamps')
            ->setProtected()
            ->setRegion('fields');
        if (empty($tableComment) || !$tableComment->isInterfaces()) {
            $ret->setValue('false', LiteralTypes::NONE);
        } else {
            $val = 'false';
            if ($tableComment->getInterfaces()->contains(ICreatedAtUtc::class)) {
                $val = 'true';
            }
            $ret->setValue($val, LiteralTypes::NONE);
        }
        return $ret;
    }


    /**
     * @param PhpClass $model
     * @param array|TableFieldInfo[] $fieldInfos
     * @return ClassField
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    protected function handleAllowedFields(PhpClass &$model, array $fieldInfos): ClassField
    {
        $ret = $model->addField('allowedFields')
            ->setProtected()
            ->setRegion('fields');
        if (empty($fieldInfos)) {
            $ret->setValue('[]', LiteralTypes::NONE);
        } else {
            $fields = array();
            $setPri = false;
            foreach ($fieldInfos as $fldInf) {
                if ($fldInf->getKey() == 'PRI' && ! $setPri) {
                    $setPri = true;
                    continue;
                }
                $fName = $fldInf->getFieldName();
                $fields[] = new CodeLine("'" . $fName . "',", 2);
            }
            $strVal = '[';
            if (!empty($fields)) {
                $strVal .= PHP_EOL;
            }
            foreach ($fields as $v) {
                $strVal .= $v . PHP_EOL;
            }
            $strVal .= Str::contains($strVal, PHP_EOL) ? (new CodeLine(']', 1)) : ']';
            $ret->setValue($strVal, LiteralTypes::NONE);
        }
        return $ret;
    }

    /**
     * @param PhpClass $model
     * @param array|TableFieldInfo[] $fieldInfos
     * @return ClassField
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    protected function handleValidationRules(PhpClass &$model, array $fieldInfos): ClassField
    {
        $ret = $model->addField('validationRules')
            ->setProtected()
            ->setRegion('fields');
        if (empty($fieldInfos)) {
            $ret->setValue('[]', LiteralTypes::NONE);
        } else {
            $rules = array();
            foreach ($fieldInfos as $fldInf) {
                if (
                    $fldInf->getNullable() ||
                    $fldInf->getKey() === 'PRI' ||
                    in_array($fldInf->getFieldName(), $this->notSetterFields) ||
                    $fldInf->getType() == 'bool'
                ) {
                    continue;
                }
                if (!$fldInf->issetComment() || !$fldInf->getComment()->getValidations()->any(function (Validation $v) {
                        return $v->getUseModel();
                    })) {
                    continue;
                }
                $fName = $fldInf->getFieldName();

                $rules[$fName] = $fldInf->getComment()->getValidations()->where(function (Validation $v) {
                    return $v->getUseModel( );
                })->toTypedClass(ValidationCollection::class)->__toString();
            }
            $strVal = '[';
            if (!empty($rules)) {
                $strVal .= PHP_EOL;
            }
            foreach ($rules as $k => $v) {
                $strVal .= (new CodeLine(Str::format("'{0}' => '{1}',", $k, $v), 2)) . PHP_EOL;
            }
            $strVal .= empty($rules) ? ']' : (new CodeLine(']', 1));
            $ret->setValue($strVal, LiteralTypes::NONE);
        }
        return $ret;
    }

    /**
     * @param PhpClass $model
     * @param array|TableFieldInfo[] $fieldInfos
     * @return ClassField
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    protected function handlePrimaryKey(PhpClass &$model, array $fieldInfos): ClassField
    {
        $ret = $model->addField('primaryKey')
            ->setProtected()
            ->setRegion('fields');
        if (empty($fieldInfos)) {
            $ret->setValue('id', LiteralTypes::SINGLE_QUOTES);
        } else {
            $val = 'id';
            foreach ($fieldInfos as $fldInf) {
                if ($fldInf->getKey() == 'PRI') {
                    $val = $fldInf->getFieldName();
                    break;
                }
            }
            $ret->setValue($val, LiteralTypes::SINGLE_QUOTES);
        }
        return $ret;
    }

}
