<?php


namespace GekTools\Commands;


use CodeIgniter\CLI\BaseCommand;
use CodeIgniter\CLI\CLI;
use CodeIgniter\Database\Migration;
use Config\Services;
use Gek\Collections\Enumerable;
use Gek\Filesystem\Filesystem;
use Gek\Infrastructure\Str;
use Gek\PhpLang\CodeFactory;
use Gek\PhpLang\LiteralTypes;
use GekTools\Tools\Traits\MigrationTrait;

class CreateMigration extends BaseCommand
{

    protected $group = 'Ci4Tools';
    protected $name = 'create:migration';
    protected $description = 'Yeni bir migration dosyası oluşturur.';

    /**
     * the Command's usage
     *
     * @var string
     */
    protected $usage = 'create:migration [migration_name] [Options]';

    /**
     * the Command's Arguments
     *
     * @var array
     */
    protected $arguments = [
        'migration_name' => 'Migrasyon dosya adı',
    ];

    /**
     * the Command's Options
     *
     * @var array
     */
    protected $options = [
        '-n' => 'Namespace ayarlar.',
        '-t' => 'Tablo adını ayarlar.',
        '-p' => 'pk adını ayarlar.',
    ];

    /**
     * Actually execute a command.
     * This has to be over-ridden in any concrete implementation.
     *
     * @param array $params
     */
    public function run(array $params)
    {
        helper('inflector');

        $name = array_shift($params);

        if (empty($name)) {
            $name = CLI::prompt(lang('Migrations.nameMigration'));
        }

        if (empty($name)) {
            CLI::error(lang('Migrations.badCreateName'));
            return;
        }

        $ns = $params['-n'] ?? CLI::getOption('n');
        $tableName = $params['-t'] ?? CLI::getOption('t');
        if (!is_string($tableName)) {
            $tableName = '';
        }
        $pkName = $params['-p'] ?? CLI::getOption('p');
        if (!is_string($pkName)) {
            $pkName = 'id';
        }
        $homepath = APPPATH;

        if (!empty($ns)) {
            // Get all namespaces
            $namespaces = Services::autoloader()->getNamespace();

            foreach ($namespaces as $namespace => $path) {
                if ($namespace === $ns) {
                    $homepath = realpath(reset($path));
                    break;
                }
            }
        } else {
            $ns = 'App';
        }

        // Always use UTC/GMT so global teams can work together
        $config = config('Migrations');
        $fileName = gmdate($config->timestampFormat) . $name;

        // full path
        $path = $homepath . '/Database/Migrations/' . $fileName . '.php';

        // Class name should be pascal case now (camel case with upper first letter)
        $name = pascalize($name);

        $migClass = CodeFactory::classCreate($name)
            ->setNamespace($ns . '\\Database\\Migrations')
            ->setExtends(Migration::class)
            ->addTrait(MigrationTrait::class);
        $migClass->addField('tableName', 'string')
            ->setValue($tableName, LiteralTypes::SINGLE_QUOTES)
            ->setProtected();
        $migClass->addField('pkName', 'string')
            ->setValue($pkName, LiteralTypes::SINGLE_QUOTES)
            ->setProtected();
        $codeNs = str_replace('\\','\\\\', $ns);
        $codeEntityName = str_replace('-','_',$tableName);
        if(Str::endsWith($codeEntityName,'ies')){
            $codeEntityName = substr($codeEntityName,0,-3) .'y';
        }elseif (Str::endsWith($codeEntityName,'s')){
            $codeEntityName = substr($codeEntityName,0,-1);
        }
        if(Str::contains($codeEntityName,'_')){
            $entNameArr = explode('_', $codeEntityName);
            $codeEntityName = Enumerable::fromArray($entNameArr)
                ->aggregate(function (&$res, $itm){
                    $res .= ucfirst($itm);
                },'');
        }else{
            $codeEntityName = ucfirst($codeEntityName);
        }
        $migClass->addMethod('up')
            ->addCodeLine('$this->entityName = \''. $codeEntityName.'\';')
            ->addCodeLine('$this->namespace = \''.$codeNs.'\';')
            ->addCodeLine('$this->addPrimaryKeyField($this->pkName);')
            ->addCodeLine('// add fields')
            ->addCodeLine('$this->createTable($this->tableName);');

        $migClass->addMethod('down')
            ->addCodeLine('$this->forge->dropTable($this->tableName);');

        $fs = new Filesystem();

        try {
            $fs->dumpFile($path, "<?php \n\n" . $migClass->toIndentedString());
        } catch (\Throwable $exp) {
            CLI::error($exp->getMessage());
            return;
        }


        CLI::write('Dosya oluşturuldu : ' . CLI::color(str_replace($homepath, $ns, $path), 'green'));
    }
}
