<?php


namespace GekTools\Tools\Database;

use Closure;
use CodeIgniter\Database\BaseBuilder;
use CodeIgniter\Database\BaseResult;
use CodeIgniter\Database\ConnectionInterface;
use CodeIgniter\Database\Exceptions\DatabaseException;
use CodeIgniter\Database\Query;
use CodeIgniter\Database\ResultInterface;
use GekTools\Tools\BaseModel;

/**
 * Class BaseBuilder
 * @package GekTools\Tools\Database
 * @mixin   \CodeIgniter\Model
 * @mixin   BaseModel
 * @mixin   BaseBuilder
 */
interface IBaseBuilderMixinFix
{
    /**
     * Constructor
     *
     * @param  string|array                              $tableName
     * @param  \CodeIgniter\Database\ConnectionInterface $db
     * @param  array                                     $options
     * @throws DatabaseException
     */
    public function __construct($tableName, ConnectionInterface &$db, array $options = null);

    /**
     * Sets a test mode status.
     *
     * @param boolean $mode Mode to set
     *
     * @return self
     */
    public function testMode(bool $mode = true);

    /**
     * Returns an array of bind values and their
     * named parameters for binding in the Query object later.
     *
     * @return array
     */
    public function getBinds(): array;

    /**
     * Ignore
     *
     * Set ignore Flag for next insert,
     * update or delete query.
     *
     * @param boolean $ignore
     *
     * @return self
     */
    public function ignore(bool $ignore = true);

    /**
     * Select
     *
     * Generates the SELECT portion of the query
     *
     * @param string|array $select
     * @param boolean      $escape
     *
     * @return BaseBuilder
     */
    public function select($select = '*', bool $escape = null);

    /**
     * Select Max
     *
     * Generates a SELECT MAX(field) portion of a query
     *
     * @param string $select The field
     * @param string $alias  An alias
     *
     * @return self
     */
    public function selectMax(string $select = '', string $alias = '');

    /**
     * Select Min
     *
     * Generates a SELECT MIN(field) portion of a query
     *
     * @param string $select The field
     * @param string $alias  An alias
     *
     * @return self
     */
    public function selectMin(string $select = '', string $alias = '');

    /**
     * Select Average
     *
     * Generates a SELECT AVG(field) portion of a query
     *
     * @param string $select The field
     * @param string $alias  An alias
     *
     * @return self
     */
    public function selectAvg(string $select = '', string $alias = '');

    /**
     * Select Sum
     *
     * Generates a SELECT SUM(field) portion of a query
     *
     * @param string $select The field
     * @param string $alias  An alias
     *
     * @return self
     */
    public function selectSum(string $select = '', string $alias = '');

    /**
     * Select Count
     *
     * Generates a SELECT COUNT(field) portion of a query
     *
     * @param string $select The field
     * @param string $alias  An alias
     *
     * @return self
     */
    public function selectCount(string $select = '', string $alias = '');

    /**
     * DISTINCT
     *
     * Sets a flag which tells the query string compiler to add DISTINCT
     *
     * @param boolean $val
     *
     * @return self
     */
    public function distinct(bool $val = true);

    /**
     * From
     *
     * Generates the FROM portion of the query
     *
     * @param mixed   $from      can be a string or array
     * @param boolean $overwrite Should we remove the first table existing?
     *
     * @return self
     */
    public function from($from, bool $overwrite = false);


    /**
     * JOIN
     *
     * Generates the JOIN portion of the query
     *
     * @param string  $table
     * @param string  $cond   The join condition
     * @param string  $type   The type of join
     * @param boolean $escape Whether not to try to escape identifiers
     *
     * @return self
     */
    public function join(string $table, string $cond, string $type = '', bool $escape = null);


    /**
     * WHERE
     *
     * Generates the WHERE portion of the query.
     * Separates multiple calls with 'AND'.
     *
     * @param mixed   $key
     * @param mixed   $value
     * @param boolean $escape
     *
     * @return self
     */
    public function where($key, $value = null, bool $escape = null);


    /**
     * OR WHERE
     *
     * Generates the WHERE portion of the query.
     * Separates multiple calls with 'OR'.
     *
     * @param mixed   $key
     * @param mixed   $value
     * @param boolean $escape
     *
     * @return self
     */
    public function orWhere($key, $value = null, bool $escape = null);


    /**
     * WHERE IN
     *
     * Generates a WHERE field IN('item', 'item') SQL query,
     * joined with 'AND' if appropriate.
     *
     * @param string               $key    The field to search
     * @param array|string|Closure $values The values searched on, or anonymous function with subquery
     * @param boolean              $escape
     *
     * @return self
     */
    public function whereIn(string $key = null, $values = null, bool $escape = null);

    /**
     * OR WHERE IN
     *
     * Generates a WHERE field IN('item', 'item') SQL query,
     * joined with 'OR' if appropriate.
     *
     * @param string               $key    The field to search
     * @param array|string|Closure $values The values searched on, or anonymous function with subquery
     * @param boolean              $escape
     *
     * @return self
     */
    public function orWhereIn(string $key = null, $values = null, bool $escape = null);

    /**
     * WHERE NOT IN
     *
     * Generates a WHERE field NOT IN('item', 'item') SQL query,
     * joined with 'AND' if appropriate.
     *
     * @param string               $key    The field to search
     * @param array|string|Closure $values The values searched on, or anonymous function with subquery
     * @param boolean              $escape
     *
     * @return self
     */
    public function whereNotIn(string $key = null, $values = null, bool $escape = null);

    /**
     * OR WHERE NOT IN
     *
     * Generates a WHERE field NOT IN('item', 'item') SQL query,
     * joined with 'OR' if appropriate.
     *
     * @param string               $key    The field to search
     * @param array|string|Closure $values The values searched on, or anonymous function with subquery
     * @param boolean              $escape
     *
     * @return self
     */
    public function orWhereNotIn(string $key = null, $values = null, bool $escape = null);

    /**
     * HAVING IN
     *
     * Generates a HAVING field IN('item', 'item') SQL query,
     * joined with 'AND' if appropriate.
     *
     * @param string               $key    The field to search
     * @param array|string|Closure $values The values searched on, or anonymous function with subquery
     * @param boolean              $escape
     *
     * @return self
     */
    public function havingIn(string $key = null, $values = null, bool $escape = null);

    /**
     * OR HAVING IN
     *
     * Generates a HAVING field IN('item', 'item') SQL query,
     * joined with 'OR' if appropriate.
     *
     * @param string               $key    The field to search
     * @param array|string|Closure $values The values searched on, or anonymous function with subquery
     * @param boolean              $escape
     *
     * @return self
     */
    public function orHavingIn(string $key = null, $values = null, bool $escape = null);

    /**
     * HAVING NOT IN
     *
     * Generates a HAVING field NOT IN('item', 'item') SQL query,
     * joined with 'AND' if appropriate.
     *
     * @param string               $key    The field to search
     * @param array|string|Closure $values The values searched on, or anonymous function with subquery
     * @param boolean              $escape
     *
     * @return self
     */
    public function havingNotIn(string $key = null, $values = null, bool $escape = null);

    /**
     * OR HAVING NOT IN
     *
     * Generates a HAVING field NOT IN('item', 'item') SQL query,
     * joined with 'OR' if appropriate.
     *
     * @param string               $key    The field to search
     * @param array|string|Closure $values The values searched on, or anonymous function with subquery
     * @param boolean              $escape
     *
     * @return self
     */
    public function orHavingNotIn(string $key = null, $values = null, bool $escape = null);

    /**
     * LIKE
     *
     * Generates a %LIKE% portion of the query.
     * Separates multiple calls with 'AND'.
     *
     * @param mixed   $field
     * @param string  $match
     * @param string  $side
     * @param boolean $escape
     * @param boolean $insensitiveSearch IF true, will force a case-insensitive search
     *
     * @return self
     */
    public function like($field, string $match = '', string $side = 'both', bool $escape = null, bool $insensitiveSearch = false);


    /**
     * NOT LIKE
     *
     * Generates a NOT LIKE portion of the query.
     * Separates multiple calls with 'AND'.
     *
     * @param mixed   $field
     * @param string  $match
     * @param string  $side
     * @param boolean $escape
     * @param boolean $insensitiveSearch IF true, will force a case-insensitive search
     *
     * @return self
     */
    public function notLike($field, string $match = '', string $side = 'both', bool $escape = null, bool $insensitiveSearch = false);


    /**
     * OR LIKE
     *
     * Generates a %LIKE% portion of the query.
     * Separates multiple calls with 'OR'.
     *
     * @param mixed   $field
     * @param string  $match
     * @param string  $side
     * @param boolean $escape
     * @param boolean $insensitiveSearch IF true, will force a case-insensitive search
     *
     * @return self
     */
    public function orLike($field, string $match = '', string $side = 'both', bool $escape = null, bool $insensitiveSearch = false);

    /**
     * OR NOT LIKE
     *
     * Generates a NOT LIKE portion of the query.
     * Separates multiple calls with 'OR'.
     *
     * @param mixed   $field
     * @param string  $match
     * @param string  $side
     * @param boolean $escape
     * @param boolean $insensitiveSearch IF true, will force a case-insensitive search
     *
     * @return self
     */
    public function orNotLike($field, string $match = '', string $side = 'both', bool $escape = null, bool $insensitiveSearch = false);

    /**
     * LIKE with HAVING clause
     *
     * Generates a %LIKE% portion of the query.
     * Separates multiple calls with 'AND'.
     *
     * @param mixed   $field
     * @param string  $match
     * @param string  $side
     * @param boolean $escape
     *
     * @return self
     */
    public function havingLike($field, string $match = '', string $side = 'both', bool $escape = null, bool $insensitiveSearch = false);


    /**
     * NOT LIKE with HAVING clause
     *
     * Generates a NOT LIKE portion of the query.
     * Separates multiple calls with 'AND'.
     *
     * @param mixed   $field
     * @param string  $match
     * @param string  $side
     * @param boolean $escape
     *
     * @return self
     */
    public function notHavingLike($field, string $match = '', string $side = 'both', bool $escape = null, bool $insensitiveSearch = false);

    /**
     * OR LIKE with HAVING clause
     *
     * Generates a %LIKE% portion of the query.
     * Separates multiple calls with 'OR'.
     *
     * @param mixed   $field
     * @param string  $match
     * @param string  $side
     * @param boolean $escape
     *
     * @return self
     */
    public function orHavingLike($field, string $match = '', string $side = 'both', bool $escape = null, bool $insensitiveSearch = false);

    /**
     * OR NOT LIKE with HAVING clause
     *
     * Generates a NOT LIKE portion of the query.
     * Separates multiple calls with 'OR'.
     *
     * @param mixed   $field
     * @param string  $match
     * @param string  $side
     * @param boolean $escape
     *
     * @return self
     */
    public function orNotHavingLike($field, string $match = '', string $side = 'both', bool $escape = null, bool $insensitiveSearch = false);


    /**
     * Starts a query group.
     *
     * @return self
     */
    public function groupStart();

    /**
     * Starts a query group, but ORs the group
     *
     * @return self
     */
    public function orGroupStart();

    /**
     * Starts a query group, but NOTs the group
     *
     * @return self
     */
    public function notGroupStart();

    /**
     * Starts a query group, but OR NOTs the group
     *
     * @return self
     */
    public function orNotGroupStart();

    /**
     * Ends a query group
     *
     * @return self
     */
    public function groupEnd();


    /**
     * Starts a query group for HAVING clause.
     *
     * @return self
     */
    public function havingGroupStart();

    /**
     * Starts a query group for HAVING clause, but ORs the group.
     *
     * @return self
     */
    public function orHavingGroupStart();

    /**
     * Starts a query group for HAVING clause, but NOTs the group.
     *
     * @return self
     */
    public function notHavingGroupStart();

    /**
     * Starts a query group for HAVING clause, but OR NOTs the group.
     *
     * @return self
     */
    public function orNotHavingGroupStart();

    /**
     * Ends a query group for HAVING clause.
     *
     * @return self
     */
    public function havingGroupEnd();

    /**
     * GROUP BY
     *
     * @param string|array $by
     * @param boolean      $escape
     *
     * @return self
     */
    public function groupBy($by, bool $escape = null);

    /**
     * HAVING
     *
     * Separates multiple calls with 'AND'.
     *
     * @param string|array $key
     * @param mixed        $value
     * @param boolean      $escape
     *
     * @return self
     */
    public function having($key, $value = null, bool $escape = null);

    /**
     * OR HAVING
     *
     * Separates multiple calls with 'OR'.
     *
     * @param string|array $key
     * @param mixed        $value
     * @param boolean      $escape
     *
     * @return self
     */
    public function orHaving($key, $value = null, bool $escape = null);

    /**
     * ORDER BY
     *
     * @param string  $orderBy
     * @param string  $direction ASC, DESC or RANDOM
     * @param boolean $escape
     *
     * @return self
     */
    public function orderBy(string $orderBy, string $direction = '', bool $escape = null);

    /**
     * LIMIT
     *
     * @param integer $value  LIMIT value
     * @param integer $offset OFFSET value
     *
     * @return self
     */
    public function limit(int $value = null, ?int $offset = 0);

    /**
     * Sets the OFFSET value
     *
     * @param integer $offset OFFSET value
     *
     * @return self
     */
    public function offset(int $offset);

    /**
     * The "set" function.
     *
     * Allows key/value pairs to be set for insert(), update() or replace().
     *
     * @param string|array|object $key    Field name, or an array of field/value pairs
     * @param string              $value  Field value, if $key is a single field
     * @param boolean             $escape Whether to escape values and identifiers
     *
     * @return self
     */
    public function set($key, ?string $value = '', bool $escape = null);

    /**
     * Returns the previously set() data, alternatively resetting it
     * if needed.
     *
     * @param boolean $clean
     *
     * @return array
     */
    public function getSetData(bool $clean = false): array;

    /**
     * Get SELECT query string
     *
     * Compiles a SELECT query string and returns the sql.
     *
     * @param boolean $reset TRUE: resets QB values; FALSE: leave QB values alone
     *
     * @return string
     */
    public function getCompiledSelect(bool $reset = true): string;

    /**
     * Get
     *
     * Compiles the select statement based on the other functions called
     * and runs the query
     *
     * @param integer $limit  The limit clause
     * @param integer $offset The offset clause
     * @param boolean $reset  Are we want to clear query builder values?
     *
     * @return ResultInterface
     */
    public function get(int $limit = null, int $offset = 0, bool $reset = true);

    /**
     * "Count All" query
     *
     * Generates a platform-specific query string that counts all records in
     * the specified database
     *
     * @param boolean $reset Are we want to clear query builder values?
     *
     * @return integer|string when $test = true
     */
    public function countAll(bool $reset = true);

    /**
     * "Count All Results" query
     *
     * Generates a platform-specific query string that counts all records
     * returned by an Query Builder query.
     *
     * @param boolean $reset
     *
     * @return integer|string when $test = true
     */
    public function countAllResults(bool $reset = true);

    /**
     * Get compiled 'where' condition string
     *
     * Compiles the set conditions and returns the sql statement
     *
     * @return string
     */
    public function getCompiledQBWhere();

    /**
     * Get_Where
     *
     * Allows the where clause, limit and offset to be added directly
     *
     * @param string|array $where  Where condition
     * @param integer      $limit  Limit value
     * @param integer      $offset Offset value
     * @param boolean      $reset  Are we want to clear query builder values?
     *
     * @return ResultInterface
     */
    public function getWhere($where = null, int $limit = null, ?int $offset = 0, bool $reset = true);

    /**
     * Insert_Batch
     *
     * Compiles batch insert strings and runs the queries
     *
     * @param array   $set       An associative array of insert values
     * @param boolean $escape    Whether to escape values and identifiers
     * @param integer $batchSize Batch size
     *
     * @return integer Number of rows inserted or FALSE on failure
     * @throws DatabaseException
     */
    public function insertBatch(array $set = null, bool $escape = null, int $batchSize = 100);

    /**
     * The "setInsertBatch" function.  Allows key/value pairs to be set for batch inserts
     *
     * @param mixed   $key
     * @param string  $value
     * @param boolean $escape
     *
     * @return self|null
     */
    public function setInsertBatch($key, string $value = '', bool $escape = null);

    /**
     * Get INSERT query string
     *
     * Compiles an insert query and returns the sql
     *
     * @param boolean $reset TRUE: reset QB values; FALSE: leave QB values alone
     *
     * @throws DatabaseException
     *
     * @return string
     */
    public function getCompiledInsert(bool $reset = true): string;

    /**
     * Insert
     *
     * Compiles an insert string and runs the query
     *
     * @param array   $set    An associative array of insert values
     * @param boolean $escape Whether to escape values and identifiers
     *
     * @throws DatabaseException
     *
     * @return BaseResult|Query|false
     */
    public function insert(array $set = null, bool $escape = null);

    /**
     * Replace
     *
     * Compiles an replace into string and runs the query
     *
     * @param array $set An associative array of insert values
     *
     * @return BaseResult|Query|string|false
     * @throws DatabaseException
     */
    public function replace(array $set = null);

    /**
     * Get UPDATE query string
     *
     * Compiles an update query and returns the sql
     *
     * @param boolean $reset TRUE: reset QB values; FALSE: leave QB values alone
     *
     * @return string
     */
    public function getCompiledUpdate(bool $reset = true): string;

    /**
     * UPDATE
     *
     * Compiles an update string and runs the query.
     *
     * @param array   $set   An associative array of update values
     * @param mixed   $where
     * @param integer $limit
     *
     * @throws DatabaseException
     *
     * @return boolean    TRUE on success, FALSE on failure
     */
    public function update(array $set = null, $where = null, int $limit = null): bool;

    /**
     * Update_Batch
     *
     * Compiles an update string and runs the query
     *
     * @param array   $set       An associative array of update values
     * @param string  $index     The where key
     * @param integer $batchSize The size of the batch to run
     *
     * @return mixed    Number of rows affected, SQL string, or FALSE on failure
     * @throws \CodeIgniter\Database\Exceptions\DatabaseException
     */
    public function updateBatch(array $set = null, string $index = null, int $batchSize = 100);

    /**
     * The "setUpdateBatch" function.  Allows key/value pairs to be set for batch updating
     *
     * @param array|object $key
     * @param string       $index
     * @param boolean      $escape
     *
     * @return self|null
     * @throws \CodeIgniter\Database\Exceptions\DatabaseException
     */
    public function setUpdateBatch($key, string $index = '', bool $escape = null);

    /**
     * Empty Table
     *
     * Compiles a delete string and runs "DELETE FROM table"
     *
     * @return boolean    TRUE on success, FALSE on failure
     */
    public function emptyTable();

    /**
     * Truncate
     *
     * Compiles a truncate string and runs the query
     * If the database does not support the truncate() command
     * This function maps to "DELETE FROM table"
     *
     * @return boolean    TRUE on success, FALSE on failure
     */
    public function truncate();

    /**
     * Get DELETE query string
     *
     * Compiles a delete query string and returns the sql
     *
     * @param boolean $reset TRUE: reset QB values; FALSE: leave QB values alone
     *
     * @return string
     */
    public function getCompiledDelete(bool $reset = true): string;

    /**
     * Delete
     *
     * Compiles a delete string and runs the query
     *
     * @param mixed   $where      The where clause
     * @param integer $limit      The limit clause
     * @param boolean $reset_data
     *
     * @return mixed
     * @throws \CodeIgniter\Database\Exceptions\DatabaseException
     */
    public function delete($where = '', int $limit = null, bool $reset_data = true);


    /**
     * Increments a numeric column by the specified value.
     *
     * @param string  $column
     * @param integer $value
     *
     * @return boolean
     */
    public function increment(string $column, int $value = 1);

    /**
     * Decrements a numeric column by the specified value.
     *
     * @param string  $column
     * @param integer $value
     *
     * @return boolean
     */
    public function decrement(string $column, int $value = 1);

    /**
     * Reset Query Builder values.
     *
     * Publicly-visible method to reset the QB values.
     *
     * @return self
     */
    public function resetQuery();

}