<?php


namespace GekTools\Membership;



use CodeIgniter\Session\Session;
use CodeIgniter\I18n\Time;
use Config\Services;
use GekTools\Membership\Entities\Auth;
use GekTools\Membership\Entities\User;
use GekTools\Membership\Models\AuthsModel;
use GekTools\Tools\Cache\DataCache;
use GekTools\Tools\Cache\Ttl;

class MembershipService
{

    protected const LOGIN_KEY = 'login';
    protected const LOGGED_KEY = 'logged';
    protected const AUTH_KEY = 'auth';
    protected const TOKEN_COOKIE = 'gkat'; // Gek Kullanıcı Auth Token
    protected const GUEST_KEY = 'gekguest';
    protected const GUEST_GUID_COOKIE = 'gmgn'; // Gek Misafir Guid No

    #region fields

    /**
     * @var User|null
     */
    protected ?User $cachedUser = null;

    /**
     * @var User|null
     */
    protected ?User $cachedGuest = null;

    /**
     * @var Auth|null
     */
    protected ?Auth $cachedAuth = null;

    /**
     * @var AuthsModel
     */
    protected AuthsModel $authsModel;

    /**
     * @var UserService
     */
    protected UserService $userService;

    /**
     * @var Session
     */
    protected Session $session;

    /**
     * @var DataCache
     */
    protected DataCache $dataCache;


    #endregion fields

    #region ctor

    public function __construct()
    {

        $this->authsModel = model(AuthsModel::class);
        $this->userService = UserService::instance();
        $this->session = Services::session();
        $this->dataCache = DataCache::instance();
        helper('cookie');

        $this->getAuthsModel()
            ->onAfterUpdate(function ($eventData){
                $fi = Auth::getFieldsInfo();
                $cacheKey = DataCache::createKey(Auth::class,[$fi->token => $eventData['data'][$fi->token]]);
                DataCache::instance()->delete($cacheKey);
                return $eventData;
            });
        $this->getAuthsModel()
            ->onBeforeDelete(function ($eventData){
                $fi = Auth::getFieldsInfo();
                $id = $eventData['id'];
                $aModel = AuthsModel::instance(false);
                if(is_array($id)){
                    $id = array_shift($id);
                }
                /** @var Auth $auth */
                $auth = $aModel->getById($id);
                if($auth){
                    $cacheKey = DataCache::createKey(Auth::class,[$fi->token => $auth->getToken()]);
                    DataCache::instance()->delete($cacheKey);
                }

                return $eventData;
            });

    }

    #endregion ctor

    #region Properties

    /**
     * @return UserService
     */
    public function getUserService(): UserService
    {
        return $this->userService;
    }

    /**
     * @return AuthsModel
     */
    public function getAuthsModel(): AuthsModel
    {
        return $this->authsModel;
    }

    #endregion Properties

    #region methods

    public function singIn(User $user, bool $rememberMe = false){
        $auth = $this->createAuth($user,$rememberMe);
        $this->setSignData($user,$auth);
    }

    public function singOut(){
        if($this->session->get(self::LOGIN_KEY) == true){
            $authToken = $this->session->get(self::AUTH_KEY);
        }else{
            $authToken = get_cookie(self::TOKEN_COOKIE, false);
        }
        $auth = $this->getAuthByToken($authToken);
        if($auth){
            $this->getAuthsModel()->deleteEntity($auth);
        }

        $this->session->remove(self::LOGIN_KEY);
        $this->session->remove(self::LOGGED_KEY);
        $this->session->remove(self::AUTH_KEY);
        set_cookie(self::TOKEN_COOKIE,'','');
        $this->cachedUser = null;
        $this->cachedAuth = null;
    }

    /**
     * @return User|null
     * @throws \Exception
     */
    public function getAuthenticatedUser():?User{
        if($this->cachedUser != null){
            return $this->cachedUser;
        }

        if($this->session->get(self::LOGIN_KEY) == true){
            $userGuid = $this->session->get(self::LOGGED_KEY);
            $authToken = $this->session->get(self::AUTH_KEY);
            $this->cachedUser = $this->getUserService()->getUserByUserGuid($userGuid);
            $this->cachedAuth = $this->getAuthByToken($authToken);
            return  $this->cachedUser;
        }

        $authToken = get_cookie(self::TOKEN_COOKIE, false);
        if($authToken !== null){
            $auth = $this->getAuthByToken($authToken);
            if($auth === null){
                $this->singOut();
                return null;
            }

            if(! $auth->getRemember() || ! $this->verifyAuth($auth) ){
                $this->singOut();
                return null;
            }

            $user = $this->getUserService()->getUserByUserGuid($auth->getUserGuid());
            if($user == null || $user->getStatus()->toInt() != UserStatus::ACTIVE){
                $this->singOut();
                return null;
            }
            $this->setSignData($user,$auth);
        }

        return $this->cachedUser;

    }

    /**
     * @return User|null
     * @throws \Exception
     */
    public function getGuestUser():?User{
        if($this->cachedGuest != null){
            return $this->cachedGuest;
        }
        $guest = null;
        $gguid = $this->session->get(self::GUEST_KEY);
        if(empty($gguid)){
            $gguid = get_cookie(self::GUEST_GUID_COOKIE, false);
        }
        if(!empty($gguid)){
            $guest = $this->userService->getUserByUserGuid($gguid);
        }

        if(empty($guest)){
            $guest = $this->userService->insertNewGuestUser();
        }

        if(!empty($guest)){
            if($guest->isRegistered()){
                $guest = null;
                $this->session->remove(self::GUEST_KEY);
                set_cookie(self::GUEST_GUID_COOKIE,'','');
            }else{
                $this->session->set(self::GUEST_KEY,$guest->getUserGuid());
                set_cookie(self::GUEST_GUID_COOKIE,$guest->getUserGuid(),Ttl::fromMonths(15));
            }
        }
        $this->cachedGuest = $guest;
        return $this->cachedGuest;

    }

    /**
     * @param string $authToken
     * @return Auth|null
     */
    public function getAuthByToken(string $authToken):?Auth{
        $fi = Auth::getFieldsInfo();
        $cacheKey = $this->dataCache::createKey(Auth::class,[$fi->token => $authToken]);
        return $this->dataCache->getOrSave(
            $cacheKey,
            [$this,'getAuthByTokenNoCache'],
            [$authToken],
            Ttl::fromMinutes(3)
        );
    }

    /**
     * @param string $authToken
     * @return Auth|null
     */
    public function getAuthByTokenNoCache(string $authToken):?Auth{
        return $this->getAuthsModel()->getByToken($authToken);
    }

    #endregion methods

    #region utils

    protected function setSignData(User $user,Auth $auth){
        $this->session->set(self::LOGIN_KEY, true);
        $this->session->set(self::LOGGED_KEY, $user->getUserGuid());
        $this->session->set(self::AUTH_KEY,  $auth->getToken());

        set_cookie(self::TOKEN_COOKIE,$auth->getToken(),Ttl::fromMonths(15));

        $this->cachedUser = $user;
        $this->cachedAuth = $auth;
    }

    /**
     * @param User $user
     * @param bool $rememberMe
     * @return Auth
     * @throws \ReflectionException
     */
    protected function createAuth(User $user, bool $rememberMe):Auth{
        $request = Services::request();
        $auth = new Auth();
        $expDate = Time::now('UTC');
        if($rememberMe){
            $expDate = $expDate->addMonths(15);
        }
        $auth->setUserGuid($user->getUserGuid())
            ->setAgentString($request->getUserAgent()->getAgentString())
            ->setPlatform($request->getUserAgent()->getPlatform())
            ->setBrowser($request->getUserAgent()->getBrowser())
            ->setIsMobile($request->getUserAgent()->isMobile())
            ->setRemember($rememberMe)
            ->setToken($this->createAuthToken())
            ->setIpAddress($request->getIPAddress())
            ->setExpiredDateUtc($expDate);
        $this->getAuthsModel()->save($auth);
        return $auth;
    }

    /**
     * @return string
     */
    protected function createAuthToken():string {
        return md5(uniqid(mt_rand(), true));
    }

    /**
     * @param Auth $auth
     * @return bool
     * @throws \Exception
     */
    public function verifyAuth(Auth $auth){
        $request = Services::request();

        if(! $auth->getExpiredDateUtc()->isAfter(Time::now('UTC'))){
            return false;
        }

        if( $auth->getIsMobile() != $request->getUserAgent()->isMobile()){
            return false;
        }
        if( $auth->getBrowser() != $request->getUserAgent()->getBrowser()){
            return false;
        }

        if( $auth->getPlatform() != $request->getUserAgent()->getPlatform()){
            return false;
        }

        if( $auth->getAgentString() != $request->getUserAgent()->getAgentString()){
            return false;
        }

        return true;
    }

    #endregion utils

    #region static

    /**
     * @return MembershipService
     */
    public static function instance(): self
    {
        return service('membershipService', true);
    }


    #endregion static

}
