<?php


namespace GekTools\Tools\Database;


use CodeIgniter\CLI\CLI;
use CodeIgniter\Database\BaseConnection;
use Config\Database;

class DbInfo
{
    #region fields
    /**
     * @var BaseConnection
     */
    private BaseConnection $db;
    /**
     * @var array|null
     */
    private $typesMaps  = null;

    #endregion fields

    #region ctor

    /**
     * DbInfo constructor.
     * @param string|array|null $group
     * @param bool $getShared
     */
    public function __construct($group = null, bool $getShared = true)
    {
        $this->db = Database::connect($group,$getShared);
    }

    #endregion ctor

    #region methods

    /**
     * @param string $tableName
     * @return array|TableFieldInfo[]
     * @throws \ReflectionException
     */
    public function getFieldInfos(string $tableName):array {
        $fields = $this->getTableFields($tableName);
        $this->fillTypesMaps();
        $infos = array();
        foreach ($fields as $row){
            $cInfo = $this->getFieldInfo($row);
            $comment = $this->getFieldComment($tableName,$row->Field);
            if(!empty($comment)){
                $comment = unserialize($comment);
                $cInfo->setComment($comment);
            }
            $infos[] = $cInfo;
        }
        return $infos;
    }

    /**
     * @param string $tableName
     * @return TableComment|null
     */
    public function getTableComment(string $tableName):?TableComment {
        $sql = 'SELECT table_comment ' . PHP_EOL;
        $sql .= 'FROM information_schema.tables ' . PHP_EOL;
        $sql .= "WHERE table_schema = '" . $this->db->getDatabase() .
            "' AND table_name = '" . $tableName ."' " . PHP_EOL;
        $q = $this->db->query($sql);
        $res =  $q->getRow(0,'array');
        $tableComment = null;
        if($res != null){
            $comment = $res['table_comment'];
            $comment = trim($comment," \t\n\r\0\x0B'");
            if(!empty($comment)){
                $tableComment = unserialize($comment);
            }
        }
        return $tableComment;
    }

    /**
     * @param string $tableName
     * @param string $fieldName
     * @return string|null
     */
    public function getFieldComment(string $tableName, string $fieldName):?string
    {
        $sql = 'SELECT COLUMN_NAME, COLUMN_COMMENT ' . PHP_EOL;
        $sql .= 'FROM information_schema.columns ' . PHP_EOL;
        $sql .= 'WHERE table_schema = '. "'" . $this->db->getDatabase() ."' ";
        $sql .= 'AND table_name = '. "'" . $tableName ."' ";
        $sql .= 'AND COLUMN_NAME = '. "'" . $fieldName ."' " . PHP_EOL;
        $q = $this->db->query($sql);
        $res =  $q->getRow(0,'array');
        if($res != null){
            return $res['COLUMN_COMMENT'];
        }
        return null;
    }

    /**
     * @param string $tableName
     * @return array
     */
    public function getTableFields(string $tableName):array {
        $sql = 'SHOW COLUMNS FROM '.$tableName;
        $q = $this->db->query($sql);
        return $q->getResult();
    }

    /**
     * @return array
     */
    public function getAllTableNames():array {
        $sql = 'SELECT TABLE_NAME ' . PHP_EOL;
        $sql .= 'FROM information_schema.tables ' . PHP_EOL;
        $sql .= "WHERE table_schema = '" . $this->db->getDatabase() .
            "' AND table_type = 'base table' " . PHP_EOL;
        $q = $this->db->query($sql);
        $res =  $q->getResult('object');
        return $res;
    }

    #endregion methods

    #region utils

    /**
     * @param $row
     * @return TableFieldInfo
     * @throws \ReflectionException
     */
    private function getFieldInfo($row):TableFieldInfo{
        $info = new TableFieldInfo($row->Field);
        $strDbType = '';
        $maxLength = '';
        sscanf($row->Type, '%[a-z](%d)',
            $strDbType,
            $maxLength
        );
        $strDbType = strtoupper(trim($strDbType));
        $dbType = new DbTypes($strDbType);
        $nullable = (strtoupper($row->Null) == 'YES') ? true : false;
        $info->setDbType($dbType)
            ->setLength($maxLength)
            ->setNullable($nullable)
            ->setKey(empty($row->Key) ? null : $row->Key)
            ->setExtra(empty($row->Extra) ? null : $row->Extra)
            ->setType($this->typesMaps[strtolower($strDbType)]);
        if($row->Default === null){
            if($nullable){
                $info->setDefault($row->Default);
            }
        }else{
            $info->setDefault($row->Default);
        }

        return $info;
    }


    private function fillTypesMaps():void {
        if($this->typesMaps != null){
            return;
        }
        $tmp = array();
        $tmp["bool"] =
        $tmp["boolean"] =
        $tmp["bit"] =
        $tmp["tinyint"] = "bool";

        $tmp["int"]  =
        $tmp["integer"] =
        $tmp["smallint"] =
        $tmp["mediumint"] =
        $tmp["bigint"] =
        $tmp["year"] = 'int';

        $tmp["float"] =
        $tmp["double"] =
        $tmp["real"] =
        $tmp["decimal"] =
        $tmp["numeric"] =
        $tmp["dec"] =
        $tmp["fixed"] =
        $tmp["serial"] =  'float';

        $tmp["json"] =
        $tmp["date"] =
        $tmp["timestamp"] =
        $tmp["datetime"] =
        $tmp["time"] =
        $tmp["char"] =
        $tmp["varchar"] =
        $tmp["tinytext"] =
        $tmp["text"] =
        $tmp["mediumtext"] =
        $tmp["longtext"] =
        $tmp["set"] =
        $tmp["enum"] =
        $tmp["nchar"] =
        $tmp["nvarchar"] = 'string';
        $this->typesMaps = $tmp;
        $tmp = null;
    }

    #endregion utils
}