<?php


namespace GekTools\Membership;


use CodeIgniter\I18n\Time;
use Config\Services;
use GekTools\Membership\Entities\User;
use GekTools\Membership\Settings\MembershipSettings;
use GekTools\Settings\SettingsService;

class UserRegistrationsService
{

    #region fields

    /**
     * @var UserService
     */
    protected UserService $userService;

    /**
     * @var MembershipSettings
     */
    protected MembershipSettings $settings;

    #endregion fields

    #region ctor

    public function __construct()
    {
        $this->userService = UserService::instance();
        $this->settings = SettingsService::instance()->getSettings(MembershipSettings::class);
    }

    #endregion ctor

    #region properties

    /**
     * @return UserService
     */
    public function getUserService(): UserService
    {
        return $this->userService;
    }

    /**
     * @return MembershipSettings
     */
    public function getSettings(): MembershipSettings
    {
        return $this->settings;
    }

    #endregion properties

    #region methods

    public function validateUser(string $userNameOrEmail, string $password, ?User &$user){
        $user = $this->getUserService()->getUserForLogin($userNameOrEmail);

        if($user === null){
            return LoginResults::USER_NOT_EXIST;
        }

        if($user->getDeletedAtUtc() !== null){
            return LoginResults::DELETED;
        }

        if($user->getStatus()->toInt() !== UserStatus::ACTIVE){
            return LoginResults::NOT_ACTIVE;
        }

        if(! $this->validatePassword($user,$password)){
            return LoginResults::WRONG_PASSWORD;
        }

        $request = Services::request();
        $ip = $request->getIPAddress();

        $nowDate = Time::now('UTC');
        $user->setLastLoginDateUtc($nowDate)
            ->setLastActivityDataUtc($nowDate)
            ->setLastIp($ip);
        $this->getUserService()->getUserModel()->save($user);

        return LoginResults::SUCCESSFUL;
    }

    /**
     * @param User $user
     * @param string $email
     * @param string $name
     * @param string $lastName
     * @param string|null $username
     * @param string $password
     * @param string $roleSystemName
     * @param UserStatus $status
     * @return RegistrationResult
     * @throws \ReflectionException
     */
    public function registerUser(User $user, string $email, string $name, string $lastName, ?string $username, ?string $phone, string $password, string $roleSystemName, UserStatus $status){
        $result = new RegistrationResult();
        if($user->isSearchEngineAccount()){
            $result->addError('Arama motorları kayıt olamaz.');
            return $result;
        }
        if($user->isBackgroundTaskAccount()){
            $result->addError('Arka plan görev hesabı kayıt olamaz.');
            return $result;
        }
        if($user->isRegistered()){
            $result->addError('Kullanıcı zaten kayıtlı.');
            return $result;
        }
        if(empty($email)){
            $result->addError('E-posta boş!');
            return $result;
        }
        if(empty($password)){
            $result->addError('Şifre boş!');
            return $result;
        }

        if($this->getSettings()->isUserNamesEnabled() && empty($username)){
            $result->addError('Kullanıcı adı boş!');
            return $result;
        }

        if(
            $this->getSettings()->isPhoneEnabled() &&
            $this->getSettings()->isPhoneRequired()  &&
            empty($phone)
        ){
            $result->addError('Telefon boş!');
            return $result;
        }

        if($this->getUserService()->getUserByEmail($email,true) !== null){
            $result->addError('Bu e-posta zaten kayıtlı.');
            return $result;
        }

        if($this->getSettings()->isUserNamesEnabled()){
            if($this->getUserService()->getUserByUserName($username,true) !== null){
                $result->addError('Bu kullanıcı adı kullanımda.');
                return $result;
            }
        }

        if($this->getSettings()->isPhoneEnabled() && ! empty($phone)){
            if($this->getUserService()->getUserByPhone($phone,true) !== null){
                $result->addError('Telefon numarası kullanımda.');
                return $result;
            }
        }



        $role = $this->getUserService()->getUserRoleBySystemName($roleSystemName);

        if(empty($role)){
            $result->addError('Rol mevcut değil.');
            return $result;
        }

        $user->setEmail($email)
            ->setPasswordFormat(PasswordFormat::MD5())
            ->setPasswordHash(md5($password))
            ->setName($name)
            ->setLastName($lastName);
        if($this->getSettings()->isUserNamesEnabled()){
            $user->setUserName($username);
        }

        $this->getUserService()->getUserModel()->save($user);


        $guestRole = $this->getUserService()->getUserRoleBySystemName(SystemRoleNames::GUESTS);

        $user->removeRole($guestRole);
        $user->addRole($role);
        $user->refresh();
        return $result;
    }



    #endregion methods

    #region utils

    /**
     * @param User $user
     * @param string $password
     * @return bool
     * @throws \Exception
     */
    protected function validatePassword(User $user, string $password){
        switch ($user->getPasswordFormat()->toInt()){
            case PasswordFormat::MD5:
                return $user->getPasswordHash() == md5($password);
                break;
            default:
                return false;
                break;
        }
    }

    #endregion utils

    #region static

    /**
     * @return UserRegistrationsService
     */
    public static function instance():self {
        return service('userRegistrationsService', true);
    }

    #endregion static

}