<?php


namespace GekTools\Membership;


use GekTools\Membership\Entities\PermissionRecord;
use GekTools\Membership\Entities\PermissionRoleMap;
use GekTools\Membership\Models\PermissionRecordsModel;
use GekTools\Membership\Models\PermissionRoleMappingsModel;
use GekTools\Tools\Cache\DataCache;
use GekTools\Tools\Cache\Ttl;

class PermissionsService
{

    #region fields

    /**
     * @var PermissionRecordsModel
     */
    protected PermissionRecordsModel $permissionRecordModel;

    /**
     * @var PermissionRoleMappingsModel
     */
    protected PermissionRoleMappingsModel $permissionRoleMappingsModel;

    /**
     * @var DataCache
     */
    protected DataCache $dataCache;



    #endregion fields

    #region ctor

    public function __construct()
    {
        $this->permissionRecordModel = model(PermissionRecordsModel::class);
        $this->permissionRoleMappingsModel = model(PermissionRoleMappingsModel::class);
        $this->dataCache = service('dataCache');

        $this->getPermissionRecordModel()->onAfterUpdate([$this,'cleanCachedPermissions']);
        $this->getPermissionRecordModel()->onAfterInsert([$this,'cleanCachedPermissions']);
        $this->getPermissionRecordModel()->onAfterDelete([$this,'cleanCachedPermissions']);

        $this->getPermissionRoleMappingsModel()->onAfterUpdate([$this,'cleanCachedPermissions']);
        $this->getPermissionRoleMappingsModel()->onAfterInsert([$this,'cleanCachedPermissions']);
        $this->getPermissionRoleMappingsModel()->onAfterDelete([$this,'cleanCachedPermissions']);
    }

    #endregion ctor

    #region properties

    /**
     * @return PermissionRecordsModel
     */
    public function getPermissionRecordModel(): PermissionRecordsModel
    {
        return $this->permissionRecordModel;
    }

    /**
     * @return DataCache
     */
    public function getDataCache(): DataCache
    {
        return $this->dataCache;
    }

    /**
     * @return PermissionRoleMappingsModel
     */
    public function getPermissionRoleMappingsModel(): PermissionRoleMappingsModel
    {
        return $this->permissionRoleMappingsModel;
    }

    #endregion properties

    #region methods

    /**
     * @param int $userRoleId
     * @return \GekTools\Tools\Collections\PagedResult|PermissionRecord[]
     */
    public function getPermissionRecordsByUserRoleId(int $userRoleId){
        $cacheKey = $this->dataCache::createKey(PermissionRecord::class,['UserRoleId' => $userRoleId]);
        return $this->dataCache->getOrSave(
            $cacheKey,
            [$this,'getPermissionRecordsByUserRoleIdNoCache'],
            [$userRoleId],
            Ttl::fromWeeks(15)
        );
    }

    /**
     * @param int $userRoleId
     * @return \GekTools\Tools\Collections\PagedResult|PermissionRecord[]
     */
    public function getPermissionRecordsByUserRoleIdNoCache(int $userRoleId){
        $mFi = PermissionRoleMap::getFieldsInfo();
        $maps = $this->getPermissionRoleMappingsModel()
            ->where($mFi->userRoleId,$userRoleId)
            ->getAll();
        $ids = $maps->select(function (PermissionRoleMap $m){
            return $m->getPermissionRecordId();
        })->toArray();
        $fi = PermissionRecord::getFieldsInfo();

        return $this->getPermissionRecordModel()
            ->whereIn($fi->id,$ids)
            ->getAll();
    }

    public function cleanCachedPermissions(){
        $startsWith = $this->dataCache::createKey(PermissionRecord::class);
        $this->dataCache->deleteByStartWith($startsWith);
    }

    #endregion methods

    #region static

    /**
     * @return static
     */
    public static function instance(): self
    {
        return service('permissionsService', true);
    }

    #endregion static

}