<?php


namespace GekTools\Tools\Validations;


use Gek\Collections\ArrayList;
use Gek\Collections\Enumerable;
use Gek\Collections\TypedListWrapper;

class ValidationCollection extends TypedListWrapper
{

    #region ctor

    /**
     * IntList constructor.
     * @param Validation ...$items
     */
    public function __construct(Validation ...$items)
    {
        parent::__construct(new ArrayList(), Validation::class);
        $this->addRange(...$items);
    }

    #endregion ctor

    #region methods

    /**
     * @param Validation $item
     */
    public function add(Validation $item):void{
        $this->innerList->add($item);
    }

    /**
     * @param Validation ...$items
     */
    public function addRange(Validation ...$items):void{
        $this->innerList->addRange($items);
    }


    /**
     * @param Validation $item
     * @return bool
     */
    public function contains(Validation $item):bool{
        return $this->innerList->contains($item);
    }

    /**
     * @param Validation $item
     * @return bool
     */
    public function remove(Validation $item):bool{
        return $this->innerList->remove($item);
    }

    /**
     * @param Validation $item
     * @return int
     */
    public function indexOf(Validation $item): int{
        return $this->innerList->indexOf($item);
    }

    /**
     * @param int $index
     * @param Validation $item
     */
    public function insert(int $index, Validation $item): void{
        $this->innerList->insert($index,$item);
    }

    /**
     * @param int $index
     * @return Validation
     */
    public function getAt(int $index):Validation {
        return $this->innerList[$index];
    }

    /**
     * @param int $index
     * @param Validation $item
     */
    public function setAt(int $index, Validation $item):void{
        $this->innerList[$index] = $item;
    }

    /**
     * @param int $index
     * @param Validation ...$items
     */
    public function insertRange(int $index, Validation ...$items):void{
        $this->innerList->insertRange($index,$items);
    }


    /**
     * @internal
     * @return array
     */
    public function toArrayForSerialization(): array
    {
        return $this->select(function (Validation $validation){
            return $validation->toArrayForSerialize();
        })->toArray();
    }

    /**
     * @param string $name
     * @param array $params
     * @param bool $useModel
     * @param bool $useModelData
     * @return Validation
     */
    public function addValidation(string $name,array $params = array(),bool $useModel = true, bool $useModelData = true):Validation{
        $res = new Validation($name);
        $res->setArguments($params)
            ->setUseModel($useModel)
            ->setUseModelData($useModelData);
        $this->add($res);
        return $res;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addAlpha(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('alpha',array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addAlphaSpace(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('alpha_space',array(),$useModel,$useModelData);
        return $this;
    }
    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addAlphaDash(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('alpha_dash',array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addAlphaNumeric(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('alpha_numeric',array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addAlphaNumericSpace(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('alpha_numeric_space',array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addAlphaNumericPunct(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('alpha_numeric_punct',array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addDecimal(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('decimal',array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param string $fieldName
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addDiffers(string $fieldName, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('differs',[$fieldName],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param int|array|int[] $length
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addExactLength($length, bool $useModel = true, bool $useModelData = true):self {
        if(!is_array($length)){
            $length = [$length];
        }
        $this->addValidation('exact_length',$length,$useModel,$useModelData);
        return $this;
    }

    /**
     * @param number $number
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addGreaterThan($number, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('greater_than',[$number],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param number $number
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addGreaterThanEqualTo($number, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('greater_than_equal_to',[$number],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addHex(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('hex',[],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addIfExist(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('if_exist',[],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param array $values
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addInList(array $values,bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('in_list',$values,$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addInteger(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('integer',[],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addIsNatural(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('is_natural',[],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addIsNaturalNoZero(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('is_natural_no_zero',[],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param array $values
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addIsNotUnique(array $values, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('is_not_unique',$values,$useModel,$useModelData);
        return $this;
    }

    /**
     * @param array $values
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addIsUnique(array $values, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('is_unique',$values,$useModel,$useModelData);
        return $this;
    }

    /**
     * @param number $number
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addLessThan($number, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('less_than',[$number],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param number $number
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addLessThanEqualTo($number, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('less_than_equal_to',[$number],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param string $field
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addMatches(string $field, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('matches',[$field],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param int $length
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addMaxLength(int $length, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('max_length',[$length],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param int $length
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addMinLength(int $length, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('min_length',[$length],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addNumeric(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('numeric',[],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param $regex
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addRegexMatch($regex, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('regex_match',[$regex],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addPermitEmpty(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('permit_empty',[],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addRequired(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('required',[],$useModel,$useModelData);
        return $this;
    }

    /**
     * @param array|string|string[] $fields
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addRequiredWith($fields,bool $useModel = true, bool $useModelData = true):self {
        if(!is_array($fields)){
            $fields = [$fields];
        }
        $this->addValidation('required_with', $fields,$useModel,$useModelData);
        return $this;
    }

    /**
     * @param string|array|string[] $fields
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addRequiredWithout($fields, bool $useModel = true, bool $useModelData = true):self {
        if(!is_array($fields)){
            $fields = [$fields];
        }
        $this->addValidation('required_without', $fields,$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addString(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('string', array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addTimezone(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('timezone', array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addValidBase64(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('valid_base64', array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addValidJson(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('valid_json', array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addValidEmail(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('valid_email', array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addValidEmails(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('valid_emails', array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param string|null $version ipv6 veya ipv4 (opsiyonel)
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addValidIp(?string $version = null,bool $useModel = true, bool $useModelData = true):self {
        $param = array();
        if(!empty($version)){
            $param[] = $version;
        }
        $this->addValidation('valid_ip', $param,$useModel,$useModelData);
        return $this;
    }

    /**
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addValidUrl(bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('valid_url', array(),$useModel,$useModelData);
        return $this;
    }

    /**
     * @param string|null $format tarih formatı örn. 'd/m/Y' (opsiyonel)
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addValidDate(?string $format = null,bool $useModel = true, bool $useModelData = true):self {
        $param = array();
        if(!empty($format)){
            $param[] = $format;
         }
        $this->addValidation('valid_date', $param,$useModel,$useModelData);
        return $this;
    }

    /**
     * @link https://codeigniter4.github.io/userguide/libraries/validation.html#available-rules
     * @param string $provider kredi kartı sağlayısı örn. 'visa' veya 'maestro' ya da 'amex'
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addValidCcNumber(string $provider,bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('valid_cc_number', array($provider),$useModel,$useModelData);
        return $this;
    }

    /**
     * @link https://codeigniter4.github.io/userguide/libraries/validation.html#available-rules
     * @param string $fieldName
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addUploaded(string $fieldName, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('uploaded', array($fieldName),$useModel,$useModelData);
        return $this;
    }

    /**
     * @link https://codeigniter4.github.io/userguide/libraries/validation.html#available-rules
     * @param string $fieldName
     * @param int $maxSizeKb kb cinsinden
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addMaxSize(string $fieldName, int $maxSizeKb, bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('max_size', array($fieldName,$maxSizeKb),$useModel,$useModelData);
        return $this;
    }

    /**
     * @link https://codeigniter4.github.io/userguide/libraries/validation.html#available-rules
     * @param string $fieldName
     * @param int $maxWidth
     * @param int $maxHeight
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addMaxDims(string $fieldName, int $maxWidth, int $maxHeight,  bool $useModel = true, bool $useModelData = true):self {
        $this->addValidation('max_dims', array($fieldName,$maxWidth,$maxHeight),$useModel,$useModelData);
        return $this;
    }

    /**
     * @link https://codeigniter4.github.io/userguide/libraries/validation.html#available-rules
     * @param string $fieldName
     * @param array $mimeTypes
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addMimeIn(string $fieldName, array $mimeTypes,  bool $useModel = true, bool $useModelData = true):self {
        $param = array($fieldName);
        $param = array_merge($param,$mimeTypes);
        $this->addValidation('mime_in', $param,$useModel,$useModelData);
        return $this;
    }

    /**
     * @link https://codeigniter4.github.io/userguide/libraries/validation.html#available-rules
     * @param string $fieldName
     * @param array $fileExtensions
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addExtIn(string $fieldName, array $fileExtensions, bool $useModel = true, bool $useModelData = true):self {
        $param = array($fieldName);
        $param = array_merge($param,$fileExtensions);
        $this->addValidation('ext_in', $param,$useModel,$useModelData);
        return $this;
    }

    /**
     * @link https://codeigniter4.github.io/userguide/libraries/validation.html#available-rules
     * @param string $fieldName
     * @param bool $useModel
     * @param bool $useModelData
     * @return $this
     */
    public function addIsImage(string $fieldName, bool $useModel = true, bool $useModelData = true):self {
        $param = array($fieldName);
        $this->addValidation('is_image', $param,$useModel,$useModelData);
        return $this;
    }

    public function __toString()
    {
        return implode('|', $this->reverse()->toArray());
    }

    #endregion methods

    #region static
    /**
     * @param array $data
     * @return static
     */
    public static function fromSerializeArray(array $data):self{
        $res = new ValidationCollection();
        $values = Enumerable::fromArray($data)->select(function (array $dt){
            return Validation::fromSerializeArray($dt);
        })->toArray();
        $res->addRange(...$values);
        return $res;
    }
    #endregion static

}
