<?php


namespace GekTools\Tools\Routing;


use Gek\Infrastructure\Str;

class RouteObject
{

    #region fields

    /** @var string */
    protected string $rawPath;

    /** @var string */
    protected string $rawController;

    /** @var string */
    protected string $controller;

    /** @var string */
    protected string $method;

    /** @var int */
    protected int $parametersCount;

    /** @var array */
    protected array $parameters;

    /** @var string */
    protected string $path;

    #endregion fields

    #region ctor

    /**
     * RouteObject constructor.
     * @param string $pathString
     * @param string $controllerString
     */
    public function __construct(string $pathString, string $controllerString)
    {
        $this->rawPath = $pathString;
        $this->rawController = $controllerString;
        $this->initialise($pathString, $controllerString);
    }

    #endregion ctor

    #region properties


    /**
     * @return string
     */
    public function getController(): string
    {
        return $this->controller;
    }

    /**
     * @return string
     */
    public function getMethod(): string
    {
        return $this->method;
    }

    /**
     * @return int
     */
    public function getParametersCount(): int
    {
        return $this->parametersCount;
    }

    /**
     * @return array
     */
    public function getParameters(): array
    {
        return $this->parameters;
    }

    /**
     * @return string
     */
    public function getPath(): string
    {
        return $this->path;
    }

    #endregion properties

    #region methods

    /**
     * @param array $params
     * @return bool
     */
    public function checkParameters(array $params)
    {

        if (count($params) != $this->parametersCount) {
            return false;
        }
        $indx = 0;
        foreach ($params as $prm) {
            if (Str::contains($prm, '(')) {
                if (trim($this->parameters[$indx]) !== trim($prm)) {
                    return false;
                }

            } elseif (!preg_match($this->parameters[$indx], $prm)) {
                return false;
            }
            $indx++;
        }
        return true;
    }


    /**
     * @param array $params
     * @return string
     */
    public function getUrl(array $params = array())
    {
        if ($this->parametersCount > 0) {
            return Str::format($this->path, ...$params);
        }
        return $this->path;
    }

#endregion methods

#region utils

    /**
     * @param string $strPath
     * @param string $strController
     */
    protected
    function initialise(string $strPath, string $strController)
    {
        $cnt = explode('::', $strController);
        $this->controller = $cnt[0];
        $mth = explode('$', $cnt[1]);
        if (count($mth) > 1) {
            $methodName = array_shift($mth);
            $methodName = str_replace('/', '', $methodName);
            $this->method = $methodName;
            $this->parametersCount = count($mth);
            $this->parameters = $this->parsePath($strPath);
            $this->path = $strPath;
        } else {
            $this->method = $cnt[1];
            $this->parametersCount = 0;
            $this->parameters = array();
            $this->path = $strPath;
        }

    }

    /**
     * @param string $strPath
     * @return array
     */
    protected
    function parsePath(string &$strPath): array
    {
        $prmFlag = false;
        $level = 0;
        $params = array();
        $curParam = '';
        $newPath = '';

        for ($i = 0; $i < strlen($strPath); $i++) {
            $cur = $strPath[$i];
            if ($cur == '(') {
                $prmFlag = true;
                $level++;
                $curParam .= $cur;
            } elseif ($cur == ')' && $prmFlag) {
                $curParam .= $cur;
                $level--;
                if ($level == 0) {
                    $prmFlag = false;
                    $newPath .= '{' . count($params) . '}';
                    $params[] = $curParam;
                    $curParam = '';
                }
            } elseif ($prmFlag) {
                $curParam .= $cur;
            } else {
                $newPath .= $cur;
            }
        }
        $strPath = $newPath;
        return $params;
    }

#endregion utils

}