<?php


namespace GekTools\Tools\Database;


use Gek\Collections\Typed\StringList;
use Gek\PhpLang\DocComments\DocComment;

/**
 * Class TableComment
 * @package Gek\Ci4Tools\Database
 * @property DocComment $docComment
 * @property string $entityName
 * @property string $baseClass
 * @property string $namespace
 * @property StringList $interfaces
 * @property StringList $uses
 */
class TableComment implements \Serializable
{

    #region fields

    private ?string $namespace = null;

    private ?string $entityName = null;

    private ?string $baseClass = null;


    private StringList $interfaces;

    private StringList $uses;

    private ?DocComment $docComment = null;

    private ?string $tableName = null;

    #endregion fields

    #region ctor

    public function __construct()
    {
        $this->interfaces = new StringList();
        $this->uses = new StringList();
    }

    #endregion ctor

    #region props

    /**
     * Tablonun bir varlık tablosu olup olmadığını belirler
     *
     * @return bool
     */
    public function isEntity(): bool
    {
        return !empty($this->entityName);
    }

    /**
     * entity name getter
     *
     * @return string|null
     */
    public function getEntityName(): ?string
    {
        return $this->entityName;
    }

    /**
     * entity name setter
     *
     * @param string|null $entityName
     * @return self
     */
    public function setEntityName(?string $entityName): self
    {
        $this->entityName = $entityName;
        return $this;
    }

    /**
     * base class getter
     *
     * @return string|null
     */
    public function getBaseClass(): ?string
    {
        return $this->baseClass;
    }

    /**
     * base class setter
     *
     * @param string|null $baseClass
     * @return TableComment
     */
    public function setBaseClass(?string $baseClass): self
    {
        $this->baseClass = $baseClass;
        return $this;
    }

    /**
     * base class tanımlı mı
     *
     * @return bool
     */
    public function isBaseClass(): bool
    {
        return !empty($this->baseClass);
    }

    /**
     * namespace getter
     *
     * @return string|null
     */
    public function getNamespace(): ?string
    {
        return $this->namespace;
    }

    /**
     * namespace setter
     *
     * @param string|null $namespace
     * @return TableComment
     */
    public function setNamespace(?string $namespace): self
    {
        $this->namespace = $namespace;
        return $this;
    }

    /**
     * namespace var mı
     *
     * @return bool
     */
    public function isNamespace(): bool
    {
        return !empty($this->namespace);
    }

    /**
     * interfaces getter
     *
     * @return StringList
     */
    public function getInterfaces(): StringList
    {
        return $this->interfaces;
    }

    /**
     * interfaces setter
     *
     * @param StringList $interfaces
     * @return TableComment
     */
    public function setInterfaces(StringList $interfaces): self
    {
        $this->interfaces = $interfaces->distinct()->toTypedClass(StringList::class);
        return $this;
    }

    /**
     * interface tanımlı mı
     *
     * @return bool
     */
    public function isInterfaces(): bool
    {
        return $this->interfaces->any();
    }


    /**
     * uses getter
     *
     * @return StringList
     */
    public function getUses(): StringList
    {
        return $this->uses;
    }

    /**
     * uses setter
     *
     * @param StringList $uses
     * @return $this
     */
    public function setUses(StringList $uses)
    {
        $this->uses = $uses->distinct()->toTypedClass(StringList::class);
        return $this;
    }

    /**
     * uses tanımlı mı
     *
     * @return bool
     */
    public function isUses(): bool
    {
        return $this->uses->any();
    }


    /**
     * table name getter
     *
     * @return string|null
     */
    public function getTableName(): ?string
    {
        return $this->tableName;
    }

    /**
     * table name setter
     *
     * @param string|null $tableName
     * @return TableComment
     */
    public function setTableName(?string $tableName): self
    {
        $this->tableName = $tableName;
        return $this;
    }

    /**
     * table name tanımlı mı
     *
     * @return bool
     */
    public function isTableName(): bool
    {
        return !empty($this->tableName);
    }

    /**
     * docComment getter
     *
     * @return DocComment|null
     */
    public function getDocComment(): ?DocComment
    {
        return $this->docComment;
    }

    /**
     * docComment setter
     *
     * @param DocComment|null $docComment
     * @return TableComment
     */
    public function setDocComment(?DocComment $docComment): self
    {
        $this->docComment = $docComment;
        return $this;
    }

    /**
     * docComment tanımlı mı
     *
     * @return bool
     */
    public function isDocComment(): bool
    {
        return !empty($this->docComment);
    }

    #endregion props

    #region Methods

    /**
     * use ekler
     *
     * @param string $use
     */
    public function addUse(string $use)
    {
        if (!$this->uses->contains($use)) {
            $this->uses->add($use);
        }
    }

    /**
     * Yeni bir interface ekler
     *
     * @param string $interface
     */
    public function addInterface(string $interface):self
    {
        if (!$this->interfaces->contains($interface)) {
            $this->interfaces->add($interface);
        }
        return $this;
    }

    public function __get($name)
    {
        $result = null;
        $method = 'get'.ucwords($name);
        if (method_exists($this, $method))
        {
            $result = $this->$method();
        }
        return $result;
    }

    public function __set($name, $value)
    {
        $method = 'set'.ucwords($name);
        if (method_exists($this, $method))
        {
            $this->$method($value);
            return $this;
        }
        $this->$name = $value;
        return $this;
    }

    #endregion Methods

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = [];
        if($this->namespace !== null){
            $data['ns'] = str_replace('\\','.',$this->namespace);
        }
        if($this->entityName !== null){
            $data['en'] = $this->entityName;
        }
        if($this->baseClass !== null){
            $data['bc'] = str_replace('\\','.',$this->baseClass);
        }

        if($this->interfaces !== null && $this->interfaces->any()){
            $data['infs'] = $this->interfaces->select(function ($itm){
                return str_replace('\\','.',$itm);
            })->toArray();
        }
        if($this->uses !== null && $this->uses->any()){
            $data['uss'] = $this->uses->select(function ($itm){
                return str_replace('\\','.',$itm);
            })->toArray();
        }
        if($this->docComment !== null && !$this->docComment->isEmpty()){
            $data['dc'] = $this->docComment;
        }

        if($this->tableName !== null){
            $data['tn'] = $this->tableName;
        }
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->namespace = isset($data['ns']) ? $data['ns'] : null;
        $this->entityName = isset($data['en']) ? $data['en'] : null;
        $this->baseClass = isset($data['bc']) ? $data['bc'] : null;
        $this->interfaces = isset($data['infs']) ? new StringList(...$data['infs']) : new StringList();
        $this->uses = isset($data['uss']) ? new StringList(...$data['uss']) : new StringList();
        $this->docComment = isset($data['dc']) ? $data['dc'] : null;
        $this->tableName = isset($data['tn']) ? $data['tn'] : null;

        $this->interfaces = $this->interfaces->select(function ($itm){
            return str_replace('.','\\',$itm);
        })->toTypedClass(StringList::class);
        $this->uses = $this->uses->select(function ($itm){
            return str_replace('.','\\',$itm);
        })->toTypedClass(StringList::class);

        if(!empty($this->namespace)){
            $this->namespace = str_replace('.','\\',$this->namespace);
        }
        if(!empty($this->baseClass)){
            $this->baseClass = str_replace('.','\\',$this->baseClass);
        }
    }

    #endregion Serializable

}
