<?php


namespace GekTools\Tools;


use CodeIgniter\Database\BaseBuilder;
use CodeIgniter\Database\BaseConnection;
use CodeIgniter\Database\BaseResult;
use CodeIgniter\Database\ConnectionInterface;
use CodeIgniter\Database\Exceptions\DataException;
use CodeIgniter\Entity;
use CodeIgniter\Exceptions\ModelException;
use CodeIgniter\Model;
use CodeIgniter\Validation\ValidationInterface;
use CodeIgniter\I18n\Time;
use GekTools\Tools\Collections\PagedResult;
use GekTools\Tools\Contrats\ICreatedAtUtc;
use GekTools\Tools\Contrats\IDisplayOrder;
use GekTools\Tools\Contrats\ISoftDeletable;
use GekTools\Tools\Contrats\IUpdatedAtUtc;
use GekTools\Tools\Database\IBaseBuilderMixinFix;

/**
 * Class BaseModel
 * @package GekTools\Tools
 * @mixin   BaseBuilder
 * @mixin   IBaseBuilderMixinFix
 */
class BaseModel extends Model
{

    #region fields

    /**
     * The column used for insert timestamps
     *
     * @var string
     */
    protected $createdField = 'createdAtUtc';

    /**
     * The column used for update timestamps
     *
     * @var string
     */
    protected $updatedField = 'updatedAtUtc';

    /**
     * The column used to save soft delete state
     *
     * @var string
     */
    protected $deletedField = 'deletedAtUtc';

    /**
     * @var string
     */
    protected $displayOrderField = 'displayOrder';

    /**
     * Query Builder object
     *
     * @var BaseBuilder|IBaseBuilderMixinFix
     */
    protected $builder;

    /**
     * If true, will set created_at, and updated_at
     * values during insert and update routines.
     *
     * @var boolean
     */
    protected $useTimestamps = true;

    protected $useUtc = true;

    #endregion fields

    #region ctor

    /**
     * BaseModel constructor.
     * @param ConnectionInterface|null $db
     * @param ValidationInterface|null $validation
     */
    public function __construct(ConnectionInterface &$db = null, ValidationInterface $validation = null)
    {
        parent::__construct($db, $validation);
    }

    #endregion ctor

    #region methods

    /**
     * @param string $event
     * @param callable $callback
     * @return bool
     */
    public function on(string $event, callable $callback): bool
    {
        if (!isset($this->$event)) {
            $this->$event = [];
        }
        if (!in_array($callback, $this->$event)) {
            array_push($this->$event, $callback);
            return true;
        }
        return false;
    }

    /**
     * @param callable $callback
     * @return bool
     */
    public function onBeforeInsert(callable $callback): bool
    {
        return $this->on('beforeInsert', $callback);
    }

    /**
     * @param callable $callback
     * @return bool
     */
    public function onAfterInsert(callable $callback): bool
    {
        return $this->on('afterInsert', $callback);
    }

    /**
     * @param callable $callback
     * @return bool
     */
    public function onBeforeUpdate(callable $callback): bool
    {
        return $this->on('beforeUpdate', $callback);
    }

    /**
     * @param callable $callback
     * @return bool
     */
    public function onAfterUpdate(callable $callback): bool
    {
        return $this->on('afterUpdate', $callback);
    }


    /**
     * @param callable $callback
     * @return bool
     */
    public function onAfterFind(callable $callback): bool
    {
        return $this->on('afterFind', $callback);
    }

    /**
     * @param callable $callback
     * @return bool
     */
    public function onAfterDelete(callable $callback): bool
    {
        return $this->on('afterDelete', $callback);
    }

    /**
     * @param callable $callback
     * @return bool
     */
    public function onBeforeDelete(callable $callback): bool
    {
        return $this->on('beforeDelete', $callback);
    }

    /**
     * Fetches the row of database from $this->table with a primary key
     * matching $id.
     *
     * @param mixed|array|null $id One primary key or an array of primary keys
     *
     * @return array|object|null    The resulting row of data, or null.
     */
    public function find($id = null)
    {
        if (is_array($id)) {
            $this->handleDisplayOrder();
        }
        return parent::find($id);
    }

    /**
     * Works with the current Query Builder instance to return
     * all results, while optionally limiting them.
     *
     * @param integer $limit
     * @param integer $offset
     *
     * @return array
     */
    public function findAll(int $limit = 0, int $offset = 0)
    {
        $this->handleDisplayOrder();
        return parent::findAll($limit, $offset);
    }

    /**
     * A convenience method that will attempt to determine whether the
     * data should be inserted or updated. Will work with either
     * an array or object. When using with custom class objects,
     * you must ensure that the class will provide access to the class
     * variables, even if through a magic method.
     *
     * @param BaseEntity|Entity|array|object $data
     *
     * @return boolean
     * @throws \ReflectionException
     */
    public function save($data): bool
    {
        if (empty($data)) {
            return true;
        }

        if (is_object($data) && isset($data->{$this->primaryKey}) && $data->{$this->primaryKey} > 0) {
            $response = $this->update($data->{$this->primaryKey}, $data);
        } elseif (is_array($data) && !empty($data[$this->primaryKey]) && $data[$this->primaryKey] > 0) {
            $response = $this->update($data[$this->primaryKey], $data);
        } else {
            $response = $this->insert($data, false);

            if ($response instanceof BaseResult) {

                $response = $response->resultID !== false;
            } elseif ($response !== false) {

                $response = true;
            }
        }

        if ($response && ($data instanceof Entity) && empty($data->{$this->primaryKey})) {
            $data->{$this->primaryKey} = $this->insertID;
        }

        return $response;
    }

    /**
     * Inserts data into the current table. If an object is provided,
     * it will attempt to convert it to an array.
     *
     * @param array|object $data
     * @param boolean $returnID Whether insert ID should be returned or not.
     *
     * @return BaseResult|integer|string|false
     * @throws \ReflectionException
     */
    public function insert($data = null, bool $returnID = true)
    {
        $result = parent::insert($data, $returnID);
        if ($result && ($data instanceof Entity)) {
            $data->{$this->primaryKey} = $this->insertID;
        }
        if ($result && ($data instanceof ICreatedAtUtc)) {
            $data->{$this->createdField} = Time::now('UTC');
        }
        return $result;
    }

    /**
     * Updates a single record in $this->table. If an object is provided,
     * it will attempt to convert it into an array.
     *
     * @param integer|array|string $id
     * @param array|object $data
     *
     * @return boolean
     * @throws \ReflectionException
     */
    public function update($id = null, $data = null): bool
    {
        $result = parent::update($id, $data);
        if ($result && ($data instanceof IUpdatedAtUtc)) {
            $data->{$this->updatedField} = Time::now('UTC');
        }
        return $result;
    }

    /**
     * @param int $id
     * @return array|object|BaseEntity|null
     */
    public function getById(int $id)
    {
        return $this->find($id);
    }

    /**
     * @param array $ids
     * @return array|object[]|array[]|BaseEntity[]
     */
    public function getByIds(array $ids)
    {
        return $this->find($ids);
    }

    /**
     * @param int $pageIndex
     * @param int $pageSize
     * @return PagedResult
     */
    public function findAllPaged(int $pageIndex = 0, int $pageSize = PHP_INT_MAX): PagedResult
    {
        $builder = $this->builder();
        if ($this->tempUseSoftDeletes === true) {
            $builder->where($this->table . '.' . $this->deletedField, null);
        }
        $this->handleDisplayOrder();
        $totalCount = $builder->countAllResults(false);
        if ($pageSize < PHP_INT_MAX) {
            $offset = ($pageIndex * $pageSize);
            $builder->limit($pageSize, $offset);
        } else {
            $offset = 0;
        }
        $row = $builder->get();

        $row = $row->getResult($this->tempReturnType);

        $eventData = $this->trigger('afterFind', ['data' => $row, 'limit' => $pageSize, 'offset' => $offset]);

        $this->tempReturnType = $this->returnType;
        $this->tempUseSoftDeletes = $this->useSoftDeletes;
        $pagedData = new PagedResult($eventData['data'], $pageIndex, $pageSize, $totalCount);
        if (empty($builder->QBOrderBy) && is_array($builder->QBOrderBy)) {
            if (!empty($builder->QBOrderBy[0]['field'])) {
                $pagedData->setOrderedField($builder->QBOrderBy[0]['field']);
            }
            if (!empty($builder->QBOrderBy[0]['direction'])) {
                $pagedData->setOrderedDirection($builder->QBOrderBy[0]['direction']);
            }
        }
        return $pagedData;
    }

    /**
     * @param int $pageIndex
     * @param int $pageSize
     * @return PagedResult
     */
    public function getAll(int $pageIndex = 0, int $pageSize = PHP_INT_MAX): PagedResult
    {
        return $this->findAllPaged($pageIndex, $pageSize);
    }

    /**
     * @param int $id
     * @return mixed
     */
    public function deleteById(int $id)
    {
        return $this->delete($id);
    }

    /**
     * @param BaseEntity $entity
     * @return bool
     * @throws \Exception
     */
    public function deleteEntity(BaseEntity $entity): bool
    {
        if ($this->deleteById($entity->getPrimaryKeyValue())) {
            if ($entity instanceof ISoftDeletable) {
                $entity->{$this->deletedField} = Time::now('UTC');
            }
            return true;
        }
        return false;
    }

    #endregion methods

    #region Utils

    protected function handleDisplayOrder()
    {
        $builder = $this->builder();
        if (empty($builder->QBOrderBy)) {
            if (class_exists($this->tempReturnType) && is_a($this->tempReturnType, IDisplayOrder::class, true)) {
                $builder->orderBy($this->displayOrderField, 'asc');
            }
        }
    }

    /**
     * A utility function to allow child models to use the type of
     * date/time format that they prefer. This is primarily used for
     * setting created_at, updated_at and deleted_at values, but can be
     * used by inheriting classes.
     *
     * The available time formats are:
     *  - 'int'      - Stores the date as an integer timestamp
     *  - 'datetime' - Stores the data in the SQL datetime format
     *  - 'date'     - Stores the date (only) in the SQL date format.
     *
     * @param integer $userData An optional PHP timestamp to be converted.
     *
     * @return mixed
     * @throws \CodeIgniter\Exceptions\ModelException;
     */
    protected function setDate(int $userData = null)
    {
        $currentDate = is_numeric($userData) ? (int)$userData : time();

        switch ($this->dateFormat) {
            case 'int':
                return $currentDate;
            case 'datetime':
                return $this->useUtc ? gmdate('Y-m-d H:i:s', $currentDate) : date('Y-m-d H:i:s', $currentDate);
            case 'date':
                return $this->useUtc ? gmdate('Y-m-d', $currentDate) : date('Y-m-d', $currentDate);
            default:
                throw ModelException::forNoDateFormat(get_class($this));
        }
    }

    /**
     * A simple event trigger for Model Events that allows additional
     * data manipulation within the model. Specifically intended for
     * usage by child models this can be used to format data,
     * save/load related classes, etc.
     *
     * It is the responsibility of the callback methods to return
     * the data itself.
     *
     * Each $eventData array MUST have a 'data' key with the relevant
     * data for callback methods (like an array of key/value pairs to insert
     * or update, an array of results, etc)
     *
     * If callbacks are not allowed then returns $eventData immediately.
     *
     * @param string $event
     * @param array $eventData
     *
     * @return mixed
     * @throws \CodeIgniter\Database\Exceptions\DataException
     */
    protected function trigger(string $event, array $eventData)
    {
        $allowed = $this->tempAllowCallbacks;
        $this->tempAllowCallbacks = $this->allowCallbacks;

        if (!$allowed) {
            return $eventData;
        }

        // Ensure it's a valid event
        if (!isset($this->{$event}) || empty($this->{$event})) {
            return $eventData;
        }

        foreach ($this->{$event} as $callback) {
            if (is_callable($callback)) {
                $eventData = call_user_func_array($callback, [$eventData]);
            } elseif (method_exists($this, $callback)) {
                $eventData = $this->{$callback}($eventData);
            } else {
                throw DataException::forInvalidMethodTriggered($callback);
            }
        }

        return $eventData;
    }

    #endregion Utils

    #region statics

    /**
     * Takes a class an returns an array of it's public and protected
     * properties as an array suitable for use in creates and updates.
     *
     * @param string|object $data
     * @param string|null $primaryKey
     * @param string $dateFormat
     * @param boolean $onlyChanged
     *
     * @return array
     * @throws \ReflectionException
     */
    public static function classToArray($data, $primaryKey = null, string $dateFormat = 'datetime', bool $onlyChanged = true): array
    {
        return parent::classToArray($data, $primaryKey, $dateFormat, $onlyChanged);
    }

    /**
     * @param bool $getShared
     * @param ConnectionInterface|null $conn
     * @return static
     */
    public static function instance(bool $getShared = true, ConnectionInterface &$conn = null): self
    {
        return model(static::class, $getShared, $conn);
    }

    #endregion statics

}
