<?php


namespace GekTools\Tasks;


use CodeIgniter\CLI\CLI;
use CodeIgniter\I18n\Time;
use Gek\Collections\ArrayList;
use GekTools\Tasks\Entities\ScheduleTask;
use GekTools\Tasks\Models\ScheduleTasksModel;

class TaskService
{
    #region fields

    /**
     * @var ScheduleTasksModel
     */
    protected ScheduleTasksModel $model;

    #endregion fields

    #region ctor

    public function __construct()
    {
        $this->model = model(ScheduleTasksModel::class);
    }

    #endregion ctor

    #region properties

    /**
     * @return ScheduleTasksModel
     */
    public function getModel(): ScheduleTasksModel
    {
        return $this->model;
    }

    #endregion properties

    #region methods

    public function runTasks(){
        /** @var ArrayList|ScheduleTask[] $tasks */
        $tasks = $this->getModel()
            ->getAll()
            ->where(function (ScheduleTask $task) {
                return $task->check();
            })->toArrayList();
        foreach ($tasks as $task){
            $this->executeScheduleTask($task);
            $this->getModel()->save($task);
        }
    }

    /**
     * @param ScheduleTask $task
     * @param bool $throwException
     * @return bool
     * @throws \Throwable
     */
    public function executeScheduleTask(ScheduleTask $task, bool $throwException = false):bool
    {
        $res = false;

        $task->setLastStartUtc(Time::now('UTC'));
        try{
            $className = $task->getClassName();
            if(class_exists($className)){
                $tsk = new $className();
                if($tsk instanceof ITask){
                    $res = static::executeTask($tsk,$throwException);
                }else{
                    log_message('error', $className . 'geçerli bir '. ITask::class .' nesnesi değil!');
                }
            }else{
                log_message('error', $className . ' bulunamadı!');
            }
        }catch (\Throwable $exp){
            $res = false;
            log_message('error', '[ERROR] {exception}', ['exception' => $exp]);
            CLI::error($exp->getMessage());
            if ($throwException) {
                throw $exp;
            }
        }
        $end = Time::now('UTC');
        $task->setLastEndUtc($end);
        if(!$res){
            if($task->getStopOnError()){
                $task->setEnabled(false);
            }
        }else{
            $task->setLastSuccessUtc($end);
        }
        return  $res;
    }

    #endregion methods

    #region statics

    /**
     * @param ITask $task
     * @param bool $throwException
     * @return bool
     * @throws \Throwable
     */
    public static function executeTask(ITask $task, bool $throwException = false): bool
    {
        try {
            $task->execute();
            CLI::write(get_class($task) . " başarıyla çalıştırıldı.");
            return true;
        } catch (\Throwable $exp) {
            log_message('error', '[ERROR] {exception}', ['exception' => $exp]);
            CLI::error($exp->getMessage());
            if ($throwException) {
                throw $exp;
            }
            return false;
        }
    }

    /**
     * @return \GekTools\Tools\Tasks\TaskService
     */
    public static function instance():self {
        return service('taskService',true);
    }

    #endregion statics

}
