<?php


namespace GekTools\Settings;


use Gek\Collections\Enumerable;
use Gek\Infrastructure\ClassHelper;
use Gek\Infrastructure\Enum;
use Gek\Infrastructure\EnumType;
use Gek\Infrastructure\FlagEnum;
use Gek\Infrastructure\FlagEnumType;
use GekTools\Settings\Entities\Setting;
use GekTools\Settings\Models\SettingsModel;
use GekTools\Tools\Cache\DataCache;
use GekTools\Tools\Cache\Ttl;
use GekTools\Tools\DocComments\SettingsTags;
use GekTools\Tools\DocComments\StandardTags;

class SettingsService
{
    #region fields

    /**
     * @var SettingsModel
     */
    protected SettingsModel $settingsModel;

    /**
     * @var DataCache
     */
    protected DataCache $dataCache;

    #endregion fields


    #region ctor

    public function __construct()
    {
        $this->settingsModel = model(SettingsModel::class);
        $this->dataCache = DataCache::instance();
    }

    #endregion ctor

    #region Properties

    /**
     * @return SettingsModel
     */
    public function getModel(): SettingsModel
    {
        return $this->settingsModel;
    }

    #endregion Properties

    #region methods

    /**
     * @param string $name
     * @return Setting|array|object|null
     */
    public function getSettingByName(string $name)
    {
        $fldInf = Setting::getFieldsInfo();
        $cacheKey = Setting::class;
        $cacheKey = str_replace('\\','.',$cacheKey);
        $cacheKey .= '_' . $fldInf->name . '-'.$name;

        return $this->dataCache->getOrSave(
            $cacheKey,
            [$this,'getSettingByNameNotCached'],
            [$name],
            Ttl::fromDays(3)
        );
    }

    /**
     * @param string $name
     * @return Setting|array|object|null
     */
    public function getSettingByNameNotCached(string $name)
    {
        $fldInf = Setting::getFieldsInfo();
        return $this->getModel()
            ->where($fldInf->name, $name)
            ->first();
    }

    /**
     * @param string $name
     * @return bool|float|int|mixed|null
     */
    public function getValue(string $name)
    {
        $setting = $this->getSettingByName($name);

        if (empty($setting)) {
            return null;
        }
        return $setting->getRealValue();
    }

    /**
     * @param string $name
     * @param $value
     * @param string|null $type
     * @param bool|null $nullable
     * @return bool|float|int|mixed|null
     * @throws \ReflectionException
     */
    public function setValue(string $name, $value, ?string $type = null, ?bool $nullable = null)
    {
        $setting = $this->getSettingByName($name);
        $cType = $type;
        $cNullable = $nullable;
        if (!empty($setting)) {
            if (empty($cType)) {
                $cType = $setting->getType();
            }
            if ($cNullable === null) {
                $cNullable = $setting->getNullable();
            }
            if (empty($type) && $value !== null) {
                $detectedType = $this->getTypeFromValue($value);
                if ($detectedType != $cType) {
                    $cType = $detectedType;
                }
            }
        } else {
            if(empty($cType)){
                $nlb = true;
                $cType = $this->getTypeFromValue($value,$nlb);
                if($cNullable === null){
                    $cNullable = $nlb;
                }
            }
            if($cNullable === null){
                $cNullable = true;
            }

            $setting = new Setting();
            $setting->setName($name);
        }
        $setting->setType($cType)
            ->setNullable($cNullable)
            ->setRealValue($value);

        $res = $this->getModel()->save($setting);
        $fldInf = Setting::getFieldsInfo();
        $cacheKey = Setting::class;
        $cacheKey .= '_' . $fldInf->name . '-'.$name;
        $cacheKey = str_replace('\\','.',$cacheKey);
        $this->dataCache->delete($cacheKey);
        return $res;
    }

    /**
     * @param mixed $value
     * @param string $type
     * @param bool $nullable
     * @return string|null
     * @throws \Exception
     */
    public function valueToString($value, string $type, bool $nullable): ?string
    {
        if ($value === null && $nullable) {
            return null;
        }
        if ($value === null) {
            throw new \Exception('value null olamaz.');
        }
        switch (true) {
            case $type === 'string':
                return $value;
                break;
            case $type === 'int':
            case $type === 'float':
            case is_subclass_of($type, EnumType::class):
            case is_subclass_of($type, FlagEnumType::class):
                return strval($value);
                break;
            case $type === 'bool':
                return $value ? '1' : '0';
                break;

            case $type === 'array':
            case $type === 'object':
            default:
                return serialize($value);
                break;
        }
    }

    /**
     * @param string|null $strValue
     * @param string $type
     * @param bool $nullable
     * @return bool|float|int|mixed|null
     */
    public function stringToValue(?string $strValue, string $type, bool $nullable)
    {
        if ($strValue === null && $nullable) {
            return null;
        }
        switch (true) {
            case $type === 'string':
                return $strValue;
                break;
            case $type === 'float':
                return floatval($strValue);
                break;
            case $type === 'bool':
                return boolval($strValue);
                break;
            case is_subclass_of($type, EnumType::class):
            case is_subclass_of($type, FlagEnumType::class):
                return new $type($strValue);
            case $type === 'int':
            case is_subclass_of($type, Enum::class):
            case is_subclass_of($type, FlagEnum::class):
                return intval($strValue);
                break;
            case $type === 'array':
            case $type === 'object':
            default:
                return unserialize($strValue);
                break;
        }
    }

    /**
     * @param $value
     * @param bool $nullable
     * @return string
     */
    public function getTypeFromValue($value, bool &$nullable = false)
    {
        $default = 'string';
        if ($value === null) {
            $nullable = true;
            return $default;
        }
        if (is_float($value)) {
            return 'float';
        }
        if (is_int($value)) {
            return 'int';
        }
        if (is_string($value)) {
            return 'string';
        }
        if (is_array($value)) {
            return 'array';
        }
        if (is_object($value) && !($value instanceof \stdClass)) {
            return get_class($value);
        }
        if (is_object($value)) {
            return 'object';
        }
        return $default;
    }

    /**
     * @param ISettings $settings
     * @return bool|float|int|mixed|null
     * @throws \ReflectionException
     */
    public function setSettings(ISettings $settings){
        $className = get_class($settings);
        $reflect = ClassHelper::reflectClass($className);
        $props = $reflect->getProperties();
        $res = null;
        foreach ($props as $prp){
            if($prp->isStatic()){
                continue;
            }
            if(!$prp->getDocComment()){
                $docCom = ClassHelper::getDocCommentObject($prp);
                if($docCom->getKey(SettingsTags::IGNORE)){
                    continue;
                }
            }

            /** @var ReflectionNamedType $refType */
            $refType = $prp->getType();
            $vType = null;
            $vNullable = null;
            if(!empty($refType)){
                $vType = $refType->getName();
                $vNullable = $refType->allowsNull();
            }elseif (isset($docCom) && is_string($docCom->getKey(StandardTags::VAR)) ){
                $types = $docCom->getKey(StandardTags::VAR);
                $types = explode('|',$types);
                $types = Enumerable::fromArray($types)
                    ->where(function ($itm){
                        return !empty(trim($itm));
                    })
                    ->select(function ($itm){
                        return strtolower(trim($itm));
                    })
                    ->toArrayList();
                if($types->contains('null')){
                    $vNullable = true;
                    $types->remove('null');
                }
                if($types->count() === 1){
                    $vType = $types->firstOrNull();
                }
            }
            $vName = $className . '-' . $prp->getName();
            $vName = str_replace('\\','.',$vName);
            if(!$prp->isPublic()){
                $prp->setAccessible(true);
            }
            $vValue = $prp->getValue($settings);
            $itmRes = $this->setValue($vName,$vValue,$vType,$vNullable);
            if($res !== false ){
                $res = $itmRes;
            }
        }
        $cacheKey = 'ISettings_';
        $cacheKey .= $className;
        $cacheKey = str_replace('\\','.',$cacheKey);
        $this->dataCache->delete($cacheKey);
        return $res === null ? false : $res;
    }

    /**
     * @param ISettings|string $settingsClass
     * @return ISettings|mixed
     * @throws \ReflectionException
     */
    public function getSettings($settingsClass):ISettings{
        $className = is_object($settingsClass) ? get_class($settingsClass) : $settingsClass;
        $cacheKey = 'ISettings_';
        $cacheKey .= $className;
        $cacheKey = str_replace('\\','.',$cacheKey);

        return $this->dataCache->getOrSave(
            $cacheKey,
            [$this,'getSettingsNoCache'],
            [$settingsClass],
            Ttl::fromWeeks(2)
        );


    }

    /**
     * @param ISettings|string $settingsClass
     * @return ISettings
     * @throws \ReflectionException
     */
    public function getSettingsNoCache($settingsClass):ISettings{
        if(is_object($settingsClass)){
            $settingsClass = get_class($settingsClass);
        }
        if(!is_a($settingsClass,ISettings::class,true)){
            throw new \Exception($settingsClass . 'Geçerli bir ' . ISettings::class . " sınıfı değil.");
        }
        $nameStart = $settingsClass . '-';
        $nameStart = str_replace('\\','.',$nameStart);
        $fldInf = Setting::getFieldsInfo();
        $values = $this->getModel()
            ->like($fldInf->name,$nameStart,'after')
            ->getAll();
        $result = new $settingsClass();

        $reflect = ClassHelper::reflectClass($settingsClass);
        $props = $reflect->getProperties();
        foreach ($props as $prp){
            if($prp->isStatic()){
                continue;
            }
            if(!$prp->getDocComment()){
                $docCom = ClassHelper::getDocCommentObject($prp);
                if($docCom->getKey(SettingsTags::IGNORE)){
                    continue;
                }
            }
            $nameForSetting = $nameStart . $prp->getName();
            /** @var Setting $curSetting */
            $curSetting = $values->firstOrNull(function (Setting $itm) use($nameForSetting){
                return $itm->getName() == $nameForSetting;
            });
            if($curSetting === null){
                continue;
            }

            $val = $curSetting->getRealValue();
            if(!$prp->isPublic()){
                $prp->setAccessible(true);
            }
            $prp->setValue($result,$val);
        }
        return $result;
    }

    #endregion methods

    #region static

    public static function instance():self {
        return service('settingsService');
    }

    #endregion static
}
