<?php


namespace GekTools\Membership;


use CodeIgniter\I18n\Time;
use Config\Services;
use GekTools\Membership\Entities\User;
use GekTools\Membership\Entities\UserRole;
use GekTools\Membership\Entities\UserRoleMap;
use GekTools\Membership\Models\AuthsModel;
use GekTools\Membership\Models\UserRoleMappingsModel;
use GekTools\Membership\Models\UserRolesModel;
use GekTools\Membership\Models\UsersModel;
use GekTools\Membership\Settings\MembershipSettings;
use GekTools\Settings\SettingsService;
use GekTools\Tools\Cache\DataCache;
use GekTools\Tools\Cache\Ttl;
use GekTools\Tools\Guid;

class UserService
{

    #region fields

    /**
     * @var UsersModel
     */
    protected UsersModel $userModel;

    /**
     * @var UserRolesModel
     */
    protected UserRolesModel $userRolesModel;

    /**
     * @var UserRoleMappingsModel
     */
    protected UserRoleMappingsModel $userRoleMappingsModel;

    /**
     * @var MembershipSettings
     */
    protected MembershipSettings $settings;


    /**
     * @var DataCache
     */
    protected DataCache $dataCache;

    #endregion fields

    #region ctor

    public function __construct()
    {
        $this->userModel = model(UsersModel::class);
        $this->userRolesModel = model(UserRolesModel::class);
        $this->userRoleMappingsModel = model(UserRoleMappingsModel::class);
        $this->settings = SettingsService::instance()->getSettings(MembershipSettings::class);
        $this->dataCache = DataCache::instance();

        $this->userModel->onBeforeUpdate(function ($eventData){
            $fi = User::getFieldsInfo();
            $cacheKey = DataCache::createKey(User::class,[$fi->userGuid => $eventData['data'][$fi->userGuid]]);
            DataCache::instance()->delete($cacheKey);
            return $eventData;
        });
        $this->userModel->onBeforeDelete(function ($eventData){
            $id = $eventData['id'];
            if(is_array($id)){
                $id = $id[0];
            }
            if($id){
                $mdl = new UsersModel();
                /** @var User $user */
                $user = $mdl->getById($id);
                if($user){
                    $fi = User::getFieldsInfo();
                    $cacheKey = DataCache::createKey(User::class,[$fi->userGuid => $user->getUserGuid() ]);
                    DataCache::instance()->delete($cacheKey);
                }
            }

            return $eventData;
        });
    }

    #endregion ctor

    #region properties

    /**
     * @return UsersModel|mixed
     */
    public function getUserModel()
    {
        return $this->userModel;
    }

    /**
     * @return UserRolesModel
     */
    public function getUserRolesModel(): UserRolesModel
    {
        return $this->userRolesModel;
    }

    /**
     * @return UserRoleMappingsModel
     */
    public function getUserRoleMappingsModel(): UserRoleMappingsModel
    {
        return $this->userRoleMappingsModel;
    }

    /**
     * @return MembershipSettings
     */
    public function getSettings(): MembershipSettings
    {
        return $this->settings;
    }

    #endregion properties

    #region methods

    /**
     * @param string $userGuid
     * @return User|array|object|null
     */
    public function getUserByUserGuid(string $userGuid)
    {
        $fi = User::getFieldsInfo();
        $cacheKey = $this->dataCache::createKey(User::class,[$fi->userGuid => $userGuid]);

        return $this->dataCache->getOrSave(
            $cacheKey,
            [$this,'getUserByUserGuidNoCache'],
            [$userGuid],
            Ttl::fromMinutes(20)
        );
    }


    /**
     * @param string $userGuid
     * @return User|array|object|null
     */
    public function getUserByUserGuidNoCache(string $userGuid)
    {
        $fi = User::getFieldsInfo();
        return $this->getUserModel()
            ->where($fi->userGuid, $userGuid)
            ->first();
    }


    /**
     * @param string $email
     * @param bool $withDeleted
     * @return User|array|object|null
     */
    public function getUserByEmail(string $email, bool $withDeleted = false)
    {
        $fi = User::getFieldsInfo();
        return $this->getUserModel()
            ->withDeleted($withDeleted)
            ->where($fi->email, $email)
            ->first();
    }

    /**
     * @param string $userName
     * @param bool $withDeleted
     * @return User|array|object|null
     */
    public function getUserByUserName(string $userName, bool $withDeleted = false)
    {
        $fi = User::getFieldsInfo();
        return $this->getUserModel()
            ->withDeleted($withDeleted)
            ->where($fi->userName, $userName)
            ->first();
    }

    /**
     * @param string $phone
     * @param bool $withDeleted
     * @return User|array|object|null
     */
    public function getUserByPhone(string $phone, bool $withDeleted = false)
    {
        $fi = User::getFieldsInfo();
        return $this->getUserModel()
            ->withDeleted($withDeleted)
            ->where($fi->phone, $phone)
            ->first();
    }

    /**
     * @param string $userNameOrEmail
     * @return array|User|object|null
     */
    public function getUserForLogin(string $userNameOrEmail)
    {
        $this->getUserModel()->withDeleted();
        if ($this->settings->isUserNamesEnabled()) {
            $res =  $this->getUserByUserName($userNameOrEmail);
        }
        $res =  $this->getUserByEmail($userNameOrEmail);
        return $res;
    }

    /**
     * @param string $systemName
     * @return User|array|object|null
     */
    public function getUserBySystemName(string $systemName)
    {
        $fi = User::getFieldsInfo();
        return $this->getUserModel()
            ->where($fi->systemName, $systemName)
            ->first();
    }

    /**
     * @param string $userName
     * @param int|null $notId
     * @return bool
     */
    public function checkUserName(string $userName, ?int $notId = null): bool
    {
        $fi = User::getFieldsInfo();
        $query = $this->getUserModel()
            ->where($fi->userName, $userName);
        if ($notId !== null) {
            $query->whereNotIn($fi->id, [$notId]);
        }
        return $query->countAllResults() > 0;

    }

    /**
     * @param string $email
     * @param int|null $notId
     * @return bool
     */
    public function checkEmail(string $email, ?int $notId = null): bool
    {
        $fi = User::getFieldsInfo();
        $query = $this->getUserModel()
            ->where($fi->email, $email);
        if ($notId !== null) {
            $query->whereNotIn($fi->id, [$notId]);
        }
        return $query->countAllResults() > 0;

    }

    /**
     * @return User
     * @throws \ReflectionException
     */
    public function insertNewGuestUser()
    {
        $request = Services::request();
        $guest = (new User())
            ->setUserGuid(Guid::create())
            ->setLastActivityDataUtc(new Time('now', 'UTC'))
            ->setStatus(UserStatus::ACTIVE())
            ->setLastIp($request->getIPAddress());

        $this->getUserModel()->save($guest);
        $role = $this->getUserRoleBySystemName(SystemRoleNames::GUESTS);

        $this->addRoleOnUser($guest->getId(), $role->getId());
        return $guest;
    }

    /**
     * @param string $systemName
     * @return UserRole|array|object|null
     */
    public function getUserRoleBySystemName(string $systemName)
    {
        $fi = UserRole::getFieldsInfo();
        return $this->getUserRolesModel()
            ->where($fi->systemName, $systemName)
            ->first();
    }

    /**
     * @param int $userId
     * @param int $roleId
     * @return mixed
     */
    public function removeRoleOnUser(int $userId, int $roleId)
    {
        $fi = UserRoleMap::getFieldsInfo();
        return $this->getUserRoleMappingsModel()
            ->where($fi->userId, $userId)
            ->where($fi->roleId, $roleId)
            ->delete();
    }

    /**
     * @param int $userId
     * @param int $roleId
     * @return bool
     * @throws \ReflectionException
     */
    public function addRoleOnUser(int $userId, int $roleId)
    {
        $map = new UserRoleMap();
        $map->setUserId($userId)
            ->setRoleId($roleId);

        return $this->getUserRoleMappingsModel()->save($map);


    }

    /**
     *
     * @param int $userId
     * @return \GekTools\Tools\Collections\PagedResult|UserRole[]
     */
    public function getUserRolesByUserId(int $userId)
    {
        $fi = UserRoleMap::getFieldsInfo();
        $maps = $this->getUserRoleMappingsModel()
            ->where($fi->userId, $userId)
            ->getAll();
        $ids = $maps->select(function (UserRoleMap $map) {
            return $map->getRoleId();
        })->toArray();

        $fi2 = UserRole::getFieldsInfo();

        return $this->getUserRolesModel()
            ->whereIn($fi2->id, $ids)
            ->getAll();
    }

    #endregion methods

    #region static

    /**
     * @return static
     */
    public static function instance(): self
    {
        return service('userService', true);
    }

    #endregion static

}
