<?php


namespace Gek\AaApi;


class Benchmark
{
    /**
     * List of all timers.
     *
     * @var array
     */
    protected $timers = [];

    //--------------------------------------------------------------------

    /**
     * Starts a timer running.
     *
     * Multiple calls can be made to this method so that several
     * execution points can be measured.
     *
     * @param string $name The name of this timer.
     * @param float  $time Allows user to provide time.
     *
     * @return $this
     */
    public function start( $name,  $time = null)
    {
        $this->timers[strtolower($name)] = [
            'start' => ! empty($time) ? $time : microtime(true),
            'end'   => null,
        ];

        return $this;
    }

    //--------------------------------------------------------------------

    /**
     * Stops a running timer.
     *
     * If the timer is not stopped before the timers() method is called,
     * it will be automatically stopped at that point.
     *
     * @param string $name The name of this timer.
     *
     * @return $this
     */
    public function stop($name)
    {
        $name = strtolower($name);

        if (empty($this->timers[$name]))
        {
            throw new RuntimeException('Cannot stop timer: invalid name given.');
        }

        $this->timers[$name]['end'] = microtime(true);

        return $this;
    }

    //--------------------------------------------------------------------

    /**
     * Returns the duration of a recorded timer.
     *
     * @param string  $name     The name of the timer.
     * @param integer $decimals Number of decimal places.
     *
     * @return null|float       Returns null if timer exists by that name.
     *                          Returns a float representing the number of
     *                          seconds elapsed while that timer was running.
     */
    public function getElapsedTime( $name, $decimals = 4)
    {
        $name = strtolower($name);

        if (empty($this->timers[$name]))
        {
            return null;
        }

        $timer = $this->timers[$name];

        if (empty($timer['end']))
        {
            $timer['end'] = microtime(true);
        }

        return (float) number_format($timer['end'] - $timer['start'], $decimals);
    }

    //--------------------------------------------------------------------

    /**
     * Returns the array of timers, with the duration pre-calculated for you.
     *
     * @param integer $decimals Number of decimal places
     *
     * @return array
     */
    public function getTimers($decimals = 4)
    {
        $timers = $this->timers;

        foreach ($timers as &$timer)
        {
            if (empty($timer['end']))
            {
                $timer['end'] = microtime(true);
            }

            $timer['duration'] = (float) number_format($timer['end'] - $timer['start'], $decimals);
        }

        return $timers;
    }

    //--------------------------------------------------------------------

    /**
     * Checks whether or not a timer with the specified name exists.
     *
     * @param string $name
     *
     * @return boolean
     */
    public function has($name)
    {
        return array_key_exists(strtolower($name), $this->timers);
    }

}