<?php


namespace Gek\PhpLang;


use Gek\Collections\Typed\StringList;
use Gek\Infrastructure\Events\EventHandler;
use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Events\PropertyChangedEventArgs;


class UseItem implements IToIndentedString, \Serializable
{


    /** Class or namespace import */
    public const TYPE_NORMAL = 1;
    /** Function import */
    public const TYPE_FUNCTION = 2;
    /** Constant import */
    public const TYPE_CONSTANT = 3;

    #region fields

    private StringList $segments;

    private string $name;

    private int $type;

    private ?string $aliasName = null;

    public EventHandler $onChange;

    #endregion fields

    #region ctor

    public function __construct(string $fullName, ?string $aliasName = null, int $type= self::TYPE_NORMAL )
    {
        $this->segments = new StringList();
        $this->init($fullName, $aliasName, $type);
        $this->onChange = new EventHandler(PropertyChangedEventArgs::class);
    }

    #endregion ctor

    #region properties

    public function getSegments(): StringList
    {
        return $this->segments;
    }

    /**
     * @param StringList $segments
     * @return self
     */
    public function setSegments(StringList $segments): self
    {
        $evtArg = new PropertyChangedEventArgs('aliasName',$this->segments ,$segments);
        $this->segments = $segments;
        $this->onChange->invoke($this,$evtArg);
        return $this;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     * @return self
     */
    public function setName(string $name): self
    {

        $evtArg = new PropertyChangedEventArgs('aliasName',$this->aliasName ,$aliasName);
        $this->aliasName = $aliasName;
        $this->onChange->invoke($this,$evtArg);
        $this->name = $name;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getAliasName(): ?string
    {
        return $this->aliasName;
    }

    /**
     * @param string|null $aliasName
     * @return self
     */
    public function setAliasName(?string $aliasName): self
    {
        $evtArg = new PropertyChangedEventArgs('aliasName',$this->aliasName ,$aliasName);
        $this->aliasName = $aliasName;
        $this->onChange->invoke($this,$evtArg);
        return $this;
    }

    #endregion properties

    #region method

    /**
     * @return bool
     */
    public function isAliasName(): bool
    {
        return !empty($this->aliasName);
    }

    /**
     * @return string
     */
    public function getFullName(): string
    {
        $strRes = '';
        if ($this->segments->any()) {
            $strRes .= implode("\\", $this->segments->toArray());
        }
        $strRes .= "\\" . $this->name;
        return $strRes;
    }

    public function __toString()
    {
        $strRes = 'use ';
        if ($this->segments->any()) {
            $strRes .= implode("\\", $this->segments->toArray());
        }
        $strRes .= "\\" . $this->name;
        if ($this->aliasName !== null) {
            $strRes .= " as " . $this->aliasName;
        }
        $strRes .= ";";

        return $strRes;
    }

    #endregion method

    #region utils

    /**
     * @param string $fullName
     * @param string|null $aliasName
     */
    private function init(string $fullName, ?string $aliasName = null,$type = self::TYPE_NORMAL)
    {
        $this->type = $type;
        $segments = explode("\\", $fullName);
        $name = array_pop($segments);
        $this->segments->clear();
        foreach ($segments as $seg) {
            if (!empty($seg)) {
                $this->segments->add($seg);
            }
        }
        $this->name = $name;
        $this->aliasName = $aliasName;
    }

    #endregion utils


    #region IToIndentedString

    /**
     * @param int $indentLevel girinti seviyesi
     * @param string $indentChars girinti karakterleri
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        $indent = str_repeat($indentChars, $indentLevel);
        $strRes = $indent . 'use ';
        if($this->type == self::TYPE_FUNCTION){
            $strRes .= "function ";
        }
        if($this->type == self::TYPE_CONSTANT){
            $strRes .= "const ";
        }
        if ($this->segments->any()) {
            $strRes .= implode("\\", $this->segments->toArray());
        }
        $strRes .= "\\" . $this->name;
        if ($this->aliasName !== null) {
            $strRes .= " as " . $this->aliasName;
        }
        $strRes .= ";";

        return $strRes;
    }

    #endregion IToIndentedString

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = [
            's' => $this->segments->toArray(),
            'n' => $this->name,
        ];
        if ($this->isAliasName()) {
            $data['a'] = $this->aliasName;
        }
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->segments = new StringList(...$data['s']);
        $this->name = $data['n'];
        $this->aliasName = isset($data['a']) ? $data['a'] : null;
    }

    #endregion Serializable
}
