<?php


namespace Gek\PhpLang;


use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\Traits\NameAware;
use Gek\PhpLang\Traits\ValueAware;
use Gek\Infrastructure\Str;

/**
 * MethodParam sınıfı
 *
 * Metod parametresi
 *
 * @package Gek\PhpLang
 */
class MethodParam implements IUseCreator, \Serializable
{
    use NameAware;
    use ValueAware;

    #region fields

    /**
     * Parametre tipi
     *
     * @var PhpTypeDeclared|null
     */
    protected ?PhpTypeDeclared $type = null;

    /**
     *  Variable-length argument list (Üç noktalı ön ek) kullan
     * @var bool
     */
    protected bool $dotted = false;

    /**
     * referans kullan
     * @var bool
     */
    protected bool $reference = false;

    #endregion fields

    #region ctor

    /**
     * MethodParam yapıcı metod.
     * @param string $name parametre adı
     * @param string|null $type parametre türü
     * @param bool $dotted "..." ön eki kullan
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function __construct(string $name, $type = null, bool $dotted = false, bool $reference = false)
    {
        $this->setName($name)
            ->setType($type)
            ->setDotted($dotted)
            ->setReference($reference);
    }

    #endregion ctor

    #region Properties

    /**
     * @return string|null
     */
    public function getType(): ?string
    {
        return $this->type !== null ?
            $this->type->getRaw() :
            null;
    }

    /**
     * @param PhpTypeDeclared|array|string|null $type
     * @return self
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function setType($type): self
    {
        if (is_array($type)) {
            $temp = array();
            foreach ($type as $tp) {
                if ($tp === null) {
                    $tp = 'null';
                }
                if (empty($tp)) {
                    continue;
                }
                $temp[] = $tp;
            }
            $type = implode('|', $temp);
        }
        if (is_string($type)) {
            $type = new PhpTypeDeclared($type);
        }
        $this->type = $type;
        return $this;
    }

    /**
     * @param bool $dotted
     * @return $this
     */
    public function setDotted(bool $dotted = true): self
    {
        $this->dotted = $dotted;
        return $this;
    }

    /**
     * @param bool $reference
     * @return $this
     */
    public function setReference(bool $reference = true):self {
        $this->reference = $reference;
        return $this;
    }

    #endregion Properties

    #region Methods

    /**
     * @return bool
     */
    public function hasType(): bool
    {
        return $this->type !== null;
    }

    /**
     * @return bool
     */
    public function isDotted(): bool
    {
        return $this->dotted;
    }

    /**
     * @return bool
     */
    public function isReference():bool{
        return $this->reference;
    }


    public function renderForComment(): string
    {
        $strRes = '';
        $type = $this->hasType() ? $this->type->renderForComment() : '';
        if (!empty($type)) {
            $strRes .= $type . ' ';
        }
        if ($this->dotted) {
            $strRes .= '...';
        }

        $strRes .= '$' . $this->getName();
        return $strRes;
    }

    public function __toString()
    {
        $strRes = '';
        $type = $this->hasType() ? $this->type->renderForParam() : '';
        if (!empty($type)) {
            $strRes .= $type . ' ';
        }
        if ($this->isReference()) {
            $strRes .= '&';
        }
        if ($this->isDotted()) {
            $strRes .= '...';
        }
        $strRes .= '$' . $this->getName();
        if ($this->isSetValue()) {
            $strRes .= ' = ';
            if ($this->getValueRenderType() === LiteralTypes::SINGLE_QUOTES) {
                $strRes .= Str::format("'{0}'", $this->getValue());
            } elseif ($this->getValueRenderType() === LiteralTypes::DOUBLE_QUOTES) {
                $strRes .= Str::format('"{0}"', $this->getValue());
            } else {
                $strRes .= $this->getValue();
            }
        }
        return $strRes;
    }

    #endregion Methods

    #region utils

    /**
     * @param string $params
     * @return array|MethodParam[]
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     * @throws \Exception
     */
    public static function parseParams(string $params): array
    {
        $resArray = array();
        $strParamArray = explode(',', $params);
        foreach ($strParamArray as $strParam) {
            $pType = static::parseType($strParam);
            $pName = static::parseName($strParam, $isDotted, $isReference);
            $pValue = static::parseValue($strParam);

            $mp = new MethodParam($pName, $pType);
            $mp->setDotted($isDotted)
                ->setReference($isReference);
            if ($pValue !== null) {

                $renderType = LiteralTypes::NONE;
                if (Str::startsWith($pValue, '"')) {
                    $pValue = trim($pValue, '"');
                    $renderType = LiteralTypes::DOUBLE_QUOTES;
                } elseif (Str::startsWith($pValue, '\'')) {
                    $pValue = trim($pValue, "'");
                    $renderType = LiteralTypes::SINGLE_QUOTES;
                }
                $mp->setValue($pValue, $renderType);
            }
            $resArray[] = $mp;
        }
        return $resArray;
    }

    /**
     * @param string $strParam
     * @return string|null
     * @throws \Exception
     */
    private static function parseType(string &$strParam): ?string
    {
        $strParam = trim($strParam);
        if (Str::startsWith($strParam, '$')) {
            return null;
        }
        $spacePos = strpos($strParam, ' ');

        if ($spacePos === false) {
            throw new \Exception('format hatalı.(parseType)');
        }
        $type = substr($strParam, 0, $spacePos);
        $strParam = substr($strParam, $spacePos);
        return $type;
    }

    /**
     * @param string $strParam
     * @return string|null
     * @throws \Exception
     */
    private static function parseName(string &$strParam, &$isDotted, &$isReference): ?string
    {
        $strParam = trim($strParam);
        $isReference = false;
        $isDotted = false;
        if (Str::startsWith($strParam, '&')) {
            $isReference = true;
            $strParam = trim(substr($strParam, 1));
        }
        if (Str::startsWith($strParam, '...')) {
            $isDotted = true;
            $strParam = trim(substr($strParam, 3));
        }
        if (!Str::startsWith($strParam, '$')) {
            throw new \Exception('format hatalı.(parseName)');
        }
        $assignmentPos = strpos($strParam, '=');
        if ($assignmentPos === false) {
            $pName = ltrim($strParam, '$');
            $strParam = '';
        } else {
            $pName = substr($strParam, 0, $assignmentPos);
            $pName = trim($pName);
            $pName = ltrim($pName, '$');
            $strParam = substr($strParam, $assignmentPos);
        }
        return $pName;

    }

    private static function parseValue(string &$strParam): ?string
    {
        $strParam = trim($strParam, " \t\n\r\0\x0B=");

        if ($strParam === '') {
            return null;
        }
        $pValue = trim($strParam);
        $strParam = '';
        return $pValue;

    }

    #endregion utils

    #region IUseCreator

    /**
     * @return array|UseItem[]
     */
    public function getUseArray(): array
    {
        if ($this->hasType()) {
            return $this->type->getUseArray();
        }
        return array();
    }

    #endregion IUseCreator

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = [
            'n' => $this->name,
        ];
        if($this->isReference()){
            $data['ref'] = $this->reference;
        }
        if($this->isDotted()){
            $data['d'] = $this->dotted;
        }
        if ($this->hasType()) {
            $data['t'] = $this->type;
        }
        if ($this->isSetValue()) {
            $data['vl'] = $this->value;
            $data['rt'] = $this->valueRenderType;
        }
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->name = $data['n'];
        $this->reference = isset($data['ref']) ? $data['ref'] : false;
        $this->dotted = isset($data['d']) ? $data['d'] : false;
        $this->type = isset($data['t']) ? $data['t'] : null;
        if(isset($data['vl'])){
            $this->value = $data['vl'];
        }
        if(isset($data['rt'])){
            $this->valueRenderType = $data['rt'];
        }

    }

    #endregion Serializable
}
