<?php


namespace Gek\PhpLang\Collections;


use Gek\Collections\ArrayList;
use Gek\Collections\TypedListWrapper;
use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\UseItem;
use phpDocumentor\Reflection\Types\This;

/**
 * UseItemCollection Sınıfı
 *
 * İmport edilen türler koleksiyonu
 *
 * @package Gek\PhpLang\Collections
 */
class UseItemCollection extends TypedListWrapper implements IToIndentedString, \Serializable
{

    #region ctor

    /**
     * UseItemCollection Yapıcı metod.
     * @param UseItem ...$items (opsiyonel) öğeler.
     */
    public function __construct(UseItem ...$items)
    {
        parent::__construct(new ArrayList(), UseItem::class);
        $this->addRange(...$items);
    }

    #endregion ctor

    #region methods

    /**
     * öğe ekler.
     * @param UseItem $item öğe
     */
    public function add(UseItem $item): void
    {
        $this->innerList->add($item);
    }

    /**
     * çoklu öğe ekler.
     * @param UseItem ...$items öğeler.
     */
    public function addRange(UseItem ...$items): void
    {
        $this->innerList->addRange($items);
    }


    /**
     * verilen öğenin koleksiyonda olup olmadığına bakar.
     * @param UseItem $item öğe
     * @return bool varsa true yoksa false
     */
    public function contains(UseItem $item): bool
    {
        return $this->innerList->contains($item);
    }

    /**
     * öğeyi koleksiyondan kaldırır.
     * @param UseItem $item öğe
     * @return bool başarı halinde true aksi halde false.
     */
    public function remove(UseItem $item): bool
    {
        return $this->innerList->remove($item);
    }

    /**
     * verilen öğenin koleksiyondaki indeksini verir.
     * @param UseItem $item öğe
     * @return int öğenin indeksi, öğe bulunamazsa -1
     */
    public function indexOf(UseItem $item): int
    {
        return $this->innerList->indexOf($item);
    }

    /**
     * verilen indekse öğeyi ekler.
     * @param int $index indeks
     * @param UseItem $item öğe
     */
    public function insert(int $index, UseItem $item): void
    {
        $this->innerList->insert($index, $item);
    }

    /**
     * verilen indeksteki öğeyi döndürür.
     * @param int $index indeks
     * @return UseItem öğe
     */
    public function getAt(int $index): UseItem
    {
        return $this->innerList[$index];
    }

    /**
     * verilen indekse öğeyi set eder.
     * @param int $index indeks
     * @param UseItem $item öğe
     */
    public function setAt(int $index, UseItem $item): void
    {
        $this->innerList[$index] = $item;
    }

    /**
     * verilen indekse çoklu öğe ekler.
     * @param int $index indeks
     * @param UseItem ...$items öğeler
     */
    public function insertRange(int $index, UseItem ...$items): void
    {
        $this->innerList->insertRange($index, $items);
    }


    /**
     * verilen öğe dizisine göre koleksiyonu senkronize eder.
     * @param array|UseItem[] $useItems öğe dizisi
     */
    public function syncUseItems(array $useItems)
    {
        if (empty($useItems)) {
            return;
        }

        foreach ($useItems as $useItem) {
            /** @var UseItem $itm */
            $itm = $this->firstOrNull(function (UseItem $item) use ($useItem) {
                return $item->getFullName() == $useItem->getFullName();
            });
            if (!empty($itm)) {
                if ($itm->isAliasName()) {
                    $useItem->setAliasName($itm->getAliasName());
                }
            } else {
                $this->add($useItem);
            }
        }
        $that = $this->toArrayList();

        /** @var ArrayList|UseItem[] $nameConflicts */
        $nameConflicts = $this->where(function (UseItem $item) use ($that) {
            return $that->any(function (UseItem $itm) use ($item) {
                return ($itm->getName() == $item->getName()) &&
                    ($itm->getAliasName() == $item->getAliasName()) &&
                    ($itm->getFullName() != $item->getFullName())
                    ;
            });
        })->toArrayList();


        while ($nameConflicts->any()) {
            $lastName = "";
            foreach ($nameConflicts as $cnfItem) {
                if ($cnfItem->isAliasName()) {
                    if ($cnfItem->getAliasName() === $lastName || $cnfItem->getName() === $lastName) {
                        continue;
                    }
                    $lastName = $cnfItem->getAliasName();
                    /** @var UseItem $lastItem */
                    $lastItem = $this->lastOrNull(function (UseItem $lastItm) use ($lastName) {
                        return $lastItm->getAliasName() == $lastName;
                    });
                    if (is_numeric(substr($lastName, -1))) {
                        $num = intval(substr($lastName, -1)) + 1;
                        $lastName = substr($lastName, 0, -1) . $num;
                    } else {
                        $lastName .= '1';
                    }
                    $lastItem->setAliasName($lastName);

                } else {
                    if ($cnfItem->getName() === $lastName) {
                        continue;
                    }
                    $lastName = $cnfItem->getName();
                    /** @var UseItem $lastItem */
                    $lastItem = $this->lastOrNull(function (UseItem $lastItm) use ($lastName) {
                        return $lastItm->getName() == $lastName;
                    });
                    $lastItem->setAliasName($lastName . '1');


                }
            }
            $nameConflicts = $this->where(function (UseItem $item) use ($that) {
                return $that->any(function (UseItem $itm) use ($item) {
                    return $itm->getName() == $item->getName() &&
                        $itm->getAliasName() == $item->getAliasName() &&
                        $itm->getFullName() != $item->getFullName();
                });
            })->toArrayList();
        }

    }

    /**
     * koleksiyonu metne (php koduna) çevirir.
     * @return string
     */
    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion methods

    #region IToIndentedString

    /**
     * Koleksiyonu girintili metne (php koduna) çevirir.
     * @param int $indentLevel girinti seviyesi
     * @param string $indentChars girinti karakterleri
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        return $this->aggregate(function (&$lines, UseItem $item) use ($indentLevel, $indentChars) {
            $lines .= $item->toIndentedString($indentLevel, $indentChars) . PHP_EOL;
        }, '');
    }

    #endregion IToIndentedString

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = $this->toArray();
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data =  unserialize($serialized);
        parent::__construct(new ArrayList(), UseItem::class);
        $this->addRange(...$data);
    }

    #endregion Serializable
}
