<?php


namespace Gek\PhpLang\Collections;


use Gek\Collections\ArrayList;
use Gek\Collections\TypedListWrapper;
use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\TraitItem;
use Gek\PhpLang\UseItem;

/**
 * TraitItemCollection Sınıfı
 *
 * Kullanılan traitler koleksiyonu.
 *
 * @package Gek\PhpLang\Collections
 */
class TraitItemCollection extends TypedListWrapper implements IUseCreator, IToIndentedString, \Serializable
{

    #region ctor

    /**
     * TraitItemCollection yapıcı metod.
     * @param TraitItem ...$items (opsiyonel) öğeler
     */
    public function __construct(TraitItem ...$items)
    {
        parent::__construct(new ArrayList(), TraitItem::class);
        $this->addRange(...$items);
    }

    #endregion ctor

    #region methods

    /**
     * öğe ekler
     * @param TraitItem $item öğe
     */
    public function add(TraitItem $item):void{
        $this->innerList->add($item);
    }

    /**
     * çoklu öğe ekler
     * @param TraitItem ...$items öğeler
     */
    public function addRange(TraitItem ...$items):void{
        $this->innerList->addRange($items);
    }



    /**
     * öğenin koleksiyonda olup olmadığına bakar.
     * @param TraitItem $item öğe
     * @return bool varsa true yoksa false
     */
    public function contains(TraitItem $item):bool{
        return $this->innerList->contains($item);
    }

    /**
     * öğeyi koleksiyondan kaldırır.
     * @param TraitItem  $item öğe
     * @return bool başarı durumunda true aksi halde false
     */
    public function remove(TraitItem $item):bool{
        return $this->innerList->remove($item);
    }

    /**
     * verilen öğenin koleksiyondaki indeksinin verir.
     * @param TraitItem $item öğe
     * @return int öğenin indeksi, öğe bulunamazsa -1
     */
    public function indexOf(TraitItem $item): int{
        return $this->innerList->indexOf($item);
    }

    /**
     * verilen indekse öğeyi ekler.
     * @param int $index indeks
     * @param TraitItem $item öğe
     */
    public function insert(int $index, TraitItem $item): void{
        $this->innerList->insert($index,$item);
    }

    /**
     * verilen indeksteki öğeyi verir.
     * @param int $index indeks
     * @return TraitItem öğe
     */
    public function getAt(int $index):TraitItem {
        return $this->innerList[$index];
    }

    /**
     * verilen indekse öğeyi set eder.
     * @param int $index indeks
     * @param TraitItem $item öğe
     */
    public function setAt(int $index, TraitItem $item):void{
        $this->innerList[$index] = $item;
    }

    /**
     * verilen indekse çoklu öğe ekler.
     * @param int $index indeks
     * @param TraitItem ...$items öğeler
     */
    public function insertRange(int $index, TraitItem ...$items):void{
        $this->innerList->insertRange($index,$items);
    }


    /**
     * koleksiyonu metne (php koduna) çevirir.
     * @return string
     */
    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion methods

    #region IUseCreator

    /**
     * koleksiyondaki türleri verir.
     *
     * @return array|UseItem[]
     */
    public function getUseArray(): array
    {
        return $this->where(function (TraitItem $item){
            return $item->getType()->isUseItem();
        })->aggregate(function (&$arr, TraitItem $item){
            $arr[] = $item->getType()->getUseItem();
        },array());
    }

    #endregion IUseCreator


    #region IToIndentedString

    /**
     * girintili metni (php kodunu) verir.
     * @param int $indentLevel (opsiyonel) girinti seviyesi
     * @param string $indentChars (opsiyonel) girinti karakterleri
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        return $this->aggregate(function (&$res,TraitItem $traitItem) use($indentLevel,$indentChars){
            $res .= $traitItem->toIndentedString($indentLevel,$indentChars) . PHP_EOL;
        },'');
    }

    #endregion IToIndentedString


    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = $this->toArray();
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data =  unserialize($serialized);
        parent::__construct(new ArrayList(), TraitItem::class);
        $this->addRange(...$data);
    }

    #endregion Serializable

}
