<?php


namespace Gek\PhpLang\Collections;


use Gek\Collections\ArrayList;
use Gek\Collections\TypedListWrapper;
use Gek\PhpLang\CodeLine;
use Gek\PhpLang\Contracts\IToIndentedString;

/**
 * CodeLineCollection sınıfı
 *
 * Kod satırı koleksiyonu
 * @package Gek\PhpLang\Collections
 */
class CodeLineCollection extends TypedListWrapper implements IToIndentedString, \Serializable
{

    #region ctor

    /**
     * CodeLineCollection yapıcı method.
     * @param CodeLine ...$items kode satırları
     */
    public function __construct(CodeLine ...$items)
    {
        parent::__construct(new ArrayList(), CodeLine::class);
        $this->addRange(...$items);
    }

    #endregion ctor

    #region methods

    /**
     * Kod satırı ekler
     *
     * @param CodeLine $item kod satırı
     */
    public function add(CodeLine $item): void
    {
        $this->innerList->add($item);
    }

    /**
     * çoklu kod satırı ekler
     *
     * @param CodeLine ...$items kod satırları
     */
    public function addRange(CodeLine ...$items): void
    {
        $this->innerList->addRange($items);
    }


    /**
     * Kod satırının koleksiyonda olup olmadığına bakar
     *
     * @param CodeLine $item kod satırı
     * @return bool varsa true yoksa false
     */
    public function contains(CodeLine $item): bool
    {
        return $this->innerList->contains($item);
    }

    /**
     * verilen kod satırını koleksiyondan kaldırır
     * @param CodeLine $item kod satırı
     * @return bool başarılı durumda true aksi halde false
     */
    public function remove(CodeLine $item): bool
    {
        return $this->innerList->remove($item);
    }

    /**
     * Verilen kod satırının koleksiyondaki indeksini verir.
     *
     * @param CodeLine $item kod satırı
     * @return int indeks kod satırı bulunamazsa -1 döner
     */
    public function indexOf(CodeLine $item): int
    {
        return $this->innerList->indexOf($item);
    }

    /**
     * kod satırını verilen indekse ekler.
     *
     * @param int $index indeks
     * @param CodeLine $item kod satırı
     */
    public function insert(int $index, CodeLine $item): void
    {
        $this->innerList->insert($index, $item);
    }

    /**
     * verilen indeksteki kod satırını verir.
     *
     * @param int $index indeks
     * @return CodeLine kod satırı
     */
    public function getAt(int $index): CodeLine
    {
        return $this->innerList[$index];
    }

    /**
     * verilen indekse kod satırını set eder.
     *
     * @param int $index indeks
     * @param CodeLine $item kod satırı
     */
    public function setAt(int $index, CodeLine $item): void
    {
        $this->innerList[$index] = $item;
    }

    /**
     * verilen indekse çoklu kod satırlarını ekler.
     * @param int $index indeks
     * @param CodeLine ...$items Kod satırları
     */
    public function insertRange(int $index, CodeLine ...$items): void
    {
        $this->innerList->insertRange($index, $items);
    }


    /**
     * kod satırı ekler
     *
     * @param CodeLine|string $code kod satırı veya metin
     * @param int $indentLevel girinti düzeyi
     */
    public function addCodeLine($code, int $indentLevel = 0)
    {
        if ($code instanceof CodeLine) {
            $code->setIndentLevel($indentLevel);
            $this->add($code);
        } else {
            $this->add(new CodeLine(trim($code), $indentLevel));
        }
    }

    /**
     * @param string $codes
     */
    public function addCodes(string $codes)
    {
        $paramArr = CodeLine::parseCodes($codes);
        $this->addRange(...$paramArr);
    }


    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion methods

    #region IToIndentedString

    /**
     * @param int $indentLevel
     * @param string $indentChars
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        $lines = $this->select(function (CodeLine $item)use($indentLevel,$indentChars){
            return $item->toIndentedString($indentLevel,$indentChars);
        })->toArray();
        return implode(PHP_EOL, $lines) . PHP_EOL;
    }

    #endregion IToIndentedString

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
       $data = $this->toArray();
       return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data =  unserialize($serialized);
        parent::__construct(new ArrayList(), CodeLine::class);
        $this->addRange(...$data);
    }

    #endregion Serializable

}
