<?php


namespace Gek\PhpLang;


use Gek\Collections\ArrayList;
use Gek\Collections\Enumerable;
use Gek\Infrastructure\ClassHelper;
use Gek\Infrastructure\Reflections\RCaller;
use Gek\Infrastructure\Str;
use Gek\PhpLang\Collections\UseItemCollection;
use Gek\PhpLang\DocComments\DocComment;
use phpDocumentor\Reflection\DocBlock;
use phpDocumentor\Reflection\Exception;
use phpDocumentor\Reflection\File\LocalFile;
use phpDocumentor\Reflection\Php\ProjectFactory;
use PhpParser\Node\Stmt\Use_;
use PhpParser\NodeTraverser;
use PhpParser\NodeVisitorAbstract;
use PhpParser\ParserFactory;
use ReflectionMethod;
use ReflectionProperty;

/**
 * CodeFactory sınıfı
 *
 * Kod fabrikası
 * @package Gek\PhpLang
 */
class CodeFactory
{

    /**
     * Yeni bir sınıf oluşturur.
     * @param string $name sınıf adı
     * @return PhpClass
     */
    public static function classCreate(string $name): PhpClass
    {
        ClassHelper::checkReflectSupport();
        return self::create($name);
    }

    /**
     * Verilen reflectionu kullanarak yeni bir sınıf oluşturur.
     * @param \ReflectionClass $reflectedClass yansıtılmış sınıf
     * @param bool $forceNonUserClass kullanıcı tanımlı olmayan sınıf olsa bile oluşturmak için zorla
     * @param bool $forceUseReflect kaynak dosya olsa bile oluşturma işleminde sadece yansımayı kullanmak için zorla
     * @return PhpClass sınıf
     * @throws Exception
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public static function createFromReflection(\ReflectionClass $reflectedClass, bool $forceNonUserClass = false, bool $forceUseReflect = false)
    {
        ClassHelper::checkReflectSupport();
        $fromFile = true;
        if (empty($reflectedClass->getFileName())) {
            if (false == $forceNonUserClass) {
                throw new Exception("Yansıtılmış sınıfın dosya yolu bulunamadı.");
            }
            $fromFile = false;
        }
        if ($forceUseReflect) {
            $fromFile = false;
        }
        if ($fromFile) {
            if ($reflectedClass->isInterface()) {
                return static::interfaceCreateFromFile($reflectedClass->getFileName());
            } elseif ($reflectedClass->isTrait()) {
                return static::traitCreateFromFile($reflectedClass->getFileName());
            }
            return static::classCreateFromFile($reflectedClass->getFileName());
        }
        if ($reflectedClass->isTrait()) {
            $res = static::traitCreate($reflectedClass->getShortName());
        } elseif ($reflectedClass->isInterface()) {
            $res = static::interfaceCreate($reflectedClass->getShortName());
        } else {
            $res = static::classCreate($reflectedClass->getShortName());
        }

        if (false == $reflectedClass->isTrait()) {
            if ($reflectedClass->isInterface()) {
                foreach ($reflectedClass->getInterfaces() as $interface) {
                    $res->addExtendsForInterface($interface->getName());
                }

            } elseif (!empty($reflectedClass->getParentClass())) {
                $res->setExtends($reflectedClass->getParentClass()->getName());
            }
        }

        if ($reflectedClass->isAbstract()) {
            $res->setAbstract();
        }

        if ($reflectedClass->isFinal()) {
            $res->setFinal();
        }

        if ($res->isClass()) {
            foreach ($reflectedClass->getInterfaces() as $interface) {
                $res->addImplement($interface->getName());
            }
        }

        $res->setNamespace($reflectedClass->getNamespaceName());

        if (!empty($reflectedClass->getDocComment())) {
            $docCom = DocComment::parse($reflectedClass->getDocComment());
            $res->setDocComment($docCom);
        }

        $traitAliases = $reflectedClass->getTraitAliases();
        $resAliases = array();
        foreach ($traitAliases as $alias => $fulMethodName) {
            $nmArr = explode("::", $fulMethodName);
            $traitName = $nmArr[0];
            $method = $nmArr[1];
            if (!isset($resAliases[$traitName])) {
                $resAliases[$traitName] = [];
            }
            $resAliases[$traitName][$method] = $alias;
        }
        foreach ($reflectedClass->getTraitNames() as $traitName) {
            $tr = new TraitItem($traitName);
            if (isset($resAliases[$traitName])) {
                foreach ($resAliases[$traitName] as $mthd => $alias) {
                    $tr->getAliases()->tryAddKeyValue($mthd, $alias);
                }
            }
            $res->addTrait($tr);
        }

        foreach ($reflectedClass->getReflectionConstants() as $rConst) {
            $val = static::valueToLiteral($rConst->getValue());
            $nConst = new ClassConst($rConst->getName(), $val);

            if ($rConst->isPublic()) {
                $nConst->setPublic();
            } elseif ($rConst->isProtected()) {
                $nConst->setProtected();
            } elseif ($rConst->isPrivate()) {
                $nConst->setPrivate();
            }
            $dc = $rConst->getDocComment();
            if (!empty($ds)) {
                $docCom = DocComment::parse($dc);
                $nConst->setDocComment($docCom);
            }
            $res->addMember($nConst);

        }

        $ignoredProp = new ArrayList();
        foreach ($reflectedClass->getTraits() as $trait) {
            $ignoredProp->addRange(
                Enumerable::fromArray($trait->getProperties())
                    ->select(fn(ReflectionProperty $p) => $p->getName())
            );
        }
        if (!empty($reflectedClass->getParentClass())) {
            $ignoredProp->addRange(
                Enumerable::fromArray($reflectedClass->getParentClass()->getProperties())
                    ->select(fn(ReflectionProperty $p) => $p->getName())
            );
        }


        $propDefVals = $reflectedClass->getDefaultProperties();
        foreach ($reflectedClass->getProperties() as $prop) {

            if ($ignoredProp->contains($prop->getName())) {
                continue;
            }
            $nP = $res->addProperty($prop->getName());

            if (array_key_exists($prop->getName(), $propDefVals)) {
                $nP->setValue(static::valueToLiteral($propDefVals[$prop->getName()]));
            }

            if ($prop->hasType()) {
                $typeName = $prop->getType()->getName();
                if ($prop->getType()->allowsNull() && false == Str::startsWith($typeName, "?")) {
                    $typeName = '?' . $typeName;
                }
                $nP->setType($typeName);
            }
            if ($prop->isStatic()) {
                $nP->setStatic();
            }


            if ($prop->isPublic()) {
                $nP->setPublic();
            } elseif ($prop->isProtected()) {
                $nP->setProtected();
            } elseif ($prop->isPrivate()) {
                $nP->setPrivate();
            }

            if (!empty($prop->getDocComment())) {
                $docCom = DocComment::parse($prop->getDocComment());
                $nP->setDocComment($docCom);
            }

        }

        $ignoredMethods = new ArrayList();
        foreach ($reflectedClass->getTraits() as $trait) {
            $ignoredMethods->addRange(
                Enumerable::fromArray($trait->getMethods())
                    ->select(function (ReflectionMethod $p) use ($resAliases, $trait) {
                        $mName = $p->getName();
                        if (isset($resAliases[$trait->getName()])) {
                            if (isset($resAliases[$trait->getName()][$mName])) {
                                $mName = $resAliases[$trait->getName()][$mName];
                            }
                        }
                        return $mName;
                    })
            );
        }


        foreach ($reflectedClass->getMethods() as $method) {
            if ($method->getDeclaringClass()->getName() != $reflectedClass->getName()) {
                continue;
            }
            if ($ignoredMethods->contains($method->getName())) {
                continue;
            }
            $cMethod = $res->addMethod($method->getName());
            if (!empty($method->getReturnType())) {
                $retTypeName = $method->getReturnType()->getName();
                if ($method->getReturnType()->allowsNull() && false == Str::startsWith($retTypeName, "?")) {
                    $retTypeName = "?" . $retTypeName;
                }
                $cMethod->setReturnType($retTypeName);
            }
            if ($method->isStatic()) {
                $cMethod->setStatic();
            }
            if ($method->isFinal()) {
                $cMethod->setFinal();
            }
            if ($method->isAbstract()) {
                $cMethod->setAbstract();
            }

            if ($method->isPublic()) {
                $cMethod->setPublic();
            } elseif ($method->isProtected()) {
                $cMethod->setProtected();
            } elseif ($method->isPrivate()) {
                $cMethod->setPrivate();
            }

            if (!empty($method->getDocComment())) {
                $docCom = DocComment::parse($method->getDocComment());
                $cMethod->setDocComment($docCom);
            }

            if (!empty($method->getParameters())) {
                foreach ($method->getParameters() as $argument) {
                    $param = new MethodParam($argument->getName());
                    if (!empty($argument->getType())) {
                        $argType = $argument->getType()->getName();
                        if (($argument->getType()->allowsNull() || $argument->allowsNull()) && false == Str::startsWith($argType, "?")) {
                            $argType = "?" . $argType;
                        }
                        $param->setType($argType);
                    }

                    if (!empty($argument->isPassedByReference())) {
                        $param->setReference();
                    }
                    if (!empty($argument->isVariadic())) {
                        $param->setDotted();
                    }

                    try {
                        if ($argument->isOptional()) {
                            if ($argument->isDefaultValueConstant()) {
                                $param->setValue($argument->getDefaultValueConstantName());
                            } elseif ($argument->isDefaultValueAvailable()) {
                                $param->setValue(static::valueToLiteral($argument->getDefaultValue()));
                            }
                        }
                    } catch (\ReflectionException $exp) {

                    }

                    $cMethod->addParam($param);
                }
            }
        }
        return $res;
    }


    /**
     * kaynak kod dosyasında yeni bir sınıf oluştur.
     * @param string $filePath dosya yolu
     * @param string|null $className sınıf adı
     * @return PhpClass sınıf
     * @throws Exception
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public static function classCreateFromFile(string $filePath, ?string $className = null)
    {
        ClassHelper::checkReflectSupport();
        if (false == file_exists($filePath)) {
            throw new Exception("Dosya bulunamadı: " . $filePath);
        }

        $projectFiles = [new LocalFile($filePath)];

        $projectFactory = ProjectFactory::createInstance();

        /** @var \phpDocumentor\Reflection\Php\Project $project */
        $project = $projectFactory->create("sample", $projectFiles);
        $fileArr = $project->getFiles();
        $fileArr = array_reverse($fileArr);
        /** @var \phpDocumentor\Reflection\Php\File $refFile */
        $refFile = array_pop($fileArr);

        $selClass = null;

        $refClasses = $refFile->getClasses();
        if (!empty($className)) {
            foreach ($refClasses as $key => $cClass) {
                if ($cClass->getName() == $className) {
                    $selClass = $cClass;
                    break;
                }
                if ($key == $className) {
                    $selClass = $cClass;
                    break;
                }
            }

            if (empty($selClass)) {
                throw new \Exception("Dosya içinde " . $className . " sınıfı bulunamadı");
            }
        } else {
            foreach ($refClasses as $key => $cClass) {

                $selClass = $cClass;
                break;

            }
        }
        if (empty($selClass)) {
            throw new \Exception("Dosya içinde hiçbir sınıf bulunamadı. dosya: " . $filePath);
        }

        $fqsen = $selClass->getFqsen();

        $namespace = trim(substr($fqsen, 0, -strlen($selClass->getName())), "\\");


        if (empty($namespace)) {
            $nsArr = $refFile->getNamespaces();
            foreach ($nsArr as $ns) {
                $namespace = $ns->getName();
                break;
            }
        }

        if (empty($namespace)) {
            $namespace = null;
        }

        $content = new ArrayList(
            explode(PHP_EOL, $refFile->getSource())
        );
        $startLine = $selClass->getLocation()->getLineNumber();
        $endLine = $selClass->getEndLocation()->getLineNumber();
        $skip = $startLine > 0 ? ($startLine - 1) : 0;
        $take = ($endLine - $startLine) + 1;
        $content = $content->skip($skip)
            ->take($take)
            ->toArrayList();

        $regions = static::getRegions($content, $startLine);

        $resClass = static::create($selClass->getName(), ClassTypes::CLASS_());

        if (!empty($namespace)) {
            $resClass->setNamespace($namespace);
        }

        $docBlock = $selClass->getDocBlock();
        if (!empty($docBlock)) {
            $docCom = static::convertDocComment($docBlock);
            $resClass->setDocComment($docCom);
        }

        $parser = (new ParserFactory)->create(ParserFactory::PREFER_PHP7);


        $ast = $parser->parse($refFile->getSource());
        $traverser = new NodeTraverser;
        // $flag = false;
        $visitor = new UseUseVisitor();
        $traverser->addVisitor($visitor);
        $traverser->traverse($ast);

        $useItems = $visitor->getFoundedUses();
        $resClass->setUses($useItems);


        if (!empty($selClass->getParent())) {
            $resClass->setExtends($selClass->getParent() . "");

        }

        if ($selClass->isFinal()) {
            $resClass->setFinal();
        }

        if ($selClass->isAbstract()) {
            $resClass->setAbstract();
        }

        if (!empty($selClass->getInterfaces())) {
            foreach ($selClass->getInterfaces() as $interface) {
                $resClass->addImplement($interface . "");

            }
        }

        if (!empty($selClass->getUsedTraits())) {

            $parser = (new ParserFactory)->create(ParserFactory::PREFER_PHP7);
            $ast = $parser->parse($refFile->getSource());
            $traverser = new NodeTraverser;
            // $flag = false;
            $visitor = new UseTraitsVisitor();
            $traverser->addVisitor($visitor);
            $traverser->traverse($ast);

            $traitAliassList = $visitor->getTraitAliases();

            foreach ($selClass->getUsedTraits() as $uTrait) {
                $aliasArr = array();
                $fullName = $uTrait ."";
                $shortName = $uTrait->getName();
                if(array_key_exists($fullName,$traitAliassList)){
                    $aliasArr = $traitAliassList[$fullName];
                }elseif(array_key_exists($shortName,$traitAliassList)){
                    $aliasArr = $traitAliassList[$shortName];
                }
                if(empty($aliasArr)){
                    $resClass->addTrait($fullName);
                }else{

                    $resClass->addTrait($fullName,$aliasArr);
                }


            }
        }

        if (!empty($selClass->getConstants())) {
            foreach ($selClass->getConstants() as $const) {

                $cVal = !empty($const->getValue()) ? $const->getValue() : "";

                $cConst = new ClassConst($const->getName(), $cVal);


                if (PhpVisibility::checkConstValue($const->getVisibility() . "")) {
                    $v = new PhpVisibility($const->getVisibility() . "");
                    $cConst->setVisibility($v);
                }
                if (!empty($const->getDocBlock())) {
                    $cConst->setDocComment(static::convertDocComment($const->getDocBlock()));
                }
                $ln = $const->getLocation()->getLineNumber();
                $cReg = $regions
                    ->firstOrNull(function (object $r) use ($ln) {
                        return ($r->startLine < $ln) && ($r->endLine > $ln);
                    });
                if (!empty($cReg)) {
                    $cConst->setRegion($cReg->region);
                }
                $resClass->addMember($cConst);

            }

        }


        if (!empty($selClass->getProperties())) {
            foreach ($selClass->getProperties() as $prop) {

                $cProp = $resClass->addField($prop->getName());
                if (!empty($prop->getType())) {
                    $cProp->setType($prop->getType()->__toString());
                }
                if (!empty($prop->getDefault())) {
                    $cProp->setValue($prop->getDefault());
                }

                if ($prop->isStatic()) {
                    $cProp->setStatic();
                }
                if (PhpVisibility::checkConstValue($prop->getVisibility() . "")) {
                    $v = new PhpVisibility($prop->getVisibility() . "");
                    $cProp->setVisibility($v);
                }
                if (!empty($prop->getDocBlock())) {
                    $cProp->setDocComment(static::convertDocComment($prop->getDocBlock()));
                }
                $ln = $prop->getLocation()->getLineNumber();
                $cReg = $regions
                    ->firstOrNull(function (object $r) use ($ln) {
                        return ($r->startLine < $ln) && ($r->endLine > $ln);
                    });
                if (!empty($cReg)) {
                    $cProp->setRegion($cReg->region);
                }

            }

        }

        if (!empty($selClass->getMethods())) {
            foreach ($selClass->getMethods() as $method) {


                $cMethod = $resClass->addMethod($method->getName());
                if (!empty($method->getReturnType())) {
                    $cMethod->setReturnType($method->getReturnType()->__toString());
                }
                if ($method->isStatic()) {
                    $cMethod->setStatic();
                }
                if ($method->isFinal()) {
                    $cMethod->setFinal();
                }
                if ($method->isAbstract()) {
                    $cMethod->setAbstract();
                }
                if (PhpVisibility::checkConstValue($method->getVisibility() . "")) {
                    $v = new PhpVisibility($method->getVisibility() . "");
                    $cMethod->setVisibility($v);
                }
                if (!empty($method->getDocBlock())) {
                    $cMethod->setDocComment(static::convertDocComment($method->getDocBlock()));
                }

                $ln = $method->getLocation()->getLineNumber();
                $cReg = $regions
                    ->firstOrNull(function (object $r) use ($ln) {
                        return ($r->startLine < $ln) && ($r->endLine > $ln);
                    });
                if (!empty($cReg)) {
                    $cMethod->setRegion($cReg->region);
                }

                if (!empty($method->getArguments())) {
                    foreach ($method->getArguments() as $argument) {
                        $param = new MethodParam($argument->getName());
                        if (!empty($argument->getType())) {
                            $param->setType($argument->getType()->__toString());
                        }
                        if ($argument->getDefault() !== null) {
                            $param->setValue($argument->getDefault());
                        }
                        if (!empty($argument->isByReference())) {
                            $param->setReference();
                        }
                        if (!empty($argument->isVariadic())) {
                            $param->setDotted();
                        }
                        $cMethod->addParam($param);
                    }
                }

                if ($method->isAbstract()) {
                    continue;
                }
                $traverser = new NodeTraverser;
                // $flag = false;
                $visitor = new MethodBodyVisitor($cMethod);
                $traverser->addVisitor($visitor);
                $traverser->traverse($ast);

                $body = $visitor->getMethodBody();
                if ($body !== null) {
                    $cMethod->setBody($body);
                }


            }
        }

        return $resClass;
    }

    /**
     * kaynak kod dosyasında yeni bir trait oluştur.
     * @param string $filePath dosya yolu
     * @param string|null $className trait adı
     * @return PhpClass trait
     * @throws Exception
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public static function traitCreateFromFile(string $filePath, ?string $className = null)
    {
        ClassHelper::checkReflectSupport();
        if (false == file_exists($filePath)) {
            throw new Exception("Dosya bulunamadı: " . $filePath);
        }

        $projectFiles = [new LocalFile($filePath)];

        $projectFactory = ProjectFactory::createInstance();

        /** @var \phpDocumentor\Reflection\Php\Project $project */
        $project = $projectFactory->create("sample", $projectFiles);
        $fileArr = $project->getFiles();
        $fileArr = array_reverse($fileArr);
        /** @var \phpDocumentor\Reflection\Php\File $refFile */
        $refFile = array_pop($fileArr);

        $selClass = null;

        $refClasses = $refFile->getTraits();
        if (!empty($className)) {
            foreach ($refClasses as $key => $cClass) {
                if ($cClass->getName() == $className) {
                    $selClass = $cClass;
                    break;
                }
                if ($key == $className) {
                    $selClass = $cClass;
                    break;
                }
            }

            if (empty($selClass)) {
                throw new \Exception("Dosya içinde " . $className . " sınıfı bulunamadı");
            }
        } else {
            foreach ($refClasses as $key => $cClass) {

                $selClass = $cClass;
                break;

            }
        }
        if (empty($selClass)) {
            throw new \Exception("Dosya içinde hiçbir sınıf bulunamadı. dosya: " . $filePath);
        }

        $fqsen = $selClass->getFqsen();

        $namespace = trim(substr($fqsen, 0, -strlen($selClass->getName())), "\\");


        if (empty($namespace)) {
            $nsArr = $refFile->getNamespaces();
            foreach ($nsArr as $ns) {
                $namespace = $ns->getName();
                break;
            }
        }

        if (empty($namespace)) {
            $namespace = null;
        }

        $content = new ArrayList(
            explode(PHP_EOL, $refFile->getSource())
        );
        $startLine = $selClass->getLocation()->getLineNumber();
        $endLine = $selClass->getEndLocation()->getLineNumber();
        $skip = $startLine > 0 ? ($startLine - 1) : 0;
        $take = ($endLine - $startLine) + 1;
        $content = $content->skip($skip)
            ->take($take)
            ->toArrayList();

        $regions = static::getRegions($content, $startLine);

        $resClass = static::create($selClass->getName(), ClassTypes::TRAIT());

        if (!empty($namespace)) {
            $resClass->setNamespace($namespace);
        }

        $docBlock = $selClass->getDocBlock();
        if (!empty($docBlock)) {
            $docCom = static::convertDocComment($docBlock);
            $resClass->setDocComment($docCom);
        }

        $parser = (new ParserFactory)->create(ParserFactory::PREFER_PHP7);


        $ast = $parser->parse($refFile->getSource());
        $traverser = new NodeTraverser;
        // $flag = false;
        $visitor = new UseUseVisitor();
        $traverser->addVisitor($visitor);
        $traverser->traverse($ast);

        $useItems = $visitor->getFoundedUses();
        $resClass->setUses($useItems);


        if (!empty($selClass->getUsedTraits())) {
            foreach ($selClass->getUsedTraits() as $uTrait) {
                $resClass->addTrait($uTrait . "");
            }
        }


        if (!empty($selClass->getProperties())) {
            foreach ($selClass->getProperties() as $prop) {

                $cProp = $resClass->addField($prop->getName());
                if (!empty($prop->getType())) {
                    $cProp->setType($prop->getType()->__toString());
                }
                if (!empty($prop->getDefault())) {
                    $cProp->setValue($prop->getDefault());
                }

                if ($prop->isStatic()) {
                    $cProp->setStatic();
                }
                if (PhpVisibility::checkConstValue($prop->getVisibility() . "")) {
                    $v = new PhpVisibility($prop->getVisibility() . "");
                    $cProp->setVisibility($v);
                }
                if (!empty($prop->getDocBlock())) {
                    $cProp->setDocComment(static::convertDocComment($prop->getDocBlock()));
                }
                $ln = $prop->getLocation()->getLineNumber();
                $cReg = $regions
                    ->firstOrNull(function (object $r) use ($ln) {
                        return ($r->startLine < $ln) && ($r->endLine > $ln);
                    });
                if (!empty($cReg)) {
                    $cProp->setRegion($cReg->region);
                }

            }

        }

        if (!empty($selClass->getMethods())) {
            foreach ($selClass->getMethods() as $method) {


                $cMethod = $resClass->addMethod($method->getName());
                if (!empty($method->getReturnType())) {
                    $cMethod->setReturnType($method->getReturnType()->__toString());
                }
                if ($method->isStatic()) {
                    $cMethod->setStatic();
                }
                if ($method->isFinal()) {
                    $cMethod->setFinal();
                }
                if ($method->isAbstract()) {
                    $cMethod->setAbstract();
                }
                if (PhpVisibility::checkConstValue($method->getVisibility() . "")) {
                    $v = new PhpVisibility($method->getVisibility() . "");
                    $cMethod->setVisibility($v);
                }
                if (!empty($method->getDocBlock())) {
                    $cMethod->setDocComment(static::convertDocComment($method->getDocBlock()));
                }

                $ln = $method->getLocation()->getLineNumber();
                $cReg = $regions
                    ->firstOrNull(function (object $r) use ($ln) {
                        return ($r->startLine < $ln) && ($r->endLine > $ln);
                    });
                if (!empty($cReg)) {
                    $cMethod->setRegion($cReg->region);
                }

                if (!empty($method->getArguments())) {
                    foreach ($method->getArguments() as $argument) {
                        $param = new MethodParam($argument->getName());
                        if (!empty($argument->getType())) {
                            $param->setType($argument->getType()->__toString());
                        }
                        if ($argument->getDefault() !== null) {
                            $param->setValue($argument->getDefault());
                        }
                        if (!empty($argument->isByReference())) {
                            $param->setReference();
                        }
                        if (!empty($argument->isVariadic())) {
                            $param->setDotted();
                        }
                        $cMethod->addParam($param);
                    }
                }

                if ($method->isAbstract()) {
                    continue;
                }
                $traverser = new NodeTraverser;
                // $flag = false;
                $visitor = new MethodBodyVisitor($cMethod);
                $traverser->addVisitor($visitor);
                $traverser->traverse($ast);

                $body = $visitor->getMethodBody();
                if ($body !== null) {
                    $cMethod->setBody($body);
                }


            }
        }

        return $resClass;
    }

    /**
     * kaynak kod dosyasında yeni bir arayüz oluştur.
     * @param string $filePath dosya yolu
     * @param string|null $className arayüz adı
     * @return PhpClass arayüz
     * @throws Exception
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public static function interfaceCreateFromFile(string $filePath, ?string $className = null)
    {
        ClassHelper::checkReflectSupport();
        if (false == file_exists($filePath)) {
            throw new Exception("Dosya bulunamadı: " . $filePath);
        }

        $projectFiles = [new LocalFile($filePath)];

        $projectFactory = ProjectFactory::createInstance();

        /** @var \phpDocumentor\Reflection\Php\Project $project */
        $project = $projectFactory->create("sample", $projectFiles);
        $fileArr = $project->getFiles();
        $fileArr = array_reverse($fileArr);
        /** @var \phpDocumentor\Reflection\Php\File $refFile */
        $refFile = array_pop($fileArr);

        $selClass = null;

        $refClasses = $refFile->getInterfaces();
        if (!empty($className)) {
            foreach ($refClasses as $key => $cClass) {
                if ($cClass->getName() == $className) {
                    $selClass = $cClass;
                    break;
                }
                if ($key == $className) {
                    $selClass = $cClass;
                    break;
                }
            }

            if (empty($selClass)) {
                throw new \Exception("Dosya içinde " . $className . " sınıfı bulunamadı");
            }
        } else {
            foreach ($refClasses as $key => $cClass) {

                $selClass = $cClass;
                break;

            }
        }
        if (empty($selClass)) {
            throw new \Exception("Dosya içinde hiçbir sınıf bulunamadı. dosya: " . $filePath);
        }

        $fqsen = $selClass->getFqsen();

        $namespace = trim(substr($fqsen, 0, -strlen($selClass->getName())), "\\");


        if (empty($namespace)) {
            $nsArr = $refFile->getNamespaces();
            foreach ($nsArr as $ns) {
                $namespace = $ns->getName();
                break;
            }
        }

        if (empty($namespace)) {
            $namespace = null;
        }

        $content = new ArrayList(
            explode(PHP_EOL, $refFile->getSource())
        );
        $startLine = $selClass->getLocation()->getLineNumber();
        $endLine = $selClass->getEndLocation()->getLineNumber();
        $skip = $startLine > 0 ? ($startLine - 1) : 0;
        $take = ($endLine - $startLine) + 1;
        $content = $content->skip($skip)
            ->take($take)
            ->toArrayList();

        $regions = static::getRegions($content, $startLine);

        $resClass = static::create($selClass->getName(), ClassTypes::INTERFACE());

        if (!empty($namespace)) {
            $resClass->setNamespace($namespace);
        }

        $docBlock = $selClass->getDocBlock();
        if (!empty($docBlock)) {
            $docCom = static::convertDocComment($docBlock);
            $resClass->setDocComment($docCom);
        }

        $parser = (new ParserFactory)->create(ParserFactory::PREFER_PHP7);


        $ast = $parser->parse($refFile->getSource());
        $traverser = new NodeTraverser;
        // $flag = false;
        $visitor = new UseUseVisitor();
        $traverser->addVisitor($visitor);
        $traverser->traverse($ast);

        $useItems = $visitor->getFoundedUses();
        $resClass->setUses($useItems);


        if (!empty($selClass->getParents())) {
            foreach ($selClass->getParents() as $parent) {
                $resClass->addExtendsForInterface($parent . "");
            }
        }


        if (!empty($selClass->getConstants())) {
            foreach ($selClass->getConstants() as $const) {

                $cVal = !empty($const->getValue()) ? $const->getValue() : "";

                $cConst = new ClassConst($const->getName(), $cVal);


                if (PhpVisibility::checkConstValue($const->getVisibility() . "")) {
                    $v = new PhpVisibility($const->getVisibility() . "");
                    $cConst->setVisibility($v);
                }
                if (!empty($const->getDocBlock())) {
                    $cConst->setDocComment(static::convertDocComment($const->getDocBlock()));
                }
                $ln = $const->getLocation()->getLineNumber();
                $cReg = $regions
                    ->firstOrNull(function (object $r) use ($ln) {
                        return ($r->startLine < $ln) && ($r->endLine > $ln);
                    });
                if (!empty($cReg)) {
                    $cConst->setRegion($cReg->region);
                }
                $resClass->addMember($cConst);

            }

        }


        if (!empty($selClass->getMethods())) {
            foreach ($selClass->getMethods() as $method) {


                $cMethod = $resClass->addMethod($method->getName());
                if (!empty($method->getReturnType())) {
                    $cMethod->setReturnType($method->getReturnType()->__toString());
                }
                if ($method->isStatic()) {
                    $cMethod->setStatic();
                }
                if ($method->isFinal()) {
                    $cMethod->setFinal();
                }
                if ($method->isAbstract()) {
                    $cMethod->setAbstract();
                }
                if (PhpVisibility::checkConstValue($method->getVisibility() . "")) {
                    $v = new PhpVisibility($method->getVisibility() . "");
                    $cMethod->setVisibility($v);
                }
                if (!empty($method->getDocBlock())) {
                    $cMethod->setDocComment(static::convertDocComment($method->getDocBlock()));
                }

                $ln = $method->getLocation()->getLineNumber();
                $cReg = $regions
                    ->firstOrNull(function (object $r) use ($ln) {
                        return ($r->startLine < $ln) && ($r->endLine > $ln);
                    });
                if (!empty($cReg)) {
                    $cMethod->setRegion($cReg->region);
                }

                if (!empty($method->getArguments())) {
                    foreach ($method->getArguments() as $argument) {
                        $param = new MethodParam($argument->getName());
                        if (!empty($argument->getType())) {
                            $param->setType($argument->getType()->__toString());
                        }
                        if ($argument->getDefault() !== null) {
                            $param->setValue($argument->getDefault());
                        }
                        if (!empty($argument->isByReference())) {
                            $param->setReference();
                        }
                        if (!empty($argument->isVariadic())) {
                            $param->setDotted();
                        }
                        $cMethod->addParam($param);
                    }
                }

            }
        }

        return $resClass;
    }

    /**
     * yeni bir arayüz oluştur.
     * @param string $name arayüz adı
     * @return PhpClass arayüz
     */
    public static function interfaceCreate(string $name): PhpClass
    {
        ClassHelper::checkReflectSupport();
        return self::create($name, ClassTypes::INTERFACE());
    }

    /**
     * yeni bir trait oluştur.
     * @param string $name trait adı
     * @return PhpClass trait
     */
    public static function traitCreate(string $name): PhpClass
    {
        ClassHelper::checkReflectSupport();
        return self::create($name, ClassTypes::TRAIT());
    }

    /**
     * yeni bir php dopsyası oluştur
     * @param string|null $namespace ad alanı
     * @return PhpFile php dosysdı
     */
    public static function fileCreate(?string $namespace = null): PhpFile
    {
        return new PhpFile($namespace);
    }

    /**
     * PhpClass oluşturur.
     * @param string $name sınıf adı
     * @param ClassTypes|null $classType sınıf tipi 
     * @return PhpClass
     */
    protected static function create(string $name, ?ClassTypes $classType = null): PhpClass
    {
        return new PhpClass($name, $classType);
    }

    /**
     * DocBlock objesini  DocComment objesine çevirir.
     * @param DocBlock $docBlock
     * @return DocComment
     */
    protected static function convertDocComment(DocBlock $docBlock): DocComment
    {

        $docCom = new DocComment();
        $docCom->setSummary($docBlock->getSummary())
            ->setDescription($docBlock->getDescription());
        foreach ($docBlock->getTags() as $tag) {
            $nm = $tag->getName();
            $value = trim(str_replace("@" . $nm, "", $tag->render()));
            if (!empty($value)) {
                $docCom->addTag($nm, $value);
            } else {
                $docCom->addTag($nm);
            }
        }
        return $docCom;
    }

    /**
     * Kaynak kod satırı listesinden region bildirimlerini ayrıştırır.
     * @param ArrayList $content kaynak kod satır listesi
     * @param int $startLine kaynak kodun php dosyasındaki başlangıç satırı numarası
     * @return ArrayList region bildirimleri dizisi
     */
    protected static function getRegions(ArrayList $content, int $startLine): ArrayList
    {
        $regions = $content
            ->where(function (string $line) {
                return Str::startsWith(trim($line), "#region ")
                    && !empty(trim(str_replace("#region", "", trim($line))));
            })->select(function (string $line, $key) use ($startLine) {
                $region = trim(str_replace("#region", "", $line));
                $startLine = $key + $startLine;
                return (object)[
                    'region' => $region,
                    "startLine" => $startLine,
                    "endLine" => null,
                ];
            })->toArrayList();

        foreach ($regions as $rgn) {
            $regName = $rgn->region;
            $endRegLine = $content
                ->firstOrNull(function ($ln) use ($regName) {
                    $check = trim(str_replace("#endregion", "", trim($ln)));
                    return Str::equals($check, $regName, true);
                });

            if (!empty($endRegLine)) {
                $rgn->endLine = $content->indexOf($endRegLine);
                $rgn->endLine += $startLine;
            }
        }
        return $regions;
    }

    /**
     * verilen değeri php literaline çevirir.
     * @param mixed $value değer
     * @return string php kodu 
     */
    protected static function valueToLiteral($value): string
    {
        $res = var_export($value, true);

        if ($res === "NULL") {
            $res = "null";
        }
        return $res;

    }

}

/**
 * UseUseVisitor sınıfı
 * kaynak koddan import edilen türlerin listesini almak için düğüm ziyaretçisi
 * @package Gek\PhpLang
 */
class UseUseVisitor extends NodeVisitorAbstract
{


    /**
     * import türleri koleksiyonu
     * @var UseItemCollection 
     */
    protected UseItemCollection $foundedUses;

    /**
     * UseUseVisitor yapıcı metod.
     */
    public function __construct()
    {
        $this->foundedUses = new UseItemCollection();
    }

    /**
     * ziyaretçi metodu
     * @param \PhpParser\Node $node düğüm
     * @return int|\PhpParser\Node|\PhpParser\Node[]|void|null
     */
    public function leaveNode($node)
    {
        if ($node instanceof Use_) {
            $type = $node->type;

            foreach ($node->uses as $use) {
                $name = $use->name->toString();
                $alias = null;
                if (!empty($node->alias)) {
                    $alias = $node->alias->name;
                }
                $useItem = new UseItem($name, $alias, $type);
                $this->foundedUses->add($useItem);
            }

        }
    }


    /**
     * bulunan ımport türlerini verir.
     * @return UseItemCollection import türü koleksiyonu
     */
    public function getFoundedUses(): UseItemCollection
    {
        return $this->foundedUses;
    }
}

/**
 * MethodBodyVisitor sınıfı
 * 
 * kaynak koddan metod kodlarını almak için düğüm ziyaretçisi
 * @package Gek\PhpLang
 */
class MethodBodyVisitor extends NodeVisitorAbstract
{

    /**
     * Kodları alınacak metod
     * @var ClassMethod 
     */
    protected ClassMethod $searchMethod;

    /**
     * metod kodları
     * @var string|null 
     */
    protected ?string $foundMethodBody = null;

    /**
     * MethodBodyVisitor yapıcı metod.
     * @param ClassMethod $searchMethod kodları bulunacak metod.
     */
    public function __construct(ClassMethod $searchMethod)
    {
        $this->searchMethod = $searchMethod;
    }

    /**
     * ziyaretçi metod
     * @param \PhpParser\Node $node düğüm
     * @return int|\PhpParser\Node|\PhpParser\Node[]|void|null
     */
    public function leaveNode($node)
    {
        if ($node instanceof \PhpParser\Node\Stmt\ClassMethod) {
            if ($node->name->toString() == $this->searchMethod->getName()) {
                $prettyPrinter = new \PhpParser\PrettyPrinter\Standard();
                $body = $prettyPrinter->prettyPrint($node->stmts);
                $body = str_replace(["\r", "\n"], PHP_EOL, $body);
                $this->searchMethod->setBody($body);
            }

        }
    }

    /**
     * bulunan metod kodları
     * @return string|null
     */
    public function getMethodBody(): ?string
    {
        return $this->foundMethodBody;
    }

}

/**
 * UseTraitsVisitor sınıfı
 * 
 * kaynak kodundan  kullanılan traitleri almak için düğüm ziyaretçisi
 * @package Gek\PhpLang
 */
class UseTraitsVisitor extends NodeVisitorAbstract
{

    /**
     * bulunan trait kullanımları
     * @var array 
     */
    protected array $traitAliases = array();

    /**
     * UseTraitsVisitor yapıcı metod.
     */
    public function __construct()
    {

    }

    /**
     * ziyaretçi metod
     * @param \PhpParser\Node $node düğüm
     * @return int|\PhpParser\Node|\PhpParser\Node[]|void|null
     */
    public function leaveNode($node)
    {
        if ($node instanceof \PhpParser\Node\Stmt\TraitUse) {
            $prettyPrinter = new \PhpParser\PrettyPrinter\Standard();
            $nm = $prettyPrinter->prettyPrint($node->traits);
            if (!empty($node->adaptations)) {
                $this->traitAliases[$nm] = array();
                foreach ($node->adaptations as $adaptation) {
                    if (!empty($adaptation->newName) && !empty($adaptation->method)) {
                        $m = $prettyPrinter->prettyPrint([$adaptation->method]);
                        $a = $prettyPrinter->prettyPrint([$adaptation->newName]);
                        $this->traitAliases[$nm][$m] = $a;
                    }
                }
            }

        }
    }

    /**
     * bulunan trait kullanımlarını verir.
     * @return array tarit kullanımları dizisi.
     */
    public function getTraitAliases(): array
    {
        return $this->traitAliases;
    }



}