<?php



namespace Gek\PhpLang;

use Gek\Infrastructure\Str;
use Gek\PhpLang\PhpVisibility;
use Gek\PhpLang\UseItem;
use Gek\PhpLang\ClassMember;
use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\Traits\StaticAware;
use Gek\PhpLang\Traits\AbstractAware;
use Gek\PhpLang\Traits\FinalAware;
use Gek\PhpLang\PhpTypeDeclared;
use Gek\PhpLang\Collections\MethodParamCollection;
use Gek\PhpLang\Collections\CodeLineCollection;
use Gek\PhpLang\MethodParam;
use Gek\PhpLang\CodeLine;

/**
 * ClassMethod Sınıfı
 *
 * Sınıf metodu sınıfı
 *
 * @package Gek\PhpLang
 */
class ClassMethod extends ClassMember implements IToIndentedString, IUseCreator, \Serializable
{

    use StaticAware;
    use AbstractAware;
    use FinalAware;

    #region fields

    /**
     * arayüz metodu
     * @var bool
     */
    protected bool $interfaceMethod = false;

    /**
     * metod dönüş türü
     * @var PhpTypeDeclared|null
     */
    protected ?PhpTypeDeclared $returnType = null;

    /**
     * metod parametreleri
     * @var MethodParamCollection
     */
    protected MethodParamCollection $params;

    /**
     * metod kodları
     * @var CodeLineCollection
     */
    protected CodeLineCollection $codeLines;

    #endregion fields

    #region ctor

    /**
     * ClassMethod yapıcı metod
     * @param string $name metod adı
     * @param PhpVisibility|null $visibility metod erişebilirliği
     */
    public function __construct(string $name, ?PhpVisibility $visibility = null){
        parent::__construct($name, $visibility);
        $this->params = new MethodParamCollection();
        $this->codeLines = new CodeLineCollection();
    }

    #endregion ctor

    #region Properties

    /**
     * arayüz metodu yap / kaldıri
     * @param bool $interfaceMethod true ise arayüz metodu yapar false ise kaldırır.
     * @return self
     */
    public function setInterfaceMethod(bool $interfaceMethod = true):self {
        $this->interfaceMethod = $interfaceMethod;
        return $this;
    }

    /**
     * metod dönüş türünü verir.
     * @return string|null dönüş türü
     */
    public function getReturnType():?string {
        return $this->returnType !== null ? $this->returnType->getRaw() : null;
    }

    /**
     * metod dönüş türünü set eder.
     * @param PhpTypeDeclared|array|string|null $returnType dönüş türü
     * @return self
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function setReturnType($returnType):self {
        if (is_array($returnType)) {
            $temp = array();
            foreach ($returnType as $tp) {
                if ($tp === null) {
                    $tp = 'null';
                }
                if (empty($tp)) {
                    continue;
                }
                $temp[] = $tp;
            }
            $returnType = implode('|', $temp);
        }
        if (is_string($returnType)) {
            $returnType = new PhpTypeDeclared($returnType);
        }
        $this->returnType = $returnType;
        return $this;
    }

    /**
     * metod parametrelerini verir.
     * @return MethodParamCollection parametre kolksiyonu
     */
    public function getParams():MethodParamCollection {
        return $this->params;
    }

    /**
     * metod parametrelerini set eder.
     * @param MethodParamCollection $params parametre koleksiyonu
     * @return self
     */
    public function setParams(MethodParamCollection $params):self {
        $this->params = $params;
        return $this;
    }

    /**
     * metod kodalrını verir.
     * @return CodeLineCollection kod satırı koleksiyonu
     */
    public function getCodeLines():CodeLineCollection {
        return $this->codeLines;
    }

    /**
     * metod kodlarını set eder.
     * @param CodeLineCollection $codeLines kod satırı koleksiyonu
     * @return self
     */
    public function setCodeLines(CodeLineCollection $codeLines):self {
        $this->codeLines = $codeLines;
        return $this;
    }

    /**
     * metod kodalrını metin olarak verir.
     * @return string
     */
    public function getBody():string {
        return strval($this->codeLines);
    }

    /**
     * metod kodlarını metin olarak set eder. (ayrıştırıp kod satırı koleksiyonuna çevirip set eder.)
     * @param string $codes kod metni
     * @return self
     */
    public function setBody(string $codes):self {
        $this->codeLines->clear();
        $this->codeLines->addCodes($codes);
        return $this;
    }

    #endregion Properties

    #region Methods

    /**
     * arayüz metodu mu ?
     * @return bool arayüz metoduysa true değilse false
     */
    public function isInterfaceMethod():bool {
        return $this->interfaceMethod;
    }

    /**
     * metodun parametreleri var mı?
     * @return bool varsa true yoksa false
     */
    public function hasParams():bool {
       return $this->params->any();
    }

    /**
     * metoda parametre ekler.
     * @param MethodParam|string $param parametre veya parametre adı
     * @param string|null $type (opsiyonel) parametre türü.
     * @param string|null $val (opsiyonel) parametre varsayılan değeri
     * @param int|null $renderType (opsiyonel) parametre varsayılan değeri render tipi
     * @return self
     * @throws \ReflectionException
     * @throws \Gek\Infrastructure\Exceptions\GekException
     */
    public function addParam($param, ?string $type = null, ?string $val = null, ?int $renderType = null):self {
        $this->params->addParam($param, $type, $val, $renderType);
        return $this;
    }

    /**
     * metoda çoklu parametre ekler. (php kodu metin olarak verilir)
     * @param string $params parametreler (php kodu)
     * @return self
     * @throws \ReflectionException
     * @throws \Gek\Infrastructure\Exceptions\GekException
     */
    public function addParams(string $params):self {
        $this->params->addParams($params);
        return $this;
    }

    /**
     * metoda kod satırı ekler.
     * @param CodeLine|string $code kode satırı veya kod metni
     * @param int $indentLevel (opsiyonel) girinti düzeyi.
     * @return self
     */
    public function addCodeLine($code, int $indentLevel = 0):self {
        $this->codeLines->addCodeLine($code, $indentLevel);
        return $this;
    }

    /**
     * metoda çoklu kod satırı ekler (php kodu metin olarak verilir.)
     * @param string $code metin (php kodu)
     * @return self
     */
    public function addCodeLines(string $code):self {
        $this->codeLines->addCodes($code);
        return $this;
    }

    /**
     * metodun kodu olup olmadığına bakar.
     * @return bool varsa true yoksa false.
     */
    public function hasBody():bool {
        return $this->codeLines->any();
    }

    /**
     * metodun döüş türü olup olmadığına bakar.
     * @return bool varsa true yoksa false.
     */
    public function hasReturnType():bool {
        return $this->getReturnType() !== null;
    }

    /**
     * metod için otomatik docComment oluşturur.
     * @return self
     */
    public function autoDocComment():self {
        //$this->getDocComment()->getTags()->clear();
        if ($this->params->any()) {
            foreach ($this->params as $prm) {
                if(false == $this->getDocComment()->getTags()->containsTag('param',$prm->renderForComment()))
                {
                    $this->addCommentTag('param', $prm->renderForComment());
                }
            }
        }
        if ($this->hasReturnType()) {
            $this->getDocComment()->getTags()->addOrUpdateTag('return', $this->returnType->renderForComment());
        }

        if($this->codeLines->any()){
            $throwLines = $this->getCodeLines()
                ->where(fn(CodeLine $cl) => Str::startsWith(trim($cl->getCode()),'throw'))
                ->select(fn(CodeLine $cl) => trim(Str::fixSpaces($cl->getCode())))
                ->select(function (string $cl){
                   $res = trim(str_replace(["throw new"," ",";"],"",$cl));
                   $indx = Str::indexOf($res,"(");
                   if($indx !== -1){
                       $res = trim(substr($res,0,$indx),'(');
                   }
                   return $res;
                })->toArray();
            foreach ($throwLines as $exp){
                if(false == $this->getDocComment()->getTags()->containsTag('throws',$exp))
                {
                    $this->addCommentTag('throws', $exp);
                }
            }
        }

        if(empty($this->getDocComment()->getSummary())){
            $this->getDocComment()->setSummary($this->name . " metod.");
        }
        return $this;
    }

    /**
     * metodu metne (php koduna) çevirir.
     * @return string metin (php kodu)
     */
    public function __toString(){
        return $this->toIndentedString();
    }

    #endregion Methods

    #region IToIndentedString

    /**
     * metodu girintili metne (php koduna) çevirir.
     * @param int $indentLevel girinti düzeyi
     * @param string $indentChars girinti karakterleri
     * @return string girintili metin (php kodu)
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '):string {
        $indent = str_repeat($indentChars, $indentLevel);

        $strRes = '';
        if(!$this->getDocComment()->isEmpty()){
            $strRes .= $this->getDocComment()->toIndentedString($indentLevel,$indentChars);
        }

        $strRes .= $indent . $this->visibility->getValue() . " ";
        if ($this->isAbstract()) {
            $strRes .= 'abstract ';
        } elseif ($this->isFinal()) {
            $strRes .= 'final ';
        }
        if ($this->isStatic()) {
            $strRes .= 'static ';
        }
        $strRes .= 'function '.  $this->getName();
        $strRes .= '(' . $this->params . ")";
        if ($this->hasReturnType()) {
            $retType = $this->returnType->renderForReturn();
            if(!empty($retType)){
                $strRes .= ':' . $retType . " ";
            }
        }

        if ($this->isInterfaceMethod() || $this->isAbstract()) {
            $strRes .= ';';
        } else {
            $strRes .= '{' . PHP_EOL;

            if ($this->hasBody()) {
                /** @var CodeLineCollection $lines */
                $lines = $this->codeLines->select(function (CodeLine $item) {
                    $item->incrementIndentLevel();
                    return $item;
                })->toTypedClass(CodeLineCollection::class);
                $strRes .= $lines->toIndentedString($indentLevel,$indentChars);
            }
            $strRes .= $indent . '}' . PHP_EOL;

        }

        return $strRes;
    }

    #endregion IToIndentedString

    #region IUseCreator

    /**
     * metodun import edilecek türlerini verir.
     * @return array|UseItem[]
     */
    public function getUseArray():array {
        $useArr  = $this->params->getUseArray();
        if($this->hasReturnType()){
            $useArr = array_merge($useArr, $this->returnType->getUseArray());
        }
        return $useArr;
    }

    #endregion IUseCreator

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = $this->getDataForSerialize();
        if($this->isStatic()){
            $data['stc'] = $this->static;
        }
        if($this->isFinal()){
            $data['fnl'] = $this->final;
        }
        if($this->isAbstract()){
            $data['abs'] = $this->abstract;
        }
        if($this->isInterfaceMethod()){
            $data['imtd'] = $this->interfaceMethod;
        }
        if($this->returnType !== null){
            $data['rettyp'] = $this->returnType;
        }

        if($this->params->any()){
            $data['prm'] = $this->params;
        }
        if($this->codeLines->any()){
            $data['cln'] = $this->codeLines;
        }

        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->setDataForSerialize($data);
        if(isset($data['stc'])){
            $this->static = $data['stc'];
        }
        if(isset($data['fnl'])){
            $this->final = $data['fnl'];
        }
        if(isset($data['abs'])){
            $this->abstract = $data['abs'];
        }
        if(isset($data['imtd'])){
            $this->interfaceMethod = $data['imtd'];
        }
        $this->returnType = isset($data['rettyp']) ? $data['rettyp'] : null;
        $this->params = isset($data['prm']) ? $data['prm'] : new MethodParamCollection();
        $this->codeLines = isset($data['cln']) ? $data['cln'] : new CodeLineCollection();
    }

    #endregion Serializable
}
