<?php


namespace Gek\PhpLang;


use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Traits\ValueAware;
use Gek\Infrastructure\Str;

/**
 * ClassConst Sınıfı
 *
 * Sınıf sabiti sınıfı
 *
 * @package Gek\PhpLang
 */
class ClassConst extends ClassMember implements IToIndentedString, \Serializable
{
    use ValueAware;



    #region ctor

    /**
     * ClassConst yapıcı metod.
     * @param string $name sabit ismi
     * @param mixed $value sabit değeri
     * @param int $valueRenderType sabit değeri render tipi
     * @throws \ReflectionException
     */
    public function __construct(string $name, $value, int $valueRenderType = LiteralTypes::NONE)
    {
        parent::__construct($name, PhpVisibility::PUBLIC());
        $this->setValue($value)
            ->setValueRenderType($valueRenderType);

    }

    #endregion ctor


    #region Methods

    /**
     * otomatik docComment oluşturur
     * @return $this
     */
    public function autoDocComment(): self
    {
        return $this;
    }

    /**
     * Sınıf sabitini metne (php koduna) çevirir.
     * @return string
     */
    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion Methods

    #region IToIndentedString

    /**
     * Sınıf sabitini girintili metne (php koduna) çevirir.
     * @param int $indentLevel girinti düzeyi
     * @param string $indentChars girinti karakterleri
     * @return string girintili metin
     */
    public function toIndentedString(int $indentLevel = 0,string $indentChars = '    '): string
    {
        $strRes = '';
        $indent = str_repeat($indentChars,$indentLevel);
        if(!$this->getDocComment()->isEmpty()){
            $strRes .= $this->getDocComment()->toIndentedString($indentLevel,$indentChars);
        }

        $strRes .= $indent . $this->visibility->getValue() . " ";

        $strRes .= 'const ' . $this->getName() . " ";
        if (isset($this->value)) {
            $strRes .= "= ";
            $defVal = $this->value;
            $defVal = strval($defVal);

            switch ($this->valueRenderType) {
                case LiteralTypes::SINGLE_QUOTES:
                    $defVal = Str::format("'{0}'", $defVal);
                    break;
                case LiteralTypes::DOUBLE_QUOTES:
                    $defVal = Str::format('"{0}"', $defVal);
            }
            $strRes .= $defVal;
        }
        $strRes .= ";";
        return $strRes . PHP_EOL;
    }

    #endregion IToIndentedString


    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = $this->getDataForSerialize();
        $data['vl'] = $this->value;
        $data['rt'] = $this->valueRenderType;
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->setDataForSerialize($data);
        $this->value = $data['vl'];
        $this->valueRenderType = $data['rt'];
    }

    #endregion Serializable

}