<?php


namespace Gek\PhpLang;


use Gek\Collections\Enumerable;
use Gek\Collections\Typed\StringDictionary;
use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Traits\NameAware;
use Gek\Infrastructure\Str;

/**
 * TraitItem sınıfı
 *
 * trait kullanımıbı tutan sınıf
 *
 * @package Gek\PhpLang
 */
class TraitItem implements IToIndentedString, \Serializable
{

    #region fields

    /**
     * takma adlar sözlüğü
     * @var StringDictionary
     */
    protected StringDictionary $aliases;

    /**
     * trait türü
     *
     * @var PhpType
     */
    protected PhpType $type;

    #endregion fields

    #region ctor

    /**
     * TraitItem yapıcı metod.
     * @param string $name trait türü adı
     * @param Enumerable|StringDictionary|array|string[]|null $aliases takma adlar
     */
    public function __construct(string $name, $aliases = null)
    {
        $this->type = PhpType::parseClassType($name);
        if (!($aliases instanceof StringDictionary)) {
            $aliases = new StringDictionary($aliases);
        }
        $this->setAliases($aliases);
    }

    #endregion ctor

    #region properties

    /**
     * trait türünü verir.
     * @return PhpType
     */
    public function getType(): PhpType
    {
        return $this->type;
    }

    /**
     * trait türünü set eder.
     * @param PhpType|string $type
     * @return $this
     */
    public function setType($type): self
    {
        if (is_string($type)) {
            $type = PhpType::parseClassType($type);
        }
        $this->type = $type;
        return $this;
    }

    /**
     * trait tür adını verir.
     * @return string
     */
    public function getTypeName(): string
    {
        return $this->type->getName();
    }

    /**
     * takma adları verir.
     * @return StringDictionary
     */
    public function getAliases(): StringDictionary
    {
        return $this->aliases;
    }

    /**
     * takma adları set eder.
     * @param StringDictionary $aliases
     * @return self
     */
    public function setAliases(StringDictionary $aliases): self
    {
        $this->aliases = $aliases;
        return $this;
    }

    #endregion properties

    #region Method

    /**
     * trait kullanımını metne (php koduna) çevirir.
     * @return string
     */
    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion Method


    #region IToIndentedString

    /**
     * trait kullanımını girintili metne (php koduna) çevirir.
     * @param int $indentLevel girinti düzeyi.
     * @param string $indentChars girinti karakterleri.
     * @return string girintili metin (php kodu)
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        $indent = str_repeat($indentChars, $indentLevel);
        $typeName = $this->type->getName();
        if ($this->type->isUseItem()) {
            $typeName = $this->type->getUseItem()->getName();
            if ($this->type->getUseItem()->isAliasName()) {
                $typeName = $this->type->getUseItem()->getAliasName();
            }
        }
        $strRes = $indent . 'use ' . $typeName;
        if ($this->getAliases()->any()) {
            $strRes .= ' {' . PHP_EOL;
            $indent = str_repeat($indentChars, $indentLevel + 1);
            foreach ($this->getAliases() as $ky => $vl) {
                $strRes .= Str::format('{0}{1} as {2};', $indent, $ky, $vl) . PHP_EOL;
            }
            $indent = str_repeat($indentChars, $indentLevel);
            $strRes .= $indent . '}';
        } else {
            $strRes .= ';';
        }
        return $strRes;
    }

    #endregion IToIndentedString

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = ['t' => $this->type];
        if ($this->aliases->any()) {
            $data['a'] = $this->aliases->toArray(true);
        }
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);

        $this->type = $data['t'];
        $this->aliases = isset($data['a']) ? new StringDictionary($data['a']) : new StringDictionary();

    }

    #endregion Serializable
}
