<?php


namespace Gek\PhpLang;



use Gek\Infrastructure\Str;

/**
 * PhpType sınıfı
 *
 * Php türü sınıfı
 *
 * @package Gek\PhpLang
 */
class PhpType implements \Serializable
{
    #region field

    /**
     * tür adı
     * @var string
     */
    protected string $name;

    /**
     * türün kullanıldığı yer(ler)
     * @var TypeUsage
     */
    protected TypeUsage $usage;

    /**
     * nullable
     * @var bool
     */
    protected bool $nullable = false;

    /**
     * tür sınıf mı
     * @var bool
     */
    protected bool $isClassType = false;

    /**
     * tür importu
     * @var UseItem|null
     */
    protected ?UseItem $useItem = null;

    #endregion field

    #region ctor

    /**
     * PhpType yapıcı metod.
     * @param string $type tür adı
     * @param bool $nullable nullable mi ? (null geçilebilir mi?)
     * @param bool $isClassType tür sınıf mı ?
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function __construct(string $type, bool $nullable = false, bool $isClassType = false)
    {
        $this->name = $type;
        $this->nullable = $nullable;
        $this->isClassType = $isClassType;
        $this->setUsage(TypeUsage::FIELD_USE());
    }

    #endregion ctor

    #region properties



    /**
     * tür adını verir
     * @return string
     */
    public function getName():string {
        return $this->name;
    }

    /**
     * tür adını set eder.
     * @param string $name
     * @return self
     */
    public function setName(string $name):self {
        $this->name = $name;
        return $this;
    }

    /**
     * tür kullanım(lar)ını verir.
     * @return TypeUsage
     */
    public function getUsage():TypeUsage{
        return $this->usage;
    }

    /**
     * tür kullanım(lar)ını set eder.     * @param TypeUsage|int $usage
     * @return self
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function setUsage($usage):self {
        if(is_int($usage)){
            $usage = new TypeUsage($usage);
        }
        $this->usage = $usage;
        return $this;
    }

    /**
     * tür boş bırakılabilir mi?
     * @param bool $nullable true ise evet false ise hayır
     * @return self
     */
    public function setNullable(bool $nullable = true):self {
        $this->nullable = $nullable;
        return $this;
    }

    /**
     * türün sınıf türü olup olmadığını set eder.
     * @param bool $isClassType true ise sınıf türü false ise değil
     * @return self
     */
    public function setClassType(bool $isClassType = true):self {
        $this->isClassType = $isClassType;
        return $this;
    }

    /**
     * tür importunu verir.
     * @return UseItem|null
     */
    public function getUseItem():?UseItem{
        return $this->useItem;
    }

    /**
     * tür importunu set eder.
     * @param UseItem|null $useItem
     * @return self
     */
    public function setUseItem(?UseItem $useItem):self {
        $this->useItem = $useItem;
        return $this;
    }

    #endregion properties

    #region method

    /**
     * tür nullable mi ?
     * @return bool nullable ise true değilse false
     */
    public function isNullable():bool {
        return $this->nullable;
    }

    /**
     * tür sınıf mı ?
     * @return bool sınıf ise true değilse false
     */
    public function isClassType():bool {
        return $this->isClassType;
    }

    /**
     * türün importu var mı ?
     * @return bool varsa true yoksa false
     */
    public function isUseItem():bool {
        return $this->useItem !== null;
    }

    /**
     * tür ifadesinden türü ayrıştırır.
     * @param string $name ifade
     * @return PhpType php türü
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public static function parseClassType($name):PhpType{
        $name = trim($name);
        $phpType = new  PhpType($name,false,true);
        if(Str::contains($name,'\\')){
            $phpType->setUseItem(new UseItem($name));
        }
        return $phpType;
    }


    #endregion method

    #region utils


    #endregion utils

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = [
            'nm' => $this->name,
            'ug' => $this->getUsage()->toInt(),
            'nl' => $this->nullable,
            'ic' => $this->isClassType
        ];
        if($this->isUseItem()){
            $data['ui'] = $this->useItem;
        }
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->name = $data['nm'];
        $this->usage = new TypeUsage($data['ug']);
        $this->nullable =  $data['nl'];
        $this->isClassType = $data['ic'];
        if(isset($data['ui'])){
           $this->useItem =  $data['ui'];
        }else{
            $this->useItem = null;
        }

    }

    #endregion Serializable
}
