<?php


namespace Gek\PhpLang;


use Gek\Collections\ArrayList;
use Gek\Collections\Enumerable;
use Gek\Collections\Typed\StringDictionary;
use Gek\PhpLang\Collections\ClassMemberCollection;
use Gek\PhpLang\Collections\PhpTypeCollection;
use Gek\PhpLang\Collections\TraitItemCollection;
use Gek\PhpLang\Collections\UseItemCollection;
use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\DocComments\DocComment;
use Gek\PhpLang\Traits\AbstractAware;
use Gek\PhpLang\Traits\DocCommentAware;
use Gek\PhpLang\Traits\FinalAware;
use Gek\PhpLang\Traits\NameAware;

/**
 * PhpClass sınıfı
 * @package Gek\PhpLang
 */
class PhpClass implements IToIndentedString, \Serializable
{
    use DocCommentAware;
    use AbstractAware;
    use FinalAware;
    use NameAware;

    #region fields

    /**
     * Sınıf tipi class|interface|trait
     *
     * @var ClassTypes
     */
    protected ClassTypes $classType;

    /**
     * Ad alanı
     * @var string|null
     */
    protected ?string $namespace = null;

    /**
     * import koleksiyonu
     *
     * @var UseItemCollection
     */
    protected UseItemCollection $uses;

    /**
     * Üst sınıf
     *
     * @var PhpType|null
     */
    protected ?PhpType $extends = null;

    /**
     * Üst arayüzler
     *
     * @var PhpTypeCollection
     */
    protected PhpTypeCollection $interfaceExtends;

    /**
     * implemente edilen arayüzler
     *
     * @var PhpTypeCollection
     */
    protected PhpTypeCollection $implements;

    /**
     * Sınıf üyesi koleksiyonu
     *
     * @var ClassMemberCollection
     */
    protected ClassMemberCollection $members;

    /**
     * Kullanılan trait koleksiyonu
     *
     * @var TraitItemCollection
     */
    protected TraitItemCollection $traits;

    /**
     * otomatik docComment oluşturum modu
     * @var bool
     */
    protected bool $autoDocComment = false;

    /**
     * otomatik import senkronizasyonu
     * @var bool
     */
    protected bool $autoSyncUses = false;

    #endregion fields

    #region ctor

    /**
     * PhpClass yapıcı metod.
     *
     * @param string $name sınıf adı
     * @param ClassTypes|null $classType sınıf tipi
     */
    public function __construct(string $name, ?ClassTypes $classType = null)
    {
        $this->setName($name);
        $this->classType = ($classType !== null) ? $classType : ClassTypes::CLASS_();
        $this->uses = new UseItemCollection();
        $this->implements = new PhpTypeCollection();
        $this->interfaceExtends = new PhpTypeCollection();
        $this->members = new ClassMemberCollection();
        $this->traits = new TraitItemCollection();
    }

    #endregion ctor

    #region property

    /**
     * Sınıf tipini verir.
     *
     * @return ClassTypes
     */
    public function getClassType(): ClassTypes
    {
        return $this->classType;
    }

    /**
     * Sınıf tipini set eder
     * @param ClassTypes $classType
     * @return $this
     */
    public function setClassType(ClassTypes $classType): self
    {
        $this->classType = $classType;
        return $this;
    }

    /**
     * Ad alanını verir
     *
     * @return string|null
     */
    public function getNamespace(): ?string
    {
        return $this->namespace;
    }

    /**
     * Ad alanını set eder
     *
     * @param string|null $namespace ad alanı
     * @return $this
     */
    public function setNamespace(?string $namespace): self
    {
        $this->namespace = $namespace;
        return $this;
    }

    /**
     * Üst sınıfı verir.
     *
     * @return PhpType|null
     */
    public function getExtends(): ?PhpType
    {
        return $this->extends;
    }

    /**
     * Üst sınıfı set eder.
     *
     * @param PhpType|string|null $extends Üst sınıf
     * @return $this
     */
    public function setExtends($extends): self
    {
        if (is_string($extends)) {
            $extends = PhpType::parseClassType($extends);
        }
        $this->extends = $extends;
        return $this;
    }

    /**
     * Implemente edilen arayüz koleksiyonunu verir.
     *
     * @return PhpTypeCollection
     */
    public function getImplements(): PhpTypeCollection
    {
        return $this->implements;
    }

    /**
     * Implemente edilen arayüz koleksiyonunu set eder.
     *
     * @param PhpTypeCollection $implements  arayüz koleksiyonu
     * @return $this
     */
    public function setImplements(PhpTypeCollection $implements): self
    {
        $this->implements = $implements;
        return $this;
    }

    /**
     * Sınıf üyesi koleksiyonunu verir.
     *
     * @return ClassMemberCollection
     */
    public function getMembers(): ClassMemberCollection
    {
        return $this->members;
    }

    /**
     *  Sınıf üyeleri koleksiyonunu set eder.
     *
     * @param ClassMemberCollection $members Sınıf öğeleri koleksiyonu
     * @return $this
     */
    public function setMembers(ClassMemberCollection $members): self
    {
        $this->members = $members;
        return $this;
    }

    /**
     * Kullanılan trait koleksiyonunu verir.
     *
     * @return TraitItemCollection
     */
    public function getTraits(): TraitItemCollection
    {
        return $this->traits;
    }

    /**
     * Kullanılan trait koleksiyonunu set eder.
     * @param TraitItemCollection $traits trait koleksiyonu
     * @return $this
     */
    public function setTraits(TraitItemCollection $traits): self
    {
        $this->traits = $traits;
        return $this;
    }

    /**
     * import koleksiyonunu verir.
     *
     * @return UseItemCollection
     */
    public function getUses(): UseItemCollection
    {
        return $this->uses;
    }

    /**
     * import koleksiyonunu set eder.
     *
     * @param UseItemCollection $uses import koleksiyonu
     * @return $this
     */
    public function setUses(UseItemCollection $uses): self
    {
        $this->uses = $uses;
        return $this;
    }

    /**
     * Üst arayüzler koleksiyonunu verir.
     *
     * @return PhpTypeCollection
     */
    public function getInterfaceExtends(): PhpTypeCollection
    {
        return $this->interfaceExtends;
    }

    /**
     * Üst arayüzler koleksiyonunu set eder.
     *
     * @param PhpTypeCollection $interfaceExtends arayüz koleksiyonu
     * @return $this
     */
    public function setInterfaceExtends(PhpTypeCollection $interfaceExtends): self
    {
        $this->interfaceExtends = $interfaceExtends;
        return $this;
    }

    /**
     * otomatik docComment modu aktif mi?
     *
     * @return bool
     */
    public function isAutoDocComment(): bool
    {
        return $this->autoDocComment;
    }

    /**
     * otomatik docComment modunu aktif ya da pasif yapar.
     * .
     * @param bool $autoDocComment true ise aktif false ise pasif
     * @return $this
     */
    public function setAutoDocComment(bool $autoDocComment): self
    {
        $this->autoDocComment = $autoDocComment;
        return $this;
    }


    #endregion property

    #region Methods

    /**
     * bu sınıf arayüz mü?
     *
     * @return bool arayüz ise true değilse false
     */
    public function isInterface():bool {
        return  $this->classType->getValue() == ClassTypes::INTERFACE;
    }

    /**
     * bu sınıf trait mi?
     *
     * @return bool tarit ise true değilse false
     */
    public function isTrait():bool {
        return  $this->classType->getValue() == ClassTypes::TRAIT;
    }

    /**
     * bu sınıf sınıf mı?
     *
     * @return bool sınıf ise true değilse false
     */
    public function isClass():bool {
        return  $this->classType->getValue() == ClassTypes::CLASS_;
    }

    /**
     * import ekler
     *
     * @param UseItem|string $use import türü
     * @param string|null $aliasName (opsiyonel) takma ad
     * @return $this
     */
    public function addUse($use, ?string $aliasName = null): self
    {
        if (is_string($use)) {
            $use = new UseItem($use);
        }
        $use->setAliasName($aliasName);
        $this->uses->add($use);
        return $this;
    }

    /**
     * Sınıfın tam adını verir.
     *
     * @return string
     */
    public function getFullName(): string
    {
        if ($this->getNamespace() != null) {
            return $this->getNamespace() . "\\" . $this->getName();
        }
        return "\\" . $this->getName();
    }

    /**
     * Üst sınıf tanımlı mı?
     *
     * @return bool tanımlı ise true değilse false
     */
    public function hasExtends(): bool
    {
        $res = $this->extends !== null;
        if($res == false && $this->isInterface()){
            $res = $this->interfaceExtends->any();
        }
        return $res;
    }

    /**
     * implemente edilen arayüz var mı?
     *
     * @return bool varsa true yoksa false
     */
    public function hasImplements(): bool
    {
        return $this->implements->any();
    }

    /**
     * implemente edilen arayüz ekler.
     *
     * @param PhpType|string $implement arayüz
     * @return self
     */
    public function addImplement($implement): self
    {
        if (is_string($implement)) {
            $implement = phpType::parseClassType($implement);
        }
        $this->implements->add($implement);
        return $this;
    }

    /**
     * Sınıf üyesi ekler.
     *
     * @param ClassMember $member sınıf üyesi
     * @return $this
     */
    public function addMember(ClassMember $member): self
    {
        $this->members->add($member);
        return $this;
    }

    /**
     * Üst arayüz ekler.
     *
     * @param PhpType|string $extends üst arayüz
     * @return $this
     */
    public function addExtendsForInterface($extends): self
    {
        if (is_string($extends)) {
            $extends = phpType::parseClassType($extends);
        }
        $this->interfaceExtends->add($extends);
        return $this;
    }

    /**
     * Adı verilen sınıf öğesini verir.
     * @param string $name öğe adı
     * @return ClassMember|null sınıf öğesi (bulunamazsa null)
     */
    public function getMemberByName(string $name):?ClassMember{
        return $this->members->firstOrNull(function (ClassMember $member)use($name){
            return $member->getName() == $name;
        });
    }

    /**
     * alan (property) ekler.
     *
     * @param string $name alan adı
     * @param PhpTypeDeclared|array|string|null $type tipi
     * @return ClassField eklenen alan
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function addField(string $name, $type = null): ClassField
    {
        $field = new ClassField($name, $type);
        $this->addMember($field);
        return $field;
    }

    /**
     * verilen ada göre alanı (property) verir.
     *
     * @param string $name alan adı
     * @return ClassField|null alan (bulunamazsa null)
     */
    public function getFieldByName(string $name):?ClassField{
        return $this->members->firstOrNull(function (ClassMember $member)use($name){
            return ($member instanceof ClassField) && $member->getName() == $name;
        });
    }

    /**
     * metod ekler.
     *
     * @param string $name metod adı
     * @return ClassMethod eklenen metod
     */
    public function addMethod(string $name): ClassMethod
    {
        $method = new ClassMethod($name);
        if($this->isInterface()){
            $method->setInterfaceMethod();
        }
        $this->members->add($method);
        return $method;
    }

    /**
     * verilen ada göre metodu verir.
     * @param string $name metod adı
     * @return ClassMethod|null metod (bulunamazsa null)
     */
    public function getMethodByName(string $name):?ClassMethod{
        return $this->members->firstOrNull(function (ClassMember $member)use($name){
            return ($member instanceof ClassMethod) && $member->getName() == $name;
        });
    }

    /**
     * yapıcı metod ekler.
     *
     * @param string|true|null $region bölge adı (true ise otomatik oluşturur)
     * @return ClassMethod // yapıcı metod
     */
    public function addConstructor($region = null): ClassMethod
    {
        if ($region === true) {
            $region = 'ctor';
        }

        return $this->addMethod('__construct')
            ->setRegion($region);
    }

    /**
     *  Kullanılan trait ekler.
     *
     * @param TraitItem|string $trait trait
     * @param Enumerable|StringDictionary|array|string[]|null $aliases
     * @return $this
     */
    public function addTrait($trait, $aliases = null): self
    {
        if (is_string($trait)) {
            $trait = new TraitItem($trait, $aliases);
        }
        $this->traits->add($trait);
        return $this;
    }

    /**
     * getter ve setter metodları ile property  ekler.
     *
     * @param string $name  property adı
     * @param PhpTypeDeclared|array|string|null $type tipi
     * @param bool $get getter methodu oluştur
     * @param bool $set setter methodu oluştur
     * @return ClassField property için eklenen alan
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function addProperty(string $name, $type = null, bool $get = true, bool $set = true):ClassField
    {
        $field = $this->addField($name, $type)
            ->setProtected()
            ->setRegion('fields')
            ->autoDocComment();
        if ($get) {
            $this->addMethod('get' . ucfirst($name))
                ->setPublic()
                ->setReturnType($type)
                ->addCodeLine('return $this->' . $name .";")
                ->setRegion('properties')
                ->autoDocComment();
        }
        if($set){
            $this->addMethod('set' . ucfirst($name))
                ->setPublic()
                ->addParam($name,$type)
                ->setReturnType('self')
                ->addCodeLine('$this->' . $name . ' = $' . $name . ';')
                ->addCodeLine('return $this;')
                ->setRegion('properties')
                ->autoDocComment();
        }
        return $field;
    }

    /**
     * sınıfı metne (php koduna) çeviirir
     * @return string menşt (php kodu)
     */
    public function __toString()
    {
        return $this->toIndentedString();
    }


    #endregion Methods

    #region IToIndentedString

    /**
     * sınıfı girintili metnini (kod halini)  verir.
     *
     * @param int $indentLevel girinti seviyesi
     * @param string $indentChars girinti karakterleri
     * @return string girintili metin (php kodu)
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        $this->syncUses();
        $indent = str_repeat($indentChars, $indentLevel);
        $res = $indent;
        if ($this->getNamespace() !== null) {
            $res .= 'namespace ' . $this->getNamespace() . ';' . PHP_EOL;
            $res .= PHP_EOL;
        }

        if ($this->uses->any()) {
            $res .= $this->uses->toIndentedString($indentLevel, $indentChars);
            $res .= PHP_EOL;
        }
        if($this->isAutoDocComment()){
            if(empty($this->getDocComment()->getSummary())){
                $summary = $this->name . " ";
                if($this->isClass()){
                    $summary .= 'sınıfı.';
                }elseif ($this->isInterface()){
                    $summary .= "arayüzü.";
                }elseif ($this->isTrait()){
                    $summary .= "trait.";
                }
                $this->getDocComment()->setSummary($summary);
            }
            if(false == $this->getDocComment()->getTags()->containsTag("package")){
                if(!empty($this->namespace)){
                    $this->getDocComment()->addTag('package', $this->namespace);
                }
            }

        }

        if (!$this->getDocComment()->isEmpty()) {
            $res .= $this->getDocComment()->toIndentedString($indentLevel, $indentChars);
        }

        $res .= $indent;

        if ($this->isFinal()) {
            $res .= 'final ';
        } elseif ($this->isAbstract()) {
            $res .= 'abstract ';
        }
        $res .= $this->getClassType()->getValue() . ' ';
        $res .= $this->getName() . ' ';

        if ($this->hasExtends() && $this->isClass()) {
            $extendsType = $this->extends->getName();
            if ($this->extends->isUseItem()) {
                $extendsType = $this->extends->getUseItem()->getName();
                if ($this->extends->getUseItem()->isAliasName()) {
                    $extendsType = $this->extends->getUseItem()->getAliasName();
                }
            }
            $res .= 'extends ' . $extendsType . ' ';
        }elseif ($this->hasExtends() && $this->isInterface()) {
            $interfaceExt = new ArrayList();
            if(!empty($this->extends)){
                $extendsType = $this->extends->getName();
                if ($this->extends->isUseItem()) {
                    $extendsType = $this->extends->getUseItem()->getName();
                    if ($this->extends->getUseItem()->isAliasName()) {
                        $extendsType = $this->extends->getUseItem()->getAliasName();
                    }
                }
                $interfaceExt->add($extendsType);
            }

            $interfaceExt->addRange(
                $this->interfaceExtends
                ->select(function (PhpType $iex){
                    $nm = $iex->getName();
                    if ($iex->isUseItem()) {
                        $nm = $iex->getUseItem()->getName();
                        if ($iex->getUseItem()->isAliasName()) {
                            $nm = $iex->getUseItem()->getAliasName();
                        }
                    }
                    return $nm;
                })
            );

            $res .= 'extends ' . implode(", ", $interfaceExt->toArray()) . ' ';
        }
        if ($this->implements->any()) {
            $res .= 'implements ';
            $implements = $this->implements->select(function (PhpType $item) {
                $typeName = $item->getName();
                if ($item->isUseItem()) {
                    $typeName = $item->getUseItem()->getName();
                    if ($item->getUseItem()->isAliasName()) {
                        $typeName = $item->getUseItem()->getAliasName();
                    }
                }
                return $typeName;
            })->toArray();
            $res .= implode(', ', $implements) . ' ';
        }
        $res .= PHP_EOL . $indent . '{' . PHP_EOL . PHP_EOL;
        $indentLevel += 1;

        if ($this->traits->any()) {
            $res .= $this->traits->toIndentedString($indentLevel, $indentChars);
            $res .= PHP_EOL;
        }

        if ($this->members->any()) {
            $sections = $this->members->select(function (ClassMember $member) {
                return $member->getRegion();
            })->distinct()->toArray();
            $autoDocCom = $this->isAutoDocComment();
            foreach ($sections as $section) {
                if ($section !== null) {
                    $res .= str_repeat($indentChars, $indentLevel) . '#region ' . $section . PHP_EOL . PHP_EOL;
                }

                $res .= $this->members->where(function (ClassMember $member) use ($section) {
                    return $member->getRegion() === $section;
                })->aggregate(function (&$mbr, IToIndentedString $member) use ($indentLevel, $indentChars,$autoDocCom) {
                    if($autoDocCom && ($member instanceof ClassMember)){
                        $member->autoDocComment();
                    }
                    $mbr .= $member->toIndentedString($indentLevel, $indentChars) . PHP_EOL;
                }, '');

                if ($section !== null) {
                    $res .= str_repeat($indentChars, $indentLevel) . '#endregion ' . $section . PHP_EOL . PHP_EOL;
                }
            }
        }

        $res .= $indent . '}' . PHP_EOL;
        return $res;

    }

    #endregion IToIndentedString

    #region utils

    /**
     * import edilen türleri sınıf öğelerini tarayarak senkronize eder.
     */
    public function syncUses()
    {
        $useItemArr = new ArrayList();
        if ($this->hasExtends()) {
            if(!empty($this->getExtends())){
                if ($this->getExtends()->isUseItem()) {
                    $useItemArr->add($this->getExtends()->getUseItem());
                }
            }
            if($this->isInterface()){
                $useItemArr->addRange(
                    $this->interfaceExtends->getUseArray()
                );
            }

        }
        if ($this->hasImplements()) {
            $useItemArr->addRange($this->getImplements()->getUseArray());
        }
        if ($this->traits->any()) {
            $useItemArr->addRange($this->traits->getUseArray());
        }
        if ($this->getMembers()->any()) {
            $membersUses = $this->getMembers()
                ->where(function ($item) {
                    return ($item instanceof IUseCreator);
                })->aggregate(function (&$res, IUseCreator $itm) {
                    $res = array_merge($res, $itm->getUseArray());
                }, array());
            $useItemArr->addRange($membersUses);
        }
        $this->uses->syncUseItems($useItemArr->toArray());
    }

    #endregion utils

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = [
            'nm' => $this->name,
        ];
        if ($this->isAbstract()) {
            $data['abs'] = $this->abstract;
        }
        if ($this->isFinal()) {
            $data['fnl'] = $this->final;
        }

        if (!$this->getDocComment()->isEmpty()) {
            $data['dc'] = $this->docComment;
        }

        if ($this->classType->getValue() !== ClassTypes::CLASS_) {
            $data['clst'] = $this->classType->getValue();
        }

        if ($this->namespace !== null) {
            $data['ns'] = $this->namespace;
        }
        if ($this->uses->any()) {
            $data['uss'] = $this->uses;
        }
        if ($this->extends !== null) {
            $data['ext'] = $this->extends;
        }

        if ($this->implements->any()) {
            $data['imp'] = $this->implements;
        }

        if ($this->members->any()) {
            $data['mbrs'] = $this->members;
        }

        if ($this->traits->any()) {
            $data['trts'] = $this->traits;
        }

        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->name = $data['nm'];
        if (isset($data['abs'])) {
            $this->abstract = $data['abs'];
        }
        if (isset($data['fnl'])) {
            $this->final = $data['fnl'];
        }
        $this->docComment = isset($data['dc']) ? $data['dc'] : new DocComment();
        $this->classType = isset($data['clst']) ? new ClassTypes($data['clst']) : ClassTypes::CLASS_();
        $this->namespace = isset($data['ns']) ? $data['ns'] : null;
        $this->uses = isset($data['uss']) ? $data['uss'] : new UseItemCollection();
        $this->extends = isset($data['ext']) ? $data['ext'] : null;
        $this->implements = isset($data['imp']) ? $data['imp'] : new PhpTypeCollection();
        $this->members = isset($data['mbrs']) ? $data['mbrs'] : new ClassMemberCollection();
        $this->traits = isset($data['trts']) ? $data['trts'] : new TraitItemCollection();
    }

    #endregion Serializable

}
