<?php


namespace Gek\PhpLang\Collections;


use Gek\Collections\ArrayList;
use Gek\Collections\TypedListWrapper;
use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\PhpType;
use Gek\PhpLang\UseItem;

/**
 * PhpTypeCollection Sınıfı
 *
 * Php tip (tür) koleksiyonu
 *
 * @package Gek\PhpLang\Collections
 */
class PhpTypeCollection extends TypedListWrapper implements IUseCreator, \Serializable
{

    #region ctor

    /**
     * PhpTypeCollection yapıcı metod.
     * @param PhpType ...$items (opsiyonel) öğeler.
     */
    public function __construct(PhpType ...$items)
    {
        parent::__construct(new ArrayList(), PhpType::class);
        $this->addRange(...$items);
    }

    #endregion ctor

    #region methods

    /**
     * öğe ekler.
     * @param PhpType $item öğe
     */
    public function add(PhpType $item):void{
        $this->innerList->add($item);
    }

    /**
     * çoklu öğe ekler.
     * @param PhpType ...$items öğeler
     */
    public function addRange(PhpType ...$items):void{
        $this->innerList->addRange($items);
    }



    /**
     * öğenin koleksiyonda olup olmadığına bakar.
     * @param PhpType $item öğe
     * @return bool varsa true yoksa false
     */
    public function contains(PhpType $item):bool{
        return $this->innerList->contains($item);
    }

    /**
     * öğeyi koleksiyonda kaldırır.
     * @param PhpType  $item öğe
     * @return bool başarı durumunda true aksi halde false
     */
    public function remove(PhpType $item):bool{
        return $this->innerList->remove($item);
    }

    /**
     * verilen öğenin koleksiyondaki indeksini verir.
     * @param PhpType $item öğe
     * @return int öğenin indeksi, öğe bulunamazsa -1
     */
    public function indexOf(PhpType $item): int{
        return $this->innerList->indexOf($item);
    }

    /**
     * verilen indekse öğeyi ekler.
     * @param int $index indeks
     * @param PhpType $item öğe
     */
    public function insert(int $index, PhpType $item): void{
        $this->innerList->insert($index,$item);
    }

    /**
     * verilen indeksteki öğeyi verir.
     * @param int $index indeks
     * @return PhpType öğe
     */
    public function getAt(int $index):PhpType {
        return $this->innerList[$index];
    }

    /**
     * verilen indekse öğeyi set eder.
     * @param int $index indeks
     * @param PhpType $item öğe
     */
    public function setAt(int $index, PhpType $item):void{
        $this->innerList[$index] = $item;
    }

    /**
     * verilen indekse çoklu öğe ekler.
     * @param int $index indeks
     * @param PhpType ...$items öğeler
     */
    public function insertRange(int $index, PhpType ...$items):void{
        $this->innerList->insertRange($index,$items);
    }

    #endregion methods

    #region IUseCreator


    /**
     * koleksiyondaki türleri verir
     * @return array|UseItem[]
     */
    public function getUseArray(): array
    {
        return $this->where(function (PhpType $item){
            return $item->isUseItem();
        })->select(function (PhpType $item){
            return $item->getUseItem();
        })->toArray();
    }

    #endregion IUseCreator

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = $this->toArray();
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data =  unserialize($serialized);
        parent::__construct(new ArrayList(), PhpType::class);
        $this->addRange(...$data);
    }

    #endregion Serializable
}
