<?php


namespace Gek\PhpLang\Collections;


use Gek\Collections\ArrayList;
use Gek\Collections\TypedListWrapper;
use Gek\PhpLang\ClassMethod;
use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\MethodParam;
use Gek\PhpLang\PhpTypeDeclared;
use Gek\PhpLang\UseItem;

/**
 * MethodParamCollection Sınıfı
 *
 * Metod parametreleri koleksiyonu
 *
 * @package Gek\PhpLang\Collections
 */
class MethodParamCollection extends TypedListWrapper implements IUseCreator, \Serializable
{

    #region ctor

    /**
     * MethodParamCollection yapıcı sınıf.
     * @param MethodParam ...$items (opsiyonel) öğeler
     */
    public function __construct(MethodParam ...$items)
    {
        parent::__construct(new ArrayList(), MethodParam::class);
        $this->addRange(...$items);
    }

    #endregion ctor

    #region methods

    /**
     * öğe ekler
     * @param MethodParam $item öğe
     */
    public function add(MethodParam $item):void{
        $this->innerList->add($item);
    }

    /**
     * çoklu öğe ekler
     * @param MethodParam ...$items öğeler
     */
    public function addRange(MethodParam ...$items):void{
        $this->innerList->addRange($items);
    }



    /**
     * öğenin koleksiyonda olup olmadığına bakar.
     * @param MethodParam $item öğe
     * @return bool varsa true yoksa false
     */
    public function contains(MethodParam $item):bool{
        return $this->innerList->contains($item);
    }

    /**
     * öğeyi koleksiyondan kaldırır
     * @param MethodParam  $item öğe
     * @return bool Başarı halinde true aksi halde false
     */
    public function remove(MethodParam $item):bool{
        return $this->innerList->remove($item);
    }

    /**
     * öğenin koleksiyondaki indeksini verir.
     * @param MethodParam $item öpe
     * @return int öğenin indeksi, öğe bulunamazsa -1
     */
    public function indexOf(MethodParam $item): int{
        return $this->innerList->indexOf($item);
    }

    /**
     * verilen indekse öğe ekler
     * @param int $index indeks
     * @param MethodParam $item öğe
     */
    public function insert(int $index, MethodParam $item): void{
        $this->innerList->insert($index,$item);
    }

    /**
     * verilen indeksteki öğeyi verir.
     * @param int $index indeks
     * @return MethodParam öğe
     */
    public function getAt(int $index):MethodParam {
        return $this->innerList[$index];
    }

    /**
     * verilen indeksteki öğeyi set eder.
     * @param int $index indeks
     * @param MethodParam $item öğe
     */
    public function setAt(int $index, MethodParam $item):void{
        $this->innerList[$index] = $item;
    }

    /**
     * verilen indekse çoklu öğe ekler.
     * @param int $index indeks
     * @param MethodParam ...$items öğeler
     */
    public function insertRange(int $index, MethodParam ...$items):void{
        $this->innerList->insertRange($index,$items);
    }


    /**
     * yeni öğe oluşturup koleksiyona ekler
     * @param MethodParam|string $param parametre adı
     * @param PhpTypeDeclared|array|string|null $type (opsiyonel) parametre tipi
     * @param string|null $val (opsiyoney) parametre varsayılan değeri
     * @param int|null $renderType (opsiyonel) render tipi
     * @throws \ReflectionException
     * @throws \Gek\Infrastructure\Exceptions\GekException
     */
    public function addParam($param, $type = null, ?string $val = null, ?int $renderType = null) {
        if($param instanceof MethodParam){
            $this->add($param);
        }else{
            $param = new MethodParam($param,$type);
            $param->setValue($val,$renderType);
            $this->add($param);
        }
    }

    /**
     * verilenparametre php kodundan öğeleri ayrırştırır ve koleksiyona ekler
     * @param string $params php parametre kodu (örn. "$param, $type = null, ?string $val = null, ?int $renderType = null" )
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function addParams(string $params){
        $paramArr = MethodParam::parseParams($params);
        $this->addRange(...$paramArr);
    }

    /**
     * koleksiyonu metne (php koduna) çevirir.
     *
     * @return string
     */
    public function __toString()
    {
        return implode(', ', $this->toArray());
    }

    #endregion methods

    #region IUseCreator

    /**
     * koleksiyondaki parametre tiplerini verir.
     * @return array|UseItem[] parametre tipleri
     */
    public function getUseArray(): array
    {
        return $this->where(function (MethodParam $item){
            return $item->hasType();
        })->aggregate(function (&$arr, MethodParam $item){
            $arr = array_merge($arr, $item->getUseArray());
        },array());
    }

    #endregion IUseCreator

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        return serialize($this->toArray());
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        parent::__construct(new ArrayList(), MethodParam::class);
        $this->addRange(...$data);
    }

    #endregion Serializable
}
