<?php


namespace Gek\PhpLang\Collections;


use Gek\Collections\ArrayList;
use Gek\Collections\TypedListWrapper;
use Gek\PhpLang\DocComments\DocCommentTag;

/**
 * DocCommentTagCollection Sınıfı
 *
 * DocComment tag koleksiyonu
 *
 * @package Gek\PhpLang\Collections
 */
class DocCommentTagCollection extends TypedListWrapper implements \Serializable
{
    #region ctor

    /**
     * DocCommentTagCollection yapıcı metod.
     * @param DocCommentTag ...$items
     */
    public function __construct(DocCommentTag ...$items)
    {
        parent::__construct(new ArrayList(), DocCommentTag::class);
        $this->addRange(...$items);
    }

    #endregion ctor

    #region methods

    /**
     * Yeni öğe ekler
     *
     * @param DocCommentTag $item öğe
     */
    public function add(DocCommentTag $item):void{
        $this->innerList->add($item);
    }

    /**
     * Çoklu öğe ekler.
     * @param DocCommentTag ...$items // öüeler
     */
    public function addRange(DocCommentTag ...$items):void{
        $this->innerList->addRange($items);
    }



    /**
     * öğenin koleksiyonda olup olmadığına bakar.
     * @param DocCommentTag $item öğe
     * @return bool varsa true yoksa false
     */
    public function contains(DocCommentTag $item):bool{
        return $this->innerList->contains($item);
    }

    /**
     * öğeyi kolksiyondan kaldırır.
     *
     * @param DocCommentTag  $item jaldırılacak öğe
     * @return bool başarı halinde true aksi halde false
     */
    public function remove(DocCommentTag $item):bool{
        return $this->innerList->remove($item);
    }

    /**
     * öğenin koleksiyondaki indeksini verir.
     *
     * @param DocCommentTag $item öğe
     * @return int öğe indeksi, öğe bulunamazsa -1
     */
    public function indexOf(DocCommentTag $item): int{
        return $this->innerList->indexOf($item);
    }

    /**
     * verilen indekse öğe ekler
     * @param int $index indeks
     * @param DocCommentTag $item öğe
     */
    public function insert(int $index, DocCommentTag $item): void{
        $this->innerList->insert($index,$item);
    }

    /**
     * indekse göre öğe döndürür.
     * @param int $index indeks
     * @return DocCommentTag öğe
     */
    public function getAt(int $index):DocCommentTag {
        return $this->innerList[$index];
    }

    /**
     * indekse öğeyi set eder
     * @param int $index indeks
     * @param DocCommentTag $item öğe
     */
    public function setAt(int $index, DocCommentTag $item):void{
        $this->innerList[$index] = $item;
    }

    /**
     * verilen indekse çoklu öğe ekler.
     * @param int $index indeks
     * @param DocCommentTag ...$items öğeler
     */
    public function insertRange(int $index, DocCommentTag ...$items):void{
        $this->innerList->insertRange($index,$items);
    }

    /**
     * tagı tutan bütün öğeleri verilen değer ile günceller.
     * @param string $tag tag
     * @param string $value değer
     * @return $this
     */
    public function updateTag(string $tag, string $value):self {
        $tg = new DocCommentTag($tag,$value);
        /** @var DocCommentTag[] $lst */
        $lst = $this->where(function (DocCommentTag &$item)use($tg){
            return $item->key == $tg->key;
        })->toArrayList();
        foreach ($lst as  &$itm){
            $itm->value = $value;
        }
        return $this;
    }

    /**
     * yeni öğe ekler
     * @param string $tag öğe tagı
     * @param string $value öğe değeri
     * @return self
     */
    public function addTag(string $tag, string $value = ''):self {
        $tg = new DocCommentTag($tag,$value);
        $this->add($tg);
        return $this;
    }

    /**
     * verilen tag öğelerde varsa değerini günceller yoksa yeni öğe olarak ekler.
     * @param string $tag öğe tagı
     * @param string $value öğe değeri
     * @return self
     */
    public function addOrUpdateTag(string $tag, string $value = ''):self {
        $tg = new DocCommentTag($tag,$value);
        $isSet = $this->any(function (DocCommentTag $item)use($tg){
            return $item->key == $tg->key;
        });
        if($isSet){
            $this->updateTag($tag,$value);
        }else{
            $this->addTag($tag,$value);
        }
        return $this;
    }

    /**
     * verilen tagı kullanan öğe olup olmadığına bakar.
     * @param string $tag öğe tagı
     * @param string|null $value (opsiyonel) öğe değeri
     * @return bool varsa true yoksa false
     */
    public function containsTag(string $tag, ?string $value = null):bool {
        $tg = new DocCommentTag($tag,$value === null ? '':$value );
        if($value !== null){
            return $this->contains($tg);
        }
        return $this->any(function (DocCommentTag $item)use($tg){
            return $item->key == $tg->key;
        });
    }

    #endregion methods

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        return serialize($this->toArray());
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        parent::__construct(new ArrayList(), DocCommentTag::class);
        $this->addRange(...$data);
    }

    #endregion Serializable
}
