<?php


namespace Gek\PhpLang;


use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\Traits\StaticAware;
use Gek\PhpLang\Traits\ValueAware;
use Gek\Infrastructure\Str;
use phpDocumentor\Reflection\Types\This;

/**
 * ClassField sınıfı
 *
 * Sınıf özelliği (property) sınıfı
 *
 * @package Gek\PhpLang
 */
class ClassField extends ClassMember implements IToIndentedString, IUseCreator, \Serializable
{

    use StaticAware;
    use ValueAware;


    #region fields

    /**
     * özellik türü
     * @var PhpTypeDeclared|null
     */
    protected ?PhpTypeDeclared $type = null;

    #endregion fields

    #region ctor

    /**
     * ClassFields yapıcı metod.
     * @param string $name özellik (property) adı
     * @param PhpTypeDeclared|array|string|null $type özellik tipi
     * @param PhpVisibility|null $visibility özellik erişebilirliği
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function __construct(string $name, $type = null, ?PhpVisibility $visibility = null)
    {
        parent::__construct($name, $visibility);
        $this->setType($type);
    }

    #endregion ctor

    #region properties

    /**
     * Özelliğin (property) türünü verir.
     * @return string|null tür
     */
    public function getType(): ?string
    {
        return $this->type !== null ?
            $this->type->getRaw() :
            null;
    }

    /**
     *  Özelliğin (property) türünü set eder.
     * @param PhpTypeDeclared|array|string|null $type tür
     * @return self
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function setType($type): self
    {
        if(is_array($type)){
            $temp = array();
            foreach ($type as $tp){
                if($tp === null){
                    $tp = 'null';
                }
                if(empty($tp)){
                    continue;
                }
                $temp[] = $tp;
            }
            $type =  implode('|',$temp);
        }
        if(is_string($type)){
            $type = new PhpTypeDeclared($type);
        }
        $this->type = $type;
        return $this;
    }

    #endregion properties

    #region Methods

    /**
     * Özellik (property) için otomatik docComment oluşturur.
     * @return $this
     */
    public function autoDocComment():self {
        //$this->getDocComment()->getTags()->clear();
        if($this->type !== null && !empty($this->type->getRaw())){
            $this->getDocComment()->getTags()->addOrUpdateTag('var', $this->type->renderForComment());
        }else{
            $this->getDocComment()->getTags()->addOrUpdateTag('var', 'mixed');
        }
        if(empty($this->getDocComment()->getSummary())){
            $this->getDocComment()->setSummary($this->name);
        }
        return $this;
    }

    /**
     * Özelliği metne (php koduna) çevirir.
     *
     * @return string
     */
    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion Methods

    #region I

    /**
     * Özelliği girintili metne (php koduna ) çevirir.
     * @param int $indentLevel girinti düzeyi.
     * @param string $indentChars girinti karakterleri.
     * @return string girintili metin
     */
    public function toIndentedString(int $indentLevel = 0,string $indentChars = '    '): string
    {
        $strRes = '';
        $indent = str_repeat($indentChars,$indentLevel);
        if(!$this->getDocComment()->isEmpty()){
            $strRes .= $this->getDocComment()->toIndentedString($indentLevel,$indentChars);
        }
        $strRes .= $indent . $this->visibility->getValue() . " ";
        if($this->isStatic()){
            $strRes .= 'static ';
        }
        $type = $this->type !== null ? $this->type->renderForField() : '';
        if(!empty($type)){
            $strRes .=  $type . ' ';
        }
        $strRes .=  '$'. $this->getName();
        if(isset($this->value)){
            $strRes .= " = ";
            $defVal = $this->value;
            if($defVal === null){
                $defVal = 'null';
            }else{
                $defVal = strval($defVal);
            }
            switch ($this->valueRenderType){
                case LiteralTypes::SINGLE_QUOTES:
                    $defVal = Str::format("'{0}'", $defVal);
                    break;
                case LiteralTypes::DOUBLE_QUOTES:
                    $defVal = Str::format('"{0}"', $defVal);
            }
            $strRes .= $defVal;
        }
        $strRes .= ";";
        return $strRes . PHP_EOL;
    }

    #endregion IToIndentedString

    #region IUseCreator

    /**
     * özelliğin türlerini import için verir.
     * @return array|UseItem[]
     */
    public function getUseArray(): array
    {
        if($this->type !== null){
            return  $this->type->getUseArray();
        }
        return array();
    }

    #endregion IUseCreator

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = $this->getDataForSerialize();
        $data['vl'] = $this->value;
        $data['rt'] = $this->valueRenderType;
        if($this->isStatic()){
            $data['stc'] = $this->static;
        }
        if($this->type !== null){
            $data['typ'] = $this->type;
        }
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->setDataForSerialize($data);
        $this->value = $data['vl'];
        $this->valueRenderType = $data['rt'];
        $this->type = isset($data['typ']) ? $data['typ'] : null;
        if(isset($data['stc'])){
            $this->static = $data['stc'];
        }
    }

    #endregion Serializable




}
