<?php


namespace Gek\PhpLang;


use Gek\Collections\ArrayList;
use Gek\Infrastructure\Str;
use Gek\PhpLang\Contracts\IUseCreator;

class PhpTypeDeclared implements IUseCreator, \Serializable
{

    protected const ALL_USE = TypeUsage::FIELD_USE | TypeUsage::RETURN_USE | TypeUsage::PARAM_USE | TypeUsage::COMMENT_USE;

    protected const PHP_TYPES_USAGES = [
        // SCALAR_TYPES
        'bool' => TypeUsage::ALL,
        'boolean' => TypeUsage::COMMENT_USE,
        'int' => TypeUsage::ALL,
        'integer' => TypeUsage::COMMENT_USE,
        'float' => TypeUsage::ALL,
        'double' => TypeUsage::COMMENT_USE,
        'string' => TypeUsage::ALL,

        //COMPOUND_TYPES
        'array' => TypeUsage::ALL,
        'object' => TypeUsage::ALL,
        'callable' => TypeUsage::ALL ^ TypeUsage::FIELD_USE,
        'iterable' => TypeUsage::ALL,

        //SPECIAL_TYPES
        'resource' => TypeUsage::COMMENT_USE,
        'null' => TypeUsage::COMMENT_USE,

        //PSEUDO_TYPES
        'mixed' => TypeUsage::COMMENT_USE,
        'number' => TypeUsage::COMMENT_USE,
        'callback' => TypeUsage::COMMENT_USE,
        'void' => TypeUsage::RETURN_USE | TypeUsage::COMMENT_USE,

        //LATE STATİC BINDINGS
        'self' => TypeUsage::ALL,
        'parent' => TypeUsage::ALL,
        'static' => TypeUsage::COMMENT_USE,
    ];

    protected const TYPE_MAP = [
        'boolean' => 'bool',
        'integer' => 'int',
        'double' => 'float',
        'callback' => 'callable',
    ];

    #region fields

    protected bool $nullable = false;

    protected string $raw;

    protected ArrayList $types;

    #endregion fields

    #region ctor

    /**
     * PapTypeDeclared constructor.
     * @param string $types
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function __construct(string $types)
    {
        $this->raw = $types;
        $typesArr = array();
        $typeExplode = explode('|', $types);

        foreach ($typeExplode as $tp){
            $test = strtolower(ltrim(trim($tp),'?'));
            if(empty($test)){
                continue;
            }
            if($test === 'null'){
                $this->nullable = true;
                continue;
            }
            if(isset(self::TYPE_MAP[$test])){
                $test = self::TYPE_MAP[$test];
            }
            if(isset(self::PHP_TYPES_USAGES[$test])){
                $null = Str::startsWith(trim($tp),'?');
                if($null){
                    $this->nullable = true;
                }
                $phpType = new PhpType($test,$null,false);
                $phpType->setUsage(self::PHP_TYPES_USAGES[$test]);
                $typesArr[] = $phpType;
            }else{
                $test = trim($tp);
                $null =  Str::startsWith($test,'?');
                if($null){
                    $this->nullable = true;
                    $test = ltrim($test,'?');
                }
                $phpType = new PhpType($test,$null,true);
                $phpType->setUsage(TypeUsage::ALL());
                if(Str::contains($test,'\\')){
                    $phpType->setUseItem(new UseItem($test));
                }
                $typesArr[] = $phpType;
            }

        }
        $this->types = new ArrayList($typesArr);

    }

    #endregion ctor

    #region properties

    /**
     * @return string
     */
    public function getRaw():string {
        return $this->raw;
    }

    /**
     * @return ArrayList|PhpType[]|array
     */
    public function getTypes():ArrayList{
        return $this->types;
    }

    #endregion properties

    #region methods

    /**
     * @return bool
     */
    public function isNullable():bool {
        return $this->nullable;
    }


    /**
     * @param TypeUsage $usage
     * @return string
     */
    public function renderForUsage(TypeUsage $usage):string{
        $usage = $usage->toInt();
        if ($usage !== TypeUsage::COMMENT_USE) {
            if($this->types->count() > 1){

                return '';
            }
            /** @var PhpType $clnType */
            $clnType = $this->types->firstOrNull();
            if(($clnType->getUsage()->toInt() & $usage) !== $usage){
                return '';
            }
            $res = $clnType->getName();
            if($clnType->isClassType()){
                if($clnType->isUseItem()){
                    if($clnType->getUseItem()->isAliasName()){
                        $res = $clnType->getUseItem()->getAliasName();
                    }else{
                        $res = $clnType->getUseItem()->getName();
                    }
                }
            }
            if($this->isNullable()){
                $res = '?' . $res;
            }
            return $res;
        }
        $cleanTypes = array();
        foreach ($this->getTypes() as $phpType) {

            $clnType = $phpType->getName();
            if($phpType->isClassType() && $phpType->isUseItem()){
                if($phpType->getUseItem()->isAliasName()){
                    $clnType = $phpType->getUseItem()->getAliasName();
                }else{
                    $clnType = $phpType->getUseItem()->getName();
                }
            }
            $cleanTypes[] = $clnType;
        }

        if ($this->isNullable()) {
            $cleanTypes[] = 'null';
        }
        return implode('|', $cleanTypes);
    }

    /**
     * @return string
     */
    public function renderForField():string {
        return $this->renderForUsage(TypeUsage::FIELD_USE());
    }

    /**
     * @return string
     */
    public function renderForReturn():string {
        return $this->renderForUsage(TypeUsage::RETURN_USE());
    }

    /**
     * @return string
     */
    public function renderForParam():string {
        return $this->renderForUsage(TypeUsage::PARAM_USE());
    }

    /**
     * @return string
     */
    public function renderForComment():string {
        return $this->renderForUsage(TypeUsage::COMMENT_USE());
    }


    #endregion methods

    #region IUseCreator
    /**
     * @return array|UseItem[]
     */
    public function getUseArray(): array
    {
        return $this->types->where(function (PhpType $phpType){
            return $phpType->isClassType() && $phpType->isUseItem();
        })->select(function (PhpType $phpType){
            return $phpType->getUseItem();
        })->toArray();
    }
    #endregion IUseCreator

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = [
            'n' => $this->nullable,
            'r' => $this->raw,
            't' => $this->types->toArray(),
        ];
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->nullable = $data['n'];
        $this->raw = $data['r'];
        $this->types = new ArrayList($data['t']);
    }

    #endregion Serializable
}
